/* intrbench.cc */
/* Created by Enomoto Sanshiro on 9 August 1998. */
/* Last updated by Enomoto Sanshiro on 27 December 1999. */


#include <iostream>
#include "MushSignal.hh"
#include "MushTimer.hh"
#include "RoomVmeAccess.hh"
#include "RoomDeviceFactory.hh"
#include "module-VmeModules.hh"
#include "module-Rinei_RPV130.hh"
#include "KinokoDefs.hh"

using namespace std;


//static const int InterruptNumber = 0;  // for FORCE FGA-5x00
static const int InterruptNumber = 3;
static const int Vector = 0xfff0;

#define HANDLING_NOTIFY_CHANNEL 0
#define SCHEDULING_NOTIFY_CHANNEL 1
#define HANDLING_COUNT_CHANNEL 0
#define SCHEDULING_COUNT_CHANNEL 1


class TInterruptBenchMarker: public TMushSignalClient {
  public:
    TInterruptBenchMarker(unsigned VmeAddress, int InterruptNumber, int Vector);
    virtual ~TInterruptBenchMarker();
    virtual void Run(int NumberOfInterrupts);
    virtual void OnCatchSignal(int SignalID);
  private:
    TRoomVmeController *_VmeController;
    TRoomVmeCrate *_VmeCrate;
    TRoomVmeModule *_Scaler;
    TVmeIORegister_Rinei_RPV130 *_IORegister;
    TMushSignalHandler *_SignalHandler;
  private:
    int _InterruptCount;
    int _CaughtExitSignal;
};


TInterruptBenchMarker::TInterruptBenchMarker(unsigned VmeAddress, int InterruptNumber, int Vector)
{
    TRoomDeviceFactory* DeviceFactory = TRoomDeviceFactory::GetInstance();
    try {
	_VmeCrate = DeviceFactory->CreateVmeCrate();
	_VmeController = DeviceFactory->CreateVmeController("SBS-620");
	_Scaler = DeviceFactory->CreateVmeModule("Hoshin-V004");
	_IORegister = new TVmeIORegister_Rinei_RPV130();

	_VmeCrate->InstallController(_VmeController);
	_VmeCrate->Install(_IORegister, VmeAddress, InterruptNumber, Vector);
	_VmeCrate->Install(_Scaler, 0xff1000);

	_IORegister->Initialize();
	_Scaler->Initialize();
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TInterruptBenchMarker::TInterruptBenchMarker()",
	    "hardware exception: " + e.Message()
	);
    }

    try {
	_SignalHandler = new TMushSignalHandler();
	_SignalHandler->RegisterClient(SIGPOLL, this);
	_SignalHandler->RegisterClient(SIGTERM, this);
	_SignalHandler->RegisterClient(SIGINT, this);
    }
    catch (TSystemCallException &e) {
	throw TKinokoException(
	    "TInterruptBenchMarker::TInterruptBenchMarker()",
	    "system call exception: " + e.Message()
	);
    }

    _InterruptCount = 0;
    _CaughtExitSignal = 0;
}

TInterruptBenchMarker::~TInterruptBenchMarker()
{
    delete _SignalHandler;

    delete _Scaler;
    delete _IORegister;
    delete _VmeController;
    delete _VmeCrate;
}

void TInterruptBenchMarker::Run(int NumberOfInterrupts)
{
    _SignalHandler->StartIgnoring(SIGPOLL);
    _IORegister->EnableInterrupt();
    _IORegister->Enable(1);

    _Scaler->Disable();
    while (! _CaughtExitSignal && (_InterruptCount < NumberOfInterrupts)) {
	_Scaler->Clear();
	_Scaler->Enable();
	_IORegister->Clear();
	
	try {
	    _SignalHandler->StartHandling();
		
	    _SignalHandler->Wait();		
	    _IORegister->OutputPulse(Bit(SCHEDULING_NOTIFY_CHANNEL));

	    _SignalHandler->StartIgnoring(SIGPOLL);
	}
	catch (THardwareException &e) {
	    throw TKinokoException(
		"TInterruptBenchMarker::Run()",
		"hardware exception: " + e.Message()
	    );
	}
	catch (TSystemCallException &e) {
	    throw TKinokoException(
		"TInterruptBenchMarker::Run()",
		"system call exception: " + e.Message()
	    );
	}

	_Scaler->Disable();
	int HandlingTimeCount, SchedulingTimeCount;
	_Scaler->Read(HANDLING_COUNT_CHANNEL, HandlingTimeCount);
	_Scaler->Read(SCHEDULING_COUNT_CHANNEL, SchedulingTimeCount);
	cout << HandlingTimeCount / 10.0 << '\t';
	cout << SchedulingTimeCount / 10.0 << endl;
    }
}

void TInterruptBenchMarker::OnCatchSignal(int SignalID)
{
    _IORegister->OutputPulse(Bit(HANDLING_NOTIFY_CHANNEL));
    _InterruptCount++;

    if ((SignalID == SIGINT) || (SignalID == SIGTERM)) {
	_CaughtExitSignal = 1;
    }
}

int main(void)
{
    try {
	TInterruptBenchMarker(0x8000, InterruptNumber, Vector).Run(65536);
    }
    catch (TKinokoException &e) {
	cerr << "ERROR: " << e << endl;
    }

    return 0;
}
