/* KameGraph.cc */
/* Created by Enomoto Sanshiro on 2 April 2001. */
/* Last updated by Enomoto Sanshiro on 8 July 2009. */


#include <string>
#include <algorithm>
#include <cstring>
#include "KameGraph.hh"


using namespace std;
using namespace kame;

static const int DefaultInitialListSize = 1024;


TKameGraph::TKameGraph(void)
{
    _ListSize = DefaultInitialListSize;

    _XValueList = new double[_ListSize];
    _YValueList = new double[_ListSize];
    _XErrorList = new double[_ListSize];
    _YErrorList = new double[_ListSize];
    
    Clear();
}

TKameGraph::TKameGraph(const TKameGraph& Graph)
{
    if (this == &Graph) {
	return;
    }

    _ListSize = Graph._ListSize;

    _XValueList = new double[_ListSize];
    _YValueList = new double[_ListSize];
    _XErrorList = new double[_ListSize];
    _YErrorList = new double[_ListSize];

    _XMin = Graph._XMin;
    _XMax = Graph._XMax;
    _YMin = Graph._YMin;
    _YMax = Graph._YMax;

    _NumberOfPoints = Graph._NumberOfPoints;
    _HasErrors = Graph._HasErrors;

    _CurrentSegmentStartOffset = Graph._CurrentSegmentStartOffset;
    _SegmentSizeList = Graph._SegmentSizeList;

    memcpy(_XValueList, Graph._XValueList, _NumberOfPoints * sizeof(double));
    memcpy(_YValueList, Graph._YValueList, _NumberOfPoints * sizeof(double));
    memcpy(_XErrorList, Graph._XErrorList, _NumberOfPoints * sizeof(double));
    memcpy(_YErrorList, Graph._YErrorList, _NumberOfPoints * sizeof(double));
}

TKameGraph::TKameGraph(unsigned InitialListSize)
{
    _ListSize = InitialListSize;

    _XValueList = new double[_ListSize];
    _YValueList = new double[_ListSize];
    _XErrorList = new double[_ListSize];
    _YErrorList = new double[_ListSize];
    
    Clear();
}

TKameGraph::~TKameGraph()
{
    delete[] _XValueList;
    delete[] _YValueList;
    delete[] _XErrorList;
    delete[] _YErrorList;
}

TKameGraph& TKameGraph::operator=(const TKameGraph& Graph)
{
    if (this == &Graph) {
	return *this;
    }

    if (_ListSize != Graph._ListSize) {
	_ListSize = Graph._ListSize;
	delete[] _XValueList;
	delete[] _YValueList;
	delete[] _XErrorList;
	delete[] _YErrorList;
	_XValueList = new double[_ListSize];
	_YValueList = new double[_ListSize];
	_XErrorList = new double[_ListSize];
	_YErrorList = new double[_ListSize];
    }

    _XMin = Graph._XMin;
    _XMax = Graph._XMax;
    _YMin = Graph._YMin;
    _YMax = Graph._YMax;

    _NumberOfPoints = Graph._NumberOfPoints;
    _HasErrors = Graph._HasErrors;

    _CurrentSegmentStartOffset = Graph._CurrentSegmentStartOffset;
    _SegmentSizeList = Graph._SegmentSizeList;

    memcpy(_XValueList, Graph._XValueList, _NumberOfPoints * sizeof(double));
    memcpy(_YValueList, Graph._YValueList, _NumberOfPoints * sizeof(double));
    memcpy(_XErrorList, Graph._XErrorList, _NumberOfPoints * sizeof(double));
    memcpy(_YErrorList, Graph._YErrorList, _NumberOfPoints * sizeof(double));

    return *this;
}

void TKameGraph::Clear(void)
{
    _NumberOfPoints = 0;
    _HasErrors = false;

    _XMin = 0;
    _XMax = 0;
    _YMin = 0;
    _YMax = 0;

    _CurrentSegmentStartOffset = 0;
    if (! _SegmentSizeList.empty()) {
	_SegmentSizeList.erase(
	    _SegmentSizeList.begin(), _SegmentSizeList.end()
	);
    }
}

bool TKameGraph::HasData(void) const
{
    return (_NumberOfPoints > 0);
}

void TKameGraph::ResizeList(double*& List, unsigned NewSize, unsigned NumberOfPoints)
{
    double* OldList = List;
    List = new double[NewSize];

    if (NumberOfPoints > 0) {
	memcpy(List, OldList, NumberOfPoints * sizeof(double));
    }

    delete[] OldList;
}

void TKameGraph::BreakSegment(void)
{
    if (_NumberOfPoints == 0) {
	return;
    }

    int CurrentSegmentSize = _NumberOfPoints - _CurrentSegmentStartOffset;
    _SegmentSizeList.push_back(CurrentSegmentSize);

    _CurrentSegmentStartOffset = _NumberOfPoints;
}

unsigned TKameGraph::NumberOfSegments(void) const
{
    return _SegmentSizeList.size();
}

unsigned TKameGraph::SegmentSizeOf(unsigned SegmentIndex) const
{
    if (_SegmentSizeList.empty()) {
	return _NumberOfPoints;
    }
    else if (SegmentIndex < _SegmentSizeList.size()) {
	return _SegmentSizeList[SegmentIndex];
    }
    else {
	return 0;
    }
}

void TKameGraph::SortByXValue(void)
{
    vector<pair<double, int> > XValueList;
    XValueList.resize(_NumberOfPoints);
    for (unsigned i = 0; i < _NumberOfPoints; i++) {
	XValueList[i].first = _XValueList[i];
	XValueList[i].second = i;
    }
    
    sort(XValueList.begin(), XValueList.end());
    
    double* YValueList = new double[_NumberOfPoints];
    double* XErrorList = new double[_NumberOfPoints];
    double* YErrorList = new double[_NumberOfPoints];

    memcpy(YValueList, _YValueList, _NumberOfPoints * sizeof(double));
    memcpy(XErrorList, _XErrorList, _NumberOfPoints * sizeof(double));
    memcpy(YErrorList, _YErrorList, _NumberOfPoints * sizeof(double));

    for (unsigned i = 0; i < _NumberOfPoints; i++) {
	int PrevIndex = XValueList[i].second;
	_XValueList[i] = XValueList[i].first;
	_XErrorList[i] = XErrorList[PrevIndex];
	_YValueList[i] = YValueList[PrevIndex];
	_YErrorList[i] = YErrorList[PrevIndex];
    }

    delete[] YValueList;
    delete[] XErrorList;
    delete[] YErrorList;
}
