/* KameGraph.hh */
/* Created by Enomoto Sanshiro on 2 April 2001. */
/* Last updated by Enomoto Sanshiro on 8 July 2009. */


#ifndef __KameGraph_hh__
#define __KameGraph_hh__


#include <string>
#include <vector>
#include "KameObject.hh"


namespace kame {

class TKameGraph: public TKameObject {
  public:
    TKameGraph(void);
    TKameGraph(const TKameGraph& Graph);
    explicit TKameGraph(unsigned InitialListSize);
    virtual ~TKameGraph();
    virtual TKameGraph& operator=(const TKameGraph& Graph);
  public:
    void Clear(void);
    bool HasData(void) const;
  public:
    inline void Fill(double X, double Y);
    inline void Fill(double X, double Y, double XError, double YError);
    inline unsigned NumberOfPoints(void) const;
    inline const double* XValueList(void) const;
    inline const double* YValueList(void) const;
    inline const double* XErrorList(void) const;
    inline const double* YErrorList(void) const;
    inline double XMin(void) const;
    inline double XMax(void) const;
    inline double YMin(void) const;
    inline double YMax(void) const;
    inline double XWidth(void) const;
    inline double YWidth(void) const;
    inline bool HasErrors(void) const;
  public:
    void BreakSegment(void);
    unsigned NumberOfSegments(void) const;
    unsigned SegmentSizeOf(unsigned SegmentIndex) const;
    void SortByXValue(void);
  protected:
    void ResizeList(double*& List, unsigned NewSize, unsigned NumberOfPoints);
  protected:
    double* _XValueList;
    double* _YValueList;
    double* _XErrorList;
    double* _YErrorList;
    unsigned _NumberOfPoints;
    unsigned _ListSize;
    double _XMin, _XMax, _YMin, _YMax;
    bool _HasErrors;
  protected:
    int _CurrentSegmentStartOffset;
    std::vector<int> _SegmentSizeList;
};



inline void TKameGraph::Fill(double X, double Y)
{
    if (_NumberOfPoints >= _ListSize) {
	_ListSize *= 2;
	ResizeList(_XValueList, _ListSize, _NumberOfPoints);
	ResizeList(_YValueList, _ListSize, _NumberOfPoints);
	ResizeList(_XErrorList, _ListSize, _NumberOfPoints);
	ResizeList(_YErrorList, _ListSize, _NumberOfPoints);
    }

    _XValueList[_NumberOfPoints] = X;
    _YValueList[_NumberOfPoints] = Y;

    if (_NumberOfPoints == 0) {
	_XMin = _XMax = X;
	_YMin = _YMax = Y;
    }
    else {
	_XMin = (X < _XMin) ? X : _XMin;
	_XMax = (X > _XMax) ? X : _XMax;
	_YMin = (Y < _YMin) ? Y : _YMin;
	_YMax = (Y > _YMax) ? Y : _YMax;
    }

    _NumberOfPoints++;
}

inline void TKameGraph::Fill(double X, double Y, double XError, double YError)
{
    Fill(X, Y);

    _XErrorList[_NumberOfPoints - 1] = XError;
    _YErrorList[_NumberOfPoints - 1] = YError;

    _HasErrors = true;
}

inline unsigned TKameGraph::NumberOfPoints(void) const
{
    return _NumberOfPoints;
}

inline const double* TKameGraph::XValueList(void) const
{
    return _XValueList;
}

inline const double* TKameGraph::YValueList(void) const
{
    return _YValueList;
}

inline const double* TKameGraph::XErrorList(void) const
{
    return _HasErrors ? _XErrorList : 0;
}

inline const double* TKameGraph::YErrorList(void) const
{
    return _HasErrors ? _YErrorList : 0;
}

inline double TKameGraph::XMin(void) const
{
    return _XMin;
}

inline double TKameGraph::XMax(void) const
{
    return _XMax;
}

inline double TKameGraph::YMin(void) const
{
    return _YMin;
}

inline double TKameGraph::YMax(void) const
{
    return _YMax;
}

inline double TKameGraph::XWidth(void) const
{
    return _XMax - _XMin;
}

inline double TKameGraph::YWidth(void) const
{
    return _YMax - _YMin;
}

inline bool TKameGraph::HasErrors(void) const
{
    return _HasErrors;
}


}
#endif
