/* KameHistogram.cc */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 8 July 2009. */


#include <string>
#include <cmath>
#include <cstring>
#include "KameHistogram.hh"

using namespace std;
using namespace kame;

#define sqr(x) ((x) * (x))


TKameHistogram::TKameHistogram(void)
: _Scale(1, 0, 1)
{
    _CountStorage = new double[_Scale.NumberOfBins()];
    _ErrorStorage = 0;

    Clear();
}

TKameHistogram::TKameHistogram(const TKameHistogramScale& Scale)
: _Scale(Scale)
{
    _CountStorage = new double[_Scale.NumberOfBins()];
    _ErrorStorage = 0;

    Clear();
}

TKameHistogram::TKameHistogram(const TKameHistogram& Histogram)
: _Scale(Histogram._Scale)
{
    if (this == &Histogram) {
	return;
    }

    int NumberOfBins = Histogram._Scale.NumberOfBins();
    int Size = sizeof(double) * NumberOfBins;

    _CountStorage = new double[NumberOfBins];
    memcpy(_CountStorage, Histogram._CountStorage, Size);

    if (Histogram._ErrorStorage) {
	_ErrorStorage = new double[NumberOfBins];
	memcpy(_ErrorStorage, Histogram._ErrorStorage, Size);
    }
    else {
	_ErrorStorage = 0;
    }

    _UnderflowCounts = Histogram._UnderflowCounts;
    _OverflowCounts = Histogram._OverflowCounts;
    _NumberOfEntries = Histogram._NumberOfEntries;
    _Sum = Histogram._Sum;
    _SumOfSquared = Histogram._SumOfSquared;
    _PeakCounts = Histogram._PeakCounts;
    _PeakBinIndex = Histogram._PeakBinIndex;
    _AccumulationTime = Histogram._AccumulationTime;
}

TKameHistogram::~TKameHistogram()
{
    delete[] _CountStorage;
    delete[] _ErrorStorage;
}

TKameHistogram& TKameHistogram::operator=(const TKameHistogram& Histogram)
{
    if (this == &Histogram) {
	return *this;
    }

    _Scale = Histogram._Scale;

    int NumberOfBins = Histogram._Scale.NumberOfBins();
    if (NumberOfBins != this->_Scale.NumberOfBins()) {
	delete[] _CountStorage;
	delete[] _ErrorStorage;
	_CountStorage = new double[NumberOfBins];
	if (Histogram._ErrorStorage) {
	    _ErrorStorage = new double[NumberOfBins];
	}
	else {
	    _ErrorStorage = 0;
	}
    }

    int Size = sizeof(double) * NumberOfBins;
    memcpy(_CountStorage, Histogram._CountStorage, Size);
    if (_ErrorStorage) {
	memcpy(_ErrorStorage, Histogram._ErrorStorage, Size);
    }

    _UnderflowCounts = Histogram._UnderflowCounts;
    _OverflowCounts = Histogram._OverflowCounts;
    _NumberOfEntries = Histogram._NumberOfEntries;
    _Sum = Histogram._Sum;
    _SumOfSquared = Histogram._SumOfSquared;
    _PeakCounts = Histogram._PeakCounts;
    _PeakBinIndex = Histogram._PeakBinIndex;
    _AccumulationTime = Histogram._AccumulationTime;

    return *this;
}

void TKameHistogram::SetScale(const TKameHistogramScale& Scale)
{
    delete[] _CountStorage;
    delete[] _ErrorStorage;

    _Scale = Scale;

    _CountStorage = new double[_Scale.NumberOfBins()];
    _ErrorStorage = 0;

    Clear();
}

void TKameHistogram::Clear(void)
{
    _OverflowCounts = 0;
    _UnderflowCounts = 0;
    _NumberOfEntries = 0;

    _Sum = 0;
    _SumOfSquared = 0;

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    _AccumulationTime = -1;

    for (int i = 0; i < _Scale.NumberOfBins(); i++) {
	_CountStorage[i] = 0;
    }
}

bool TKameHistogram::HasData(void)
{
    return (_NumberOfEntries > 0);
}

void TKameHistogram::SetAccumulationTime(long AccumulationTime)
{
    _AccumulationTime = AccumulationTime;
}

void TKameHistogram::Scale(double Factor)
{
    int NumberOfBins = _Scale.NumberOfBins();

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	_CountStorage[BinIndex] *= Factor;
	_ErrorStorage[BinIndex] *= fabs(Factor);
    }
    _PeakCounts *= Factor;

    _UnderflowCounts *= Factor;
    _OverflowCounts *= Factor;

    _NumberOfEntries *= Factor;
    _Sum *= Factor;
    _SumOfSquared *= Factor;
}

void TKameHistogram::AddHistogram(TKameHistogram& Histogram, double Weight) throw(TKameException)
{
    int NumberOfBins = _Scale.NumberOfBins();
    if (Histogram.Scale().NumberOfBins() != NumberOfBins) {
	throw TKameException(
	    "TKameHistogram::AddHistogram()", "inconsistent histogram size"
	);
    }

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double Counts = (
	    CountsOf(BinIndex) + Weight * Histogram.CountsOf(BinIndex)
	);
	double Error = sqrt(
	    sqr(ErrorOf(BinIndex)) + sqr(Weight * Histogram.ErrorOf(BinIndex))
	);
	    
	_CountStorage[BinIndex] = Counts;
	_ErrorStorage[BinIndex] = Error;

	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
    }

    _UnderflowCounts += Weight * Histogram._UnderflowCounts;
    _OverflowCounts += Weight * Histogram._OverflowCounts;

    _NumberOfEntries += Weight * Histogram._NumberOfEntries;
    _Sum += Weight * Histogram._Sum;
    _SumOfSquared += Weight * Histogram._SumOfSquared;
}

void TKameHistogram::MultiplyHistogram(TKameHistogram& Histogram, double Power) throw(TKameException)
{
    int NumberOfBins = _Scale.NumberOfBins();
    if (Histogram.Scale().NumberOfBins() != NumberOfBins) {
	throw TKameException(
	    "TKameHistogram::MultiplyHistogram()", "inconsistent histogram size"
	);
    }

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    _NumberOfEntries = 0;
    _Sum = 0;
    _SumOfSquared = 0;
    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double Counts = (
	    CountsOf(BinIndex) * pow(Histogram.CountsOf(BinIndex), Power)
	);
	double Error = 0; //...
	    
	_CountStorage[BinIndex] = Counts;
	_ErrorStorage[BinIndex] = Error;

	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
	_NumberOfEntries += Counts;
	_Sum += Counts * _Scale.BinCenterOf(BinIndex);
	_SumOfSquared += sqr(Counts * _Scale.BinCenterOf(BinIndex));
    }

    _UnderflowCounts *= pow(Histogram._UnderflowCounts, Power);
    _OverflowCounts *= pow(Histogram._OverflowCounts, Power);
}



TKameHistogramScale::TKameHistogramScale(int NumberOfBins, double Min, double Max)
{
    _NumberOfBins = NumberOfBins;
    _Min = Min;
    _Max = Max;

    _BinWidth = (_Max - _Min) / _NumberOfBins;
}

TKameHistogramScale::TKameHistogramScale(const TKameHistogramScale& Scale)
{
    if (this == &Scale) {
	return;
    }

    _NumberOfBins = Scale._NumberOfBins;
    _Min = Scale._Min;
    _Max = Scale._Max;
    _BinWidth = Scale._BinWidth;
    _Title = Scale._Title;
}

TKameHistogramScale::~TKameHistogramScale()
{
}

TKameHistogramScale& TKameHistogramScale::operator=(const TKameHistogramScale& Scale)
{
    if (this == &Scale) {
	return *this;
    }

    _NumberOfBins = Scale._NumberOfBins;
    _Min = Scale._Min;
    _Max = Scale._Max;
    _BinWidth = Scale._BinWidth;
    _Title = Scale._Title;

    return *this;
}

string TKameHistogramScale::Title(void) const
{
    return _Title;
}

void TKameHistogramScale::SetTitle(const string& Title)
{
    _Title = Title;
}
