/* KameHistogram.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 8 July 2009. */


#ifndef __KameHistogram_hh__
#define __KameHistogram_hh__


#include <string>
#include <cmath>
#include "KameObject.hh"


namespace kame {


class TKameHistogramScale {
  public:
    TKameHistogramScale(int NumberOfBins, double Min, double Max);
    TKameHistogramScale(const TKameHistogramScale& Scale);
    virtual ~TKameHistogramScale();
    TKameHistogramScale& operator=(const TKameHistogramScale& Scale);
  public:
    void SetTitle(const std::string& Title);
    std::string Title(void) const;
    inline int BinNumberOf(double Data) const;
    inline double BinValueOf(int BinNumber) const;
    inline double BinWidthOf(int BinNumber) const;
    inline double BinCenterOf(int BinNumber) const;
    inline double Min(void) const;
    inline double Max(void) const;
    inline int NumberOfBins(void) const;
    inline int operator()(double DataValue) const;
    inline double operator[](int BinNumber) const;
  protected:
    std::string _Title;
    long _NumberOfBins;
    double _Min, _Max;
    double _BinWidth;
};


class TKameHistogram: public TKameObject {
  public:
    TKameHistogram(void);
    TKameHistogram(const TKameHistogram& Histogram);
    explicit TKameHistogram(const TKameHistogramScale& Scale);
    virtual ~TKameHistogram();
    virtual TKameHistogram& operator=(const TKameHistogram& Histogram);
  public:
    void SetScale(const TKameHistogramScale& Scale);
    void Clear(void);
    bool HasData(void);
    void SetAccumulationTime(long AccumulationTime);
  public:
    inline void Fill(double Data, double Weight = 1.0);
    inline void Fill(double Data, double Weight, double Error);
    inline TKameHistogramScale& Scale(void);
    inline const TKameHistogramScale& Scale(void) const;
    inline double CountsOf(int BinNumber) const;
    inline double ErrorOf(int BinNumber) const;
    inline double UnderflowCounts(void) const;
    inline double OverflowCounts(void) const;
    inline double NumberOfEntries(void) const;
    inline double Mean(void) const;
    inline double Deviation(void) const;
    inline double PeakCounts(void) const;
    inline double PeakValue(void) const;
    inline long AccumulationTime(void) const;
    inline double operator()(double DataValue) const;
    inline double operator[](int BinNumber) const;
    void Scale(double Factor);
    void AddHistogram(TKameHistogram& Histogram, double Weight = +1) throw(TKameException);
    void MultiplyHistogram(TKameHistogram& Histogram, double Power = +1) throw(TKameException);
    inline bool HasErrors(void) const;
  protected:
    TKameHistogramScale _Scale;
    double* _CountStorage;
    double* _ErrorStorage;
    double _UnderflowCounts;
    double _OverflowCounts;
    double _NumberOfEntries;
    double _Sum, _SumOfSquared;
    double _PeakCounts;
    int _PeakBinIndex;
    long _AccumulationTime;
};


inline void TKameHistogram::Fill(double Data, double Weight)
{
    if (Data < _Scale.Min()) {
	_UnderflowCounts += Weight;
    }
    else if (Data >= _Scale.Max()) {
	_OverflowCounts += Weight;
    }
    else {
	int BinIndex = _Scale.BinNumberOf(Data);
	double Counts = (_CountStorage[BinIndex] += Weight);
	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
    }

    _NumberOfEntries += Weight;

    _Sum += Weight * Data;
    _SumOfSquared += Weight * Data * Data;
}

inline void TKameHistogram::Fill(double Data, double Weight, double Error)
{
    Fill(Data, Weight);
    
    int NumberOfBins = _Scale.NumberOfBins();
    int BinIndex = _Scale.BinNumberOf(Data);

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = std::sqrt(_CountStorage[BinIndex]);
	}
    }

    if ((BinIndex >= 0) && (BinIndex < NumberOfBins)) {
	double PreviousError = ErrorOf(BinIndex);
	_ErrorStorage[BinIndex] = std::sqrt(
	    (PreviousError * PreviousError) + (Error * Error)
	);
    }
}

inline TKameHistogramScale& TKameHistogram::Scale(void)
{
    return _Scale;
}

inline const TKameHistogramScale& TKameHistogram::Scale(void) const
{
    return _Scale;
}

inline double TKameHistogram::CountsOf(int BinNumber) const
{
    return _CountStorage[BinNumber];
}

inline double TKameHistogram::ErrorOf(int BinNumber) const
{
    if (_ErrorStorage) {
	return _ErrorStorage[BinNumber];
    }
    else if (_CountStorage[BinNumber] > 0) {
	return std::sqrt(_CountStorage[BinNumber]);
    }
    
    return 0;
}

inline double TKameHistogram::OverflowCounts(void) const
{
    return _OverflowCounts;
}

inline double TKameHistogram::UnderflowCounts(void) const
{
    return _UnderflowCounts;
}

inline double TKameHistogram::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

inline double TKameHistogram::Mean(void) const
{
    return _Sum / _NumberOfEntries;
}

inline double TKameHistogram::Deviation(void) const
{
    return std::sqrt(_SumOfSquared / _NumberOfEntries - Mean() * Mean());
}

inline double TKameHistogram::PeakCounts(void) const
{
    return _PeakCounts;
}

inline double TKameHistogram::PeakValue(void) const
{
    return _Scale.BinCenterOf(_PeakBinIndex);
}

inline long TKameHistogram::AccumulationTime(void) const
{
    return _AccumulationTime;
}

inline bool TKameHistogram::HasErrors(void) const
{
    return _ErrorStorage != 0;
}

inline double TKameHistogram::operator()(double DataValue) const
{
    return _CountStorage[_Scale.BinNumberOf(DataValue)];
}

inline double TKameHistogram::operator[](int BinNumber) const
{
    return _CountStorage[BinNumber];
}



inline int TKameHistogramScale::BinNumberOf(double Data) const
{
    if (Data < _Min) {
	// Underflow
	return _NumberOfBins + 0;
    }
    else if (Data >= _Max) {
	// Overflow
	return _NumberOfBins + 1;
    }
    else {
	return (int) ((Data - _Min) / _BinWidth);
    }
}

inline double TKameHistogramScale::BinValueOf(int BinNumber) const
{
    if ((BinNumber < 0) || (BinNumber > _NumberOfBins)) {
	// out of range
	return 2.0 * _Max;
    }

    return _Min + BinNumber * _BinWidth;
}

inline double TKameHistogramScale::BinWidthOf(int BinNumber) const
{
    return _BinWidth;
}

inline double TKameHistogramScale::BinCenterOf(int BinNumber) const
{
    return BinValueOf(BinNumber) + _BinWidth / 2;
}

inline double TKameHistogramScale::Min(void) const
{
    return _Min;
}

inline double TKameHistogramScale::Max(void) const
{
    return _Max;
}

inline int TKameHistogramScale::NumberOfBins(void) const
{
    return _NumberOfBins;
}

inline int TKameHistogramScale::operator()(double DataValue) const
{
    return BinNumberOf(DataValue);
}

inline double TKameHistogramScale::operator[](int BinNumber) const
{
    return BinCenterOf(BinNumber);
}

}
#endif
