/* KameRootRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 9 July 2009. */


#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstring>
#include <iomanip>
#include "KameDefs.hh"
#include "KameNtuple.hh"
#include "KameGraph.hh"
#include "KameHistogram.hh"
#include "KameHistogram2d.hh"
#include "KameRepository.hh"
#include "KameRootRepository.hh"

#include <TFile.h>
#include <TNtuple.h>
#include <TTree.h>
#include <TLeaf.h>
#include <TH1.h>
#include <TH2.h>
#include <TGraph.h>
#include <TGraphErrors.h>
#include <TObjString.h>
#include <TMap.h>
#include <TObjArray.h>

using namespace std;
using namespace kame;


static const int MaxStringLength = 256;


TKameRootRepository::TKameRootRepository(const string& FileName)
{
    _FileName = FileName;

    _OutputFile = 0;
    _InputFile = 0;
}

TKameRootRepository::~TKameRootRepository()
{
    Close();

    delete _InputFile;
    delete _OutputFile;
}

void TKameRootRepository::OpenToWrite(void) throw(TKameException)
{
    if (_OutputFile) {
	throw TKameException(
	    "TKamePlainTextRepository::OpenToWrite()", 
	    "file already opened"
	);
    }
    if (_InputFile) {
	throw TKameException(
	    "TKamePlainTextRepository::OpenToWrite()", 
	    "file already opened in read-only mode"
	);
    }

    _OutputFile = new TFile(_FileName.c_str(), "RECREATE");
}

void TKameRootRepository::OpenToRead(void) throw(TKameException)
{
    if (_InputFile) {
	throw TKameException(
	    "TKamePlainTextRepository::OpenToRead()", 
	    "file already opened"
	);
    }
    if (_OutputFile) {
	throw TKameException(
	    "TKamePlainTextRepository::OpenToRead()", 
	    "file already opened in write-only mode"
	);
    }

    _InputFile = new TFile(_FileName.c_str());
}

void TKameRootRepository::Close(void)
{
    if (_OutputFile) {
	delete _OutputFile;
	_OutputFile = 0;
    }

    if (_InputFile) {
	delete _InputFile;
	_InputFile = 0;
    }
}

void TKameRootRepository::SavePropertyList(const TKameObject& Object, const std::string& Name) throw(TKameException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const vector<string>& PropertyNameList = Object.PropertyNameList();
    unsigned NumberOfProperties = PropertyNameList.size();

    TObjString** ValueList = new TObjString*[NumberOfProperties];
    TTree* Tree = new TTree(Name.c_str(), Name.c_str());
    for (unsigned Index = 0; Index < NumberOfProperties; Index++) {
	string Name = PropertyNameList[Index].c_str();
	string Value = Object.Property(Name);
	ValueList[Index] = new TObjString(Value.c_str());
	Tree->Branch(Name.c_str(), "TObjString", &(ValueList[Index]));
    }

    Tree->Fill();
    _OutputFile->Write(Name.c_str());

    delete Tree;
    for (unsigned i = 0; i < NumberOfProperties; i++) {
	delete ValueList[i];
    }
    delete[] ValueList;
}

void TKameRootRepository::LoadPropertyList(TKameObject& Object, const std::string& Name, int Revision) throw(TKameException)
{
    if (_InputFile == 0) {
	OpenToRead();
    }

    TObject* RootObject = _InputFile->Get(Name.c_str());
    if (RootObject == 0) {
	return;
    }
    if (strcmp(RootObject->ClassName(), "TTree") != 0) {
	return;
    }
    TTree* Tree = (TTree*) RootObject;

    TObjArray* BranchList = Tree->GetListOfBranches();
    if (BranchList == 0) {
	return;
    }
    int NumberOfProperties = BranchList->GetEntries();

    vector<string> NameList;
    TObjString** ValueList = new TObjString*[NumberOfProperties];
    for (int i = 0; i < NumberOfProperties; i++) {
	TBranch* Branch = (TBranch*) BranchList->At(i);
	const char* Name = Branch->GetName();
	NameList.push_back(Name);
	ValueList[i] = 0;
	Tree->SetBranchAddress(Name, &(ValueList[i]));
    }	

    Tree->GetEntry(0);

    for (int i = 0; i < NumberOfProperties; i++) {
	Object.Property(NameList[i]) = ValueList[i]->String().Data();
    }

    delete[] ValueList;
    delete Tree;
}

void TKameRootRepository::SaveNtuple(const TKameNtuple& Ntuple, const std::string& Name) throw(TKameException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    unsigned NumberOfRows = Ntuple.NumberOfRows();
    if (NumberOfRows == 0) {
	return;
    }

    unsigned NumberOfColumns = Ntuple.NumberOfColumns();
    vector<TKameVariant> TypeList;
    for (unsigned i = 0; i < NumberOfColumns; i++) {
	if (Ntuple.HasTypeList()) {
	    if (
		(Ntuple.ColumnTypeOf(i) == "int") || 
		(Ntuple.ColumnTypeOf(i) == "long")
	    ){
		TypeList.push_back(TKameVariant(long(0)));
	    }
	    else if (
		(Ntuple.ColumnTypeOf(i) == "float") || 
		(Ntuple.ColumnTypeOf(i) == "double")
	    ){
		TypeList.push_back(TKameVariant(double(0)));
	    }
	    else {
		TypeList.push_back(TKameVariant(string("")));
	    }
	}
	else {
	    TypeList.push_back(Ntuple[0][i]);
	}
    }

    vector<string> ColumnNameList;
    for (unsigned Index = 0; Index < NumberOfColumns; Index++) {
	string ColumnName = Ntuple.ColumnNameOf(Index);
	if (ColumnName.empty()) {
	    ostringstream os;
	    os << "Column" << setfill('0') << setw(2) << Index;
	    ColumnName = os.str();
	}
	ColumnNameList.push_back(ColumnName);
    }

    Long64_t* LongBuffer = new Long64_t[NumberOfColumns];
    Double_t* DoubleBuffer = new Double_t[NumberOfColumns];
    Char_t** StringBuffer = new Char_t*[NumberOfColumns];
    for (unsigned i = 0; i < NumberOfColumns; i++) {
	StringBuffer[i] = new Char_t[MaxStringLength];
    }
    
    TTree* Tree = new TTree(Name.c_str(), Name.c_str());
    for (unsigned Index = 0; Index < NumberOfColumns; Index++) {
	string Name = ColumnNameList[Index];
	string LeafList = Name;
	if (TypeList[Index].IsLong()) {
	    LeafList += "/L";
	    Tree->Branch(
		Name.c_str(), (void*) &(LongBuffer[Index]), LeafList.c_str()
	    );
	}
	else if (TypeList[Index].IsDouble()) {
	    LeafList += "/D";
	    Tree->Branch(
		Name.c_str(), (void*) &(DoubleBuffer[Index]), LeafList.c_str()
	    );
	}
	else {
	    LeafList += "[256]/C"; //... TODO: use actual length ...//
	    Tree->Branch(
		Name.c_str(), (void*) StringBuffer[Index], LeafList.c_str()
	    );
	}
    }

    for (unsigned i = 0; i < NumberOfRows; i++) {
	for (unsigned j = 0; j < NumberOfColumns; j++) {
	    if (TypeList[j].IsLong()) {
		LongBuffer[j] = (long) Ntuple[i][j];
	    }
	    else if (TypeList[j].IsDouble()) {
		DoubleBuffer[j] = Ntuple[i][j];
	    }
	    else {
		strncpy(
		    StringBuffer[j], string(Ntuple[i][j]).c_str(), 
		    MaxStringLength-1
		);
		StringBuffer[j][MaxStringLength-1] = '\0';
	    }
	}
	    
	Tree->Fill();
    }

    _OutputFile->Write(Name.c_str());

    delete Tree;

    for (unsigned i = 0; i < NumberOfColumns; i++) {
	delete[] StringBuffer[i];
    }
    delete[] StringBuffer;
    delete[] DoubleBuffer;
    delete[] LongBuffer;

    SavePropertyList(Ntuple, Name + "_PropertyList");
}

void TKameRootRepository::LoadNtuple(TKameNtuple& Ntuple, const std::string& Name, int Revision) throw(TKameException)
{
    if (_InputFile == 0) {
	OpenToRead();
    }

    TObject* RootObject = _InputFile->Get(Name.c_str());
    if (RootObject == 0) {
	throw TKameException(
	    "TKameRootRepository::LoadTreeNtuple()",
	    "unable to find object: " + Name
	);
    }
    if (strcmp(RootObject->ClassName(), "TTree") != 0) {
	throw TKameException(
	    "TKameRootRepository::LoadTreeNtuple()",
	    "TTree is expected: " + Name + 
	    " (" + RootObject->ClassName() + ")"
	);
    }
    TTree* Tree = (TTree*) RootObject;

    TObjArray* BranchList = Tree->GetListOfBranches();
    if (BranchList == 0) {
	return;
    }

    int NumberOfColumns = BranchList->GetEntries();
    Int_t* IntBuffer = new Int_t[NumberOfColumns];
    Long64_t* LongBuffer = new Long64_t[NumberOfColumns];
    Float_t* FloatBuffer = new Float_t[NumberOfColumns];
    Double_t* DoubleBuffer = new Double_t[NumberOfColumns];
    Char_t** StringBuffer = new Char_t*[NumberOfColumns];
    for (int i = 0; i < NumberOfColumns; i++) {
	StringBuffer[i] = new Char_t[MaxStringLength];
    }

    enum TType {
	Type_int, Type_long, Type_float, Type_double, Type_string, Type_unknown
    };
    vector<int> TypeList;
    for (int i = 0; i < NumberOfColumns; i++) {
	TBranch* Branch = (TBranch*) BranchList->At(i);
	const char* Name = Branch->GetName();
	TLeaf* Leaf = Branch->GetLeaf(Name);
	if (Leaf == 0) {
	    throw TKameException(
		"TKameRootRepository::LoadTreeNtuple()",
		"unable to find leaf: " + string(Name)
	    );
	}
	string Type = Leaf->GetTypeName();

	if (Type == "Int_t") {
	    TypeList.push_back(Type_int);
	    Tree->SetBranchAddress(Name, &IntBuffer[i]);
	    Ntuple.SetColumnNameType(i, Name, "int");
	}
	else if (Type == "Long64_t") {
	    TypeList.push_back(Type_long);
	    Tree->SetBranchAddress(Name, &LongBuffer[i]);
	    Ntuple.SetColumnNameType(i, Name, "int");
	}
	else if (Type == "Flaot_t") {
	    TypeList.push_back(Type_float);
	    Tree->SetBranchAddress(Name, &FloatBuffer[i]);
	    Ntuple.SetColumnNameType(i, Name, "float");
	}
	else if (Type == "Double_t") {
	    TypeList.push_back(Type_double);
	    Tree->SetBranchAddress(Name, &DoubleBuffer[i]);
	    Ntuple.SetColumnNameType(i, Name, "float");
	}
	else if (Type == "Char_t") {
	    TypeList.push_back(Type_string);
	    Tree->SetBranchAddress(Name, StringBuffer[i]);
	    Ntuple.SetColumnNameType(i, Name, "string");
	}
	else {
	    throw TKameException(
		"TKameRootRepository::LoadTreeNtuple()",
		"unknown field type: " + Type
	    );
	}
    }	

    int NumberOfRows = Tree->GetEntries();
    for (int i = 0; i < NumberOfRows; i++) {
	Tree->GetEntry(i);
	for (int j = 0; j < NumberOfColumns; j++) {
	    switch (TypeList[j]) {
	      case Type_int:
		Ntuple[i][j] = (long) IntBuffer[j];
		break;
	      case Type_long:
	        Ntuple[i][j] = (long) LongBuffer[j];
		break;
	      case Type_float:
		Ntuple[i][j] = (double) FloatBuffer[j];
		break;
	      case Type_double:
		Ntuple[i][j] = (double) DoubleBuffer[j];
		break;
	      case Type_string:
		Ntuple[i][j] = StringBuffer[j];
		break;
	      default:
		Ntuple[i][j] = TKameVariant();
	    }
	}
    }	

    for (int i = 0; i < NumberOfColumns; i++) {
	delete[] StringBuffer[i];
    }
    delete[] StringBuffer;
    delete[] DoubleBuffer;
    delete[] FloatBuffer;
    delete[] LongBuffer;
    delete[] IntBuffer;

    delete RootObject;

    LoadPropertyList(Ntuple, Name + "_PropertyList", Revision);
}

void TKameRootRepository::SaveGraph(const TKameGraph& Graph, const std::string& Name) throw(TKameException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    int NumberOfPoints = Graph.NumberOfPoints();
    if (NumberOfPoints == 0) {
	// ROOT cannot make a Graph object when NumberOfPoints is 0.
	return;
    }

    TGraph* RootGraph = 0;
    if (Graph.HasErrors()) {
	RootGraph = new TGraphErrors(
	    Graph.NumberOfPoints(), 
	    Graph.XValueList(), Graph.YValueList(),
	    Graph.XErrorList(), Graph.YErrorList()
	);
    }
    else {
	RootGraph = new TGraph(
	    Graph.NumberOfPoints(), Graph.XValueList(), Graph.YValueList()
	);
    }
    RootGraph->SetNameTitle(Name.c_str(), Graph.Title().c_str());

    RootGraph->Write();
    delete RootGraph;

    SavePropertyList(Graph, Name + "_PropertyList");
}

void TKameRootRepository::LoadGraph(TKameGraph& Graph, const std::string& Name, int Revision) throw(TKameException)
{
    if (_InputFile == 0) {
	OpenToRead();
    }

    TObject* RootObject = _InputFile->Get(Name.c_str());
    if (RootObject == 0) {
	throw TKameException(
	    "TKameRootRepository::LoadGraph()",
	    "unable to find graph: " + Name
	);
    }
    if (strcmp(RootObject->ClassName(), "TGraph") != 0) {
	throw TKameException(
	    "TKameRootRepository::LoadGraph()",
	    "TGraph is expected: " + Name + 
	    " (" + RootObject->ClassName() + ")"
	);
    }
    TGraph* RootGraph = (TGraph*) RootObject;

    int NumberOfPoints = RootGraph->GetN();
    double* XValueList = RootGraph->GetX();
    double* YValueList = RootGraph->GetY();
    double* XErrorList = RootGraph->GetEX();
    double* YErrorList = RootGraph->GetEY();
    for (int i = 0; i < NumberOfPoints; i++) {
	if ((XErrorList != 0) && (YErrorList != 0)) {
	    Graph.Fill(
		XValueList[i], YValueList[i], XErrorList[i], YErrorList[i]
	    );
	}
	else {
	    Graph.Fill(XValueList[i], YValueList[i]);
	}
    }

    delete RootObject;

    LoadPropertyList(Graph, Name + "_PropertyList", Revision);
}

void TKameRootRepository::SaveHistogram(const TKameHistogram& Histogram, const std::string& Name) throw(TKameException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKameHistogramScale& Scale = Histogram.Scale();
    int NumberOfBins = Scale.NumberOfBins();

    TH1* RootHistogram = new TH1D(
	Name.c_str(), Histogram.Title().c_str(),
	NumberOfBins, Scale.Min(), Scale.Max()
    );

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double BinCenter = Scale.BinCenterOf(BinIndex);
	double Counts = Histogram.CountsOf(BinIndex);
	RootHistogram->Fill(BinCenter, Counts);
    }

    RootHistogram->SetBinContent(0, Histogram.UnderflowCounts());
    RootHistogram->SetBinContent(NumberOfBins+1, Histogram.OverflowCounts());
    RootHistogram->SetEntries(Histogram.NumberOfEntries());

    TKameObject PropertyList;
    PropertyList.ImportProperty(Histogram);

    long AccumulationTime = Histogram.AccumulationTime();
    if (AccumulationTime >= 0) {
	PropertyList.Property("AccumulationTime") = AccumulationTime;
    }

    _OutputFile->Write(Name.c_str());
    delete RootHistogram;

    SavePropertyList(PropertyList, Name + "_PropertyList");
}

void TKameRootRepository::LoadHistogram(TKameHistogram& Histogram, const std::string& Name, int Revision) throw(TKameException)
{
    if (_InputFile == 0) {
	OpenToRead();
    }

    TObject* RootObject = _InputFile->Get(Name.c_str());
    if (RootObject == 0) {
	throw TKameException(
	    "TKameRootRepository::LoadHistogram()",
	    "unable to find histogram: " + Name
	);
    }
    if (strcmp(RootObject->ClassName(), "TH1D") != 0) {
	throw TKameException(
	    "TKameRootRepository::LoadHistogram()",
	    "TH1D is expected: " + Name + 
	    " (" + RootObject->ClassName() + ")"
	);
    }
    TH1* RootHistogram = (TH1*) RootObject;

    TAxis* RootAxis = RootHistogram->GetXaxis();
    int NumberOfBins = RootAxis->GetNbins();
    double Min = RootAxis->GetXmin();
    double Max = RootAxis->GetXmax();
    double Width = Max - Min;
    
    Histogram.SetScale(TKameHistogramScale(NumberOfBins, Min, Max));
    Histogram.SetTitle(RootHistogram->GetTitle());

    for (int i = 0; i < NumberOfBins; i++) {
	double BinCenter = RootAxis->GetBinCenter(i+1);
	double Counts = RootHistogram->GetBinContent(i+1);
	Histogram.Fill(BinCenter, Counts);
    }

    double UnderflowCounts = RootHistogram->GetBinContent(0);
    double OverflowCounts = RootHistogram->GetBinContent(NumberOfBins+1);
    Histogram.Fill(Min - Width/1000.0, UnderflowCounts);
    Histogram.Fill(Max + Width/1000.0, OverflowCounts);

    delete RootObject;

    LoadPropertyList(Histogram, Name + "_PropertyList", Revision);

    if (! Histogram.Property("AccumulationTime").IsVoid()) {
	Histogram.SetAccumulationTime(
	    Histogram.Property("AccumulationTime")
	);
    }
}

void TKameRootRepository::SaveHistogram2d(const TKameHistogram2d& Histogram, const std::string& Name) throw(TKameException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKameHistogramScale& XScale = Histogram.XScale();
    const TKameHistogramScale& YScale = Histogram.YScale();

    int NumberOfXBins = XScale.NumberOfBins();
    int NumberOfYBins = XScale.NumberOfBins();

    TH2* RootHistogram = new TH2D(
	Name.c_str(), Histogram.Title().c_str(),
	NumberOfXBins, XScale.Min(), XScale.Max(),
	NumberOfYBins, YScale.Min(), YScale.Max()
    );

    double NumberOfEntries = 0;
    for (int YBinIndex = 0; YBinIndex < NumberOfXBins; YBinIndex++) {
	for (int XBinIndex = 0; XBinIndex < NumberOfXBins; XBinIndex++) {
	    double XBinCenter = XScale.BinCenterOf(XBinIndex);
	    double YBinCenter = YScale.BinCenterOf(YBinIndex);
	    double Counts = Histogram.CountsOf(XBinIndex, YBinIndex);

	    RootHistogram->Fill(XBinCenter, YBinCenter, Counts);
	    NumberOfEntries += Counts;
	}
    }
    RootHistogram->SetEntries(NumberOfEntries);

    _OutputFile->Write(Name.c_str());
    delete RootHistogram;

    SavePropertyList(Histogram, Name + "_PropertyList");
}

void TKameRootRepository::LoadHistogram2d(TKameHistogram2d& Histogram, const std::string& Name, int Revision) throw(TKameException)
{
    if (_InputFile == 0) {
	OpenToRead();
    }

    TObject* RootObject = _InputFile->Get(Name.c_str());
    if (RootObject == 0) {
	throw TKameException(
	    "TKameRootRepository::LoadHistogram()",
	    "unable to find histogram: " + Name
	);
    }
    if (strcmp(RootObject->ClassName(), "TH2D") != 0) {
	throw TKameException(
	    "TKameRootRepository::LoadHistogram()",
	    "TH2D is expected: " + Name + 
	    " (" + RootObject->ClassName() + ")"
	);
    }
    TH2* RootHistogram = (TH2*) RootObject;

    TAxis* RootXAxis = RootHistogram->GetXaxis();
    TAxis* RootYAxis = RootHistogram->GetYaxis();
    int NumberOfXBins = RootXAxis->GetNbins();
    double XMin = RootXAxis->GetXmin();
    double XMax = RootXAxis->GetXmax();
    int NumberOfYBins = RootYAxis->GetNbins();
    double YMin = RootYAxis->GetXmin();
    double YMax = RootYAxis->GetXmax();
    
    Histogram.SetScale(
	TKameHistogramScale(NumberOfXBins, XMin, XMax),
	TKameHistogramScale(NumberOfYBins, YMin, YMax)
    );
    Histogram.SetTitle(RootHistogram->GetTitle());

    for (int i = 0; i < NumberOfXBins; i++) {
	double XBinCenter = RootXAxis->GetBinCenter(i+1);
	for (int j = 0; j < NumberOfYBins; j++) {
	    double YBinCenter = RootYAxis->GetBinCenter(j+1);
	    double Counts = RootHistogram->GetBinContent(i+1, j+1);
	    Histogram.Fill(XBinCenter, YBinCenter, Counts);
	}
    }
    //... fill overflow/underflow counts ...//

    delete RootObject;

    LoadPropertyList(Histogram, Name + "_PropertyList", Revision);
}
