/* KameVariant.hh */
/* Created by Enomoto Sanshiro on 29 June 2009. */
/* Last updated by Enomoto Sanshiro on 29 June 2009. */


#ifndef __KameVariant_hh__
#define __KameVariant_hh__

#include <string>
#include <iostream>
#include <sstream>
#include "KameDefs.hh"


namespace kame {


// KameVariant: Assignable to/from primitive values.
// Explicit cast is needed if used in expression context.
// Occupies 12 bytes (sizeof(int) + sizeof(double))


class TKameUnknown {
  public:
    // use RTTI to get actual type //
    TKameUnknown(void) {}
    virtual ~TKameUnknown() {}
    virtual TKameUnknown* Clone(void) const = 0;
};


class TKameVariant {
  public:
    inline TKameVariant(void);
    inline TKameVariant(int Value);
    inline TKameVariant(unsigned int Value);
    inline TKameVariant(long Value);
    inline TKameVariant(unsigned long Value);
    inline TKameVariant(float Value);
    inline TKameVariant(double Value);
    inline TKameVariant(const std::string& Value);
    inline TKameVariant(const char* Value);
    inline TKameVariant(const TKameUnknown& Value);
    inline TKameVariant(const TKameVariant& Value);
    inline TKameVariant& operator=(const TKameVariant& Value);
    inline ~TKameVariant();
    inline bool IsVoid(void) const;
    inline bool IsLong(void) const;
    inline bool IsDouble(void) const;
    inline bool IsString(void) const;
    inline bool IsUnknown(void) const;
    inline operator int() const throw(TKameException);
    inline operator unsigned int() const throw(TKameException);
    inline operator long() const throw(TKameException);
    inline operator unsigned long() const throw(TKameException);
    inline operator float() const throw(TKameException);
    inline operator double() const throw(TKameException);
    inline operator std::string() const;
    const TKameUnknown& AsUnknown(void) const throw(TKameException);
  protected:
    long AsLong(void) const throw(TKameException);
    double AsDouble(void) const throw(TKameException);
    std::string AsString(void) const;
  private:
    enum TValueType { 
	vtVoid, vtLong, vtDouble, vtString, vtUnknown
    } _Type;
    union TPrimitive {
        long _LongValue;
        double _DoubleValue;
	std::string* _StringValue;
	TKameUnknown* _UnknownValue;
    } _Primitive;
};


TKameVariant::TKameVariant(void)
: _Type(vtVoid)
{
}

TKameVariant::TKameVariant(int Value)
: _Type(vtLong)
{
    _Primitive._LongValue = Value;
}

TKameVariant::TKameVariant(unsigned int Value)
: _Type(vtLong)
{
    _Primitive._LongValue = Value;
}

TKameVariant::TKameVariant(long Value)
: _Type(vtLong)
{
    _Primitive._LongValue = Value;
}

TKameVariant::TKameVariant(unsigned long Value)
: _Type(vtLong)
{
    _Primitive._LongValue = Value;
}

TKameVariant::TKameVariant(float Value)
: _Type(vtDouble)
{
    _Primitive._DoubleValue = Value;
}

TKameVariant::TKameVariant(double Value)
: _Type(vtDouble)
{
    _Primitive._DoubleValue = Value;
}

TKameVariant::TKameVariant(const std::string& Value)
: _Type(vtString)
{
    _Primitive._StringValue = new std::string(Value);
}

TKameVariant::TKameVariant(const char* Value)
: _Type(vtString)
{
    _Primitive._StringValue = new std::string(Value);
}

TKameVariant::TKameVariant(const TKameUnknown& Value)
: _Type(vtUnknown)
{
    _Primitive._UnknownValue = Value.Clone();
}

TKameVariant::TKameVariant(const TKameVariant& Value)
: _Type(Value._Type), _Primitive(Value._Primitive)
{
    if (_Type == vtString) {
	_Primitive._StringValue = (
	    new std::string(*Value._Primitive._StringValue)
	);
    }
    else if (_Type == vtUnknown) {
	_Primitive._UnknownValue = Value._Primitive._UnknownValue->Clone();
    }
}

TKameVariant& TKameVariant::operator=(const TKameVariant& Value)
{
    if (&Value == this) {
	return *this;
    }

    if (_Type == vtString) {
	delete _Primitive._StringValue;
    }
    else if (_Type == vtUnknown) {
	delete _Primitive._UnknownValue;
    }

    if (Value._Type == vtString) {
	_Primitive._StringValue = (
	    new std::string(*Value._Primitive._StringValue)
	);
    }
    else if (Value._Type == vtUnknown) {
	_Primitive._UnknownValue = Value._Primitive._UnknownValue->Clone();
    }
    else {
	_Primitive = Value._Primitive;
    }
    _Type = Value._Type;

    return *this;
}

TKameVariant::~TKameVariant()
{
    if (_Type == vtString) {
	delete _Primitive._StringValue;
    }
    else if (_Type == vtUnknown) {
	delete _Primitive._UnknownValue;
    }
}

bool TKameVariant::IsVoid(void) const
{
    return (_Type == vtVoid);
}

bool TKameVariant::IsLong(void) const
{
    return (_Type == vtLong);
}

bool TKameVariant::IsDouble(void) const
{
    return (_Type == vtDouble);
}

bool TKameVariant::IsString(void) const
{
    return (_Type == vtString);
}

bool TKameVariant::IsUnknown(void) const
{
    return (_Type == vtUnknown);
}

TKameVariant::operator int() const throw(TKameException)
{
    return (_Type == vtLong) ? _Primitive._LongValue : AsLong();
}

TKameVariant::operator unsigned int() const throw(TKameException)
{
    return (unsigned int) (_Type == vtLong) ? _Primitive._LongValue : AsLong();
}

TKameVariant::operator long() const throw(TKameException)
{
    return (_Type == vtLong) ? _Primitive._LongValue : AsLong();
}

TKameVariant::operator unsigned long() const throw(TKameException)
{
    return (unsigned long) (_Type == vtLong) ? _Primitive._LongValue : AsLong();
}

TKameVariant::operator float() const throw(TKameException)
{
    return (_Type == vtDouble) ? _Primitive._DoubleValue : AsDouble();
}

TKameVariant::operator double() const throw(TKameException)
{
    return (_Type == vtDouble) ? _Primitive._DoubleValue : AsDouble();
}

TKameVariant::operator std::string() const
{
    return (_Type == vtString) ? *_Primitive._StringValue : AsString();
}



inline std::ostream& operator<<(std::ostream&os, const TKameVariant&Value) 
{
    if (Value.IsDouble()) {
	os << (double) Value;
    }
    else if (Value.IsLong()) {
	os << (long) Value;
    }
    else {
	os << (const std::string&) Value;
    }

    return os;
}


template <typename T>
inline TKameVariant& operator+=(TKameVariant& This, const T& Value)
{
    return This = (T) This + Value;
}

template <typename T>
inline TKameVariant& operator-=(TKameVariant& This, const T& Value)
{
    return This = (T) This - Value;
}

template <typename T>
inline TKameVariant& operator*=(TKameVariant& This, const T& Value)
{
    return This = (T) This * Value;
}

template <typename T>
inline TKameVariant& operator/=(TKameVariant& This, const T& Value)
{
    return This = (T) This / Value;
}

template <typename T>
inline bool operator==(TKameVariant& This, const T& Value)
{
    return (T) This == Value;
}

inline bool operator==(TKameVariant& This, const char Value[])
{
    return (std::string) This == std::string(Value);
}

template <typename T>
inline bool operator!=(TKameVariant& This, const T& Value)
{
    return (T) This != Value;
}

inline bool operator!=(TKameVariant& This, const char Value[])
{
    return (std::string) This != std::string(Value);
}

template <typename T>
inline bool operator>(TKameVariant& This, const T& Value)
{
    return (T) This > Value;
}

template <typename T>
inline bool operator<(TKameVariant& This, const T& Value)
{
    return (T) This < Value;
}

template <typename T>
inline bool operator>=(TKameVariant& This, const T& Value)
{
    return (T) This >= Value;
}

template <typename T>
inline bool operator<=(TKameVariant& This, const T& Value)
{
    return (T) This <= Value;
}


}
#endif
