/* KaspHistogram.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#ifndef __KaspHistogram_hh__
#define __KaspHistogram_hh__


#include <string>
#include <cmath>
#include "KaspObject.hh"


class TKaspHistogramScale {
  public:
    TKaspHistogramScale(const std::string& Title, int NumberOfBins, double Min, double Max);
    TKaspHistogramScale(int NumberOfBins, double Min, double Max);
    virtual ~TKaspHistogramScale();
    std::string Title(void) const;
    void SetTitle(const std::string& Title);
    inline int BinNumberOf(double Data) const;
    inline double BinValueOf(int BinNumber) const;
    inline double BinWidthOf(int BinNumber) const;
    inline double BinCenterOf(int BinNumber) const;
    inline double Min(void) const;
    inline double Max(void) const;
    inline int NumberOfBins(void) const;
    inline int operator()(double DataValue) const;
    inline double operator[](int BinNumber) const;
  protected:
    std::string _Title;
    long _NumberOfBins;
    double _Min, _Max;
    double _BinWidth;
};


class TKaspHistogram: public TKaspObject {
  public:
    TKaspHistogram(const std::string& Title, const TKaspHistogramScale& Scale);
    TKaspHistogram(const std::string& Title, int NumberOfBins, double Min, double Max);
    ~TKaspHistogram();
    void Clear(void);
    bool HasData(void);
    void SetAccumulationTime(long AccumulationTime);
  public:
    inline void Fill(double Data, double Weight = 1.0);
    inline void Fill(double Data, double Weight, double Error);
    inline TKaspHistogramScale& Scale(void);
    inline const TKaspHistogramScale& Scale(void) const;
    inline double CountsOf(int BinNumber) const;
    inline double ErrorOf(int BinNumber) const;
    inline double UnderflowCounts(void) const;
    inline double OverflowCounts(void) const;
    inline double NumberOfEntries(void) const;
    inline double Mean(void) const;
    inline double Deviation(void) const;
    inline double PeakCounts(void) const;
    inline double PeakValue(void) const;
    inline long AccumulationTime(void) const;
    inline double operator()(double DataValue) const;
    inline double operator[](int BinNumber) const;
    void Scale(double Factor);
    void AddHistogram(TKaspHistogram& Histogram, double Weight = +1) throw(TKaspException);
    void MultiplyHistogram(TKaspHistogram& Histogram, double Power = +1) throw(TKaspException);
    inline bool HasErrors(void) const;
  protected:
    TKaspHistogramScale _Scale;
    double* _CountStorage;
    double* _ErrorStorage;
    double _UnderflowCounts;
    double _OverflowCounts;
    double _NumberOfEntries;
    double _Sum, _SumOfSquared;
    double _PeakCounts;
    int _PeakBinIndex;
    long _AccumulationTime;
};


inline void TKaspHistogram::Fill(double Data, double Weight)
{
    if (Data < _Scale.Min()) {
	_UnderflowCounts += Weight;
    }
    else if (Data >= _Scale.Max()) {
	_OverflowCounts += Weight;
    }
    else {
	int BinIndex = _Scale.BinNumberOf(Data);
	double Counts = (_CountStorage[BinIndex] += Weight);
	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
    }

    _NumberOfEntries += Weight;

    _Sum += Weight * Data;
    _SumOfSquared += Weight * Data * Data;
}

inline void TKaspHistogram::Fill(double Data, double Weight, double Error)
{
    Fill(Data, Weight);
    
    int NumberOfBins = _Scale.NumberOfBins();
    int BinIndex = _Scale.BinNumberOf(Data);

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = std::sqrt(_CountStorage[BinIndex]);
	}
    }

    if ((BinIndex >= 0) && (BinIndex < NumberOfBins)) {
	double PreviousError = ErrorOf(BinIndex);
	_ErrorStorage[BinIndex] = std::sqrt(
	    (PreviousError * PreviousError) + (Error * Error)
	);
    }
}

inline TKaspHistogramScale& TKaspHistogram::Scale(void)
{
    return _Scale;
}

inline const TKaspHistogramScale& TKaspHistogram::Scale(void) const
{
    return _Scale;
}

inline double TKaspHistogram::CountsOf(int BinNumber) const
{
    return _CountStorage[BinNumber];
}

inline double TKaspHistogram::ErrorOf(int BinNumber) const
{
    if (_ErrorStorage) {
	return _ErrorStorage[BinNumber];
    }
    else if (_CountStorage[BinNumber] > 0) {
	return std::sqrt(_CountStorage[BinNumber]);
    }
    
    return 0;
}

inline double TKaspHistogram::OverflowCounts(void) const
{
    return _OverflowCounts;
}

inline double TKaspHistogram::UnderflowCounts(void) const
{
    return _UnderflowCounts;
}

inline double TKaspHistogram::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

inline double TKaspHistogram::Mean(void) const
{
    return _Sum / _NumberOfEntries;
}

inline double TKaspHistogram::Deviation(void) const
{
    return std::sqrt(_SumOfSquared / _NumberOfEntries - Mean() * Mean());
}

inline double TKaspHistogram::PeakCounts(void) const
{
    return _PeakCounts;
}

inline double TKaspHistogram::PeakValue(void) const
{
    return _Scale.BinCenterOf(_PeakBinIndex);
}

inline long TKaspHistogram::AccumulationTime(void) const
{
    return _AccumulationTime;
}

inline bool TKaspHistogram::HasErrors(void) const
{
    return _ErrorStorage != 0;
}

inline double TKaspHistogram::operator()(double DataValue) const
{
    return _CountStorage[_Scale.BinNumberOf(DataValue)];
}

inline double TKaspHistogram::operator[](int BinNumber) const
{
    return _CountStorage[BinNumber];
}



inline int TKaspHistogramScale::BinNumberOf(double Data) const
{
    if (Data < _Min) {
	// Underflow
	return _NumberOfBins + 0;
    }
    else if (Data >= _Max) {
	// Overflow
	return _NumberOfBins + 1;
    }
    else {
	return (int) ((Data - _Min) / _BinWidth);
    }
}

inline double TKaspHistogramScale::BinValueOf(int BinNumber) const
{
    if ((BinNumber < 0) || (BinNumber > _NumberOfBins)) {
	// out of range
	return 2.0 * _Max;
    }

    return _Min + BinNumber * _BinWidth;
}

inline double TKaspHistogramScale::BinWidthOf(int BinNumber) const
{
    return _BinWidth;
}

inline double TKaspHistogramScale::BinCenterOf(int BinNumber) const
{
    return BinValueOf(BinNumber) + _BinWidth / 2;
}

inline double TKaspHistogramScale::Min(void) const
{
    return _Min;
}

inline double TKaspHistogramScale::Max(void) const
{
    return _Max;
}

inline int TKaspHistogramScale::NumberOfBins(void) const
{
    return _NumberOfBins;
}

inline int TKaspHistogramScale::operator()(double DataValue) const
{
    return BinNumberOf(DataValue);
}

inline double TKaspHistogramScale::operator[](int BinNumber) const
{
    return BinCenterOf(BinNumber);
}

#endif
