/* KaspPlainTextRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 23 December 2002. */


#include <string>
#include <fstream>
#include <sstream>
#include <cstdio>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "MushCFileStream.hh"
#include "MushFileSystem.hh"
#include "MushMisc.hh"
#include "KaspDefs.hh"
#include "KaspNtuple.hh"
#include "KaspGraph.hh"
#include "KaspHistogram.hh"
#include "Kasp2dHistogram.hh"
#include "KaspHistory.hh"
#include "KaspTrend.hh"
#include "KaspWave.hh"
#include "KaspMap.hh"
#include "KaspTabular.hh"
#include "KaspRepository.hh"
#include "KaspPlainTextRepository.hh"

using namespace std;


TKaspPlainTextRepository::TKaspPlainTextRepository(const string& RepositoryName)
{
    // repository is a directory //
    _RepositoryName = RepositoryName;
    _IsRepositoryCreated = false;

    _IsGzipCompressionEnabled = false;
}

TKaspPlainTextRepository::TKaspPlainTextRepository(void)
{
    // repository is a file //
    _RepositoryName = "";
    _IsRepositoryCreated = false;

    _IsGzipCompressionEnabled = false;
}

TKaspPlainTextRepository::~TKaspPlainTextRepository()
{
}

void TKaspPlainTextRepository::EnableGzipCompression(void)
{
    _IsGzipCompressionEnabled = true;
}

string TKaspPlainTextRepository::FileNameOf(const string& ObjectName, int Revision)
{
    ostringstream FileNameStream;

    if (! _RepositoryName.empty()) {
	FileNameStream << _RepositoryName << "/";
    }

    FileNameStream << ObjectName;
    if (Revision >= 0) {
	FileNameStream << ";" << Revision;
    }

    if (! _RepositoryName.empty()) {
	FileNameStream << ".knt";
    }

    return FileNameStream.str();
}

ostream* TKaspPlainTextRepository::OpenOutputFile(const string& ObjectName, int Revision) throw(TKaspException)
{
    if ((! _IsRepositoryCreated) && (! _RepositoryName.empty())) {
	if (access(_RepositoryName.c_str(), F_OK) != 0) {
	    mkdir(_RepositoryName.c_str(), 0755);
	}
	else {
	    struct stat FileStat;
	    if (stat(_RepositoryName.c_str(), &FileStat) < 0) {
		throw TKaspException(
		    "TKaspPlainTextRepository::OpenOutputFile()",
		    "systemcall stat(2) failed: " + _RepositoryName
		);
	    }
	    if (! S_ISDIR(FileStat.st_mode)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::OpenOutputFile()",
		    "file already exists: " + _RepositoryName
		);
	    }
	}
    }

    string FileName = FileNameOf(ObjectName);
    if (Revision > 0) {
	string PreviousFileName = FileNameOf(ObjectName, Revision - 1);
	if (_IsGzipCompressionEnabled) {
	    rename((FileName + ".gz").c_str(), (PreviousFileName + ".gz").c_str());
	}
	else {
	    rename(FileName.c_str(), PreviousFileName.c_str());
	}
    }

    ostream* File = 0;
    if (_IsGzipCompressionEnabled) {
	string Command = "gzip - > " + FileName + ".gz";
	FILE* Pipe = popen(Command.c_str(), "w");
	if ((Pipe == NULL) || (Pipe == (void*) -1)) {
	    throw TKaspException(
		"TKaspPlainTextRepository::OpenOutputFile()",
		"unable to start 'gzip' for file: " + FileName
	    );
	}
	File = new TMushOutputCFileStream(Pipe);
	_CommandPipeTable[File] = Pipe;
    }
    else {
	File = new ofstream(FileName.c_str());
    }

    if (! *File) {
	delete File;
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenOutputFile()",
	    "unable to create file: " + FileName
	);
    }

    return File;
}

istream* TKaspPlainTextRepository::OpenInputFile(const string& ObjectName, int Revision) throw(TKaspException)
{
    string FileName = FileNameOf(ObjectName, Revision);

    istream* File = 0;
    if (TMushFileAttribute(FileName).IsReadable()) {
	File = new ifstream(FileName.c_str());
    }
    else if (TMushFileAttribute(FileName + ".gz").IsReadable()) {
	string Command = "gunzip -c " + FileName + ".gz";
	FILE* Pipe = popen(Command.c_str(), "r");
	if ((Pipe == NULL) || (Pipe == (void*) -1)) {
	    throw TKaspException(
		"TKaspPlainTextRepository::OpenInputFile()",
		"unable to start 'gunzip' for file: " + FileName
	    );
	}
	File = new TMushInputCFileStream(Pipe);
	_CommandPipeTable[File] = Pipe;
    }
    else {
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenInputFile()",
	    "unable to open file: " + FileName
	);
    }

    if (! *File) {
	delete File;
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenInputFile()",
	    "unable to open file: " + FileName
	);
    }

    return File;
}

void TKaspPlainTextRepository::CloseInputFile(istream* File) throw(TKaspException)
{
    map<ios*, FILE*>::iterator Iterator = _CommandPipeTable.find(File);
    if (Iterator != _CommandPipeTable.end()) {
	pclose(Iterator->second);
	_CommandPipeTable.erase(Iterator);
    }

    delete File;
}

void TKaspPlainTextRepository::CloseOutputFile(ostream* File) throw(TKaspException)
{
    map<ios*, FILE*>::iterator Iterator = _CommandPipeTable.find(File);
    if (Iterator != _CommandPipeTable.end()) {
	pclose(Iterator->second);
	_CommandPipeTable.erase(Iterator);
    }

    delete File;
}

bool TKaspPlainTextRepository::ReadDouble(std::istream& InputStream, double& Value) throw(TKaspException)
{
    return InputStream >> Value;
}

void TKaspPlainTextRepository::SaveNtuple(const string& Name, TKaspNtuple* Ntuple) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Ntuple" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Ntuple->Title() << endl;

    for (unsigned i = 0; i < Ntuple->NumberOfProperties(); i++) {
	*File << "# " << Ntuple->PropertyNameOf(i);
	*File << ": " << Ntuple->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: ";
    unsigned NumberOfColumns = Ntuple->NumberOfColumns();
    for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	*File << Ntuple->ColumnNameOf(Column) << " ";
    }
    *File << endl << endl;

    for (unsigned Row = 0; Row < Ntuple->NumberOfRows(); Row++) {
	for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	    *File << (*Ntuple)[Row][Column] << " ";
	}
	*File << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveGraph(const string& Name, TKaspGraph* Graph) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Graph" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Graph->Title() << endl;
    *File << "# NumberOfSegments: " << Graph->NumberOfSegments() << endl;

    for (unsigned i = 0; i < Graph->NumberOfProperties(); i++) {
	*File << "# " << Graph->PropertyNameOf(i);
	*File << ": " << Graph->PropertyValueOf(i) << endl;
    }

    if (Graph->HasErrors()) {
	*File << "# Fields: X Y XError YError" << endl;
    }
    else {
	*File << "# Fields: X Y" << endl;
    }
    *File << endl;

    const double* XValueList = Graph->XValueList();
    const double* YValueList = Graph->YValueList();
    const double* XErrorList = Graph->XErrorList();
    const double* YErrorList = Graph->YErrorList();
    unsigned NumberOfPoints = Graph->NumberOfPoints();

    unsigned SegmentIndex = 0;
    unsigned NextSegmentBoundary = Graph->SegmentSizeOf(SegmentIndex);

    for (unsigned i = 0; i < NumberOfPoints; i++) {
	if (i == NextSegmentBoundary) {
	    *File << endl;
	    NextSegmentBoundary += Graph->SegmentSizeOf(++SegmentIndex);
	}

	*File << XValueList[i] << " " << YValueList[i];
	if (Graph->HasErrors()) {
	    *File << " " << XErrorList[i] << " " << YErrorList[i];
	}
	*File << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveHistogram(const string& Name, TKaspHistogram* Histogram) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    const TKaspHistogramScale& Scale = Histogram->Scale();

    *File << "# Type: Histogram" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Histogram->Title() << endl;
    *File << "# AxisTitle: " << Scale.Title() << endl;
    *File << "# NumberOfBins: " << Scale.NumberOfBins() << endl;
    *File << "# Min: " << Scale.Min() << endl;
    *File << "# Max: " << Scale.Max() << endl;
    *File << "# Underflow: " <<  Histogram->UnderflowCounts() << endl;
    *File << "# Overflow: " <<  Histogram->OverflowCounts() << endl;

    long AccumulationTime = Histogram->AccumulationTime();
    if (AccumulationTime >= 0) {
	*File << "# AccumulationTime: " <<  AccumulationTime << endl;
    }

    for (unsigned i = 0; i < Histogram->NumberOfProperties(); i++) {
	*File << "# " << Histogram->PropertyNameOf(i);
	*File << ": " << Histogram->PropertyValueOf(i) << endl;
    }

    if (Histogram->HasErrors()) {
	*File << "# Fields: BinCenter Counts Error" << endl;
    }
    else {
	*File << "# Fields: BinCenter Counts" << endl;
    }
    *File << endl;

    for (int BinIndex = 0; BinIndex < Scale.NumberOfBins(); BinIndex++) {
	*File << Scale.BinCenterOf(BinIndex) << " ";
	*File << Histogram->CountsOf(BinIndex);
	if (Histogram->HasErrors()) {
	    *File << " " << Histogram->ErrorOf(BinIndex);
	}
	*File << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::Save2dHistogram(const string& Name, TKasp2dHistogram* Histogram) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    const TKaspHistogramScale& XScale = Histogram->XScale();
    const TKaspHistogramScale& YScale = Histogram->YScale();

    *File << "# Type: Histogram2d" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Histogram->Title() << endl;
    *File << "# XAxisTitle: " << XScale.Title() << endl;
    *File << "# YAxisTitle: " << YScale.Title() << endl;
    *File << "# NumberOfXBins: " << XScale.NumberOfBins() << endl;
    *File << "# XMin: " << XScale.Min() << endl;
    *File << "# XMax: " << XScale.Max() << endl;
    *File << "# NumberOfYBins: " << YScale.NumberOfBins() << endl;
    *File << "# YMin: " << YScale.Min() << endl;
    *File << "# YMax: " << YScale.Max() << endl;

    for (unsigned i = 0; i < Histogram->NumberOfProperties(); i++) {
	*File << "# " << Histogram->PropertyNameOf(i);
	*File << ": " << Histogram->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: XBinCenter YBinCenter Counts" << endl;
    *File << endl;

    for (int YBinIndex = 0; YBinIndex < YScale.NumberOfBins(); YBinIndex++) {
	for (int XBinIndex = 0; XBinIndex < XScale.NumberOfBins(); XBinIndex++) {
	    *File << XScale.BinCenterOf(XBinIndex) << " ";
	    *File << YScale.BinCenterOf(YBinIndex) << " ";
	    *File << Histogram->CountsOf(XBinIndex, YBinIndex) << endl;
	}
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveTrend(const string& Name, TKaspTrend* Trend) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Trend" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Trend->Title() << endl;
    *File << "# StartTime: " << Trend->StartTime() << endl;
    *File << "# TickInterval: " << Trend->TickInterval() << endl;
    *File << "# WindowLength: " << Trend->WindowLength() << endl;

    for (unsigned i = 0; i < Trend->NumberOfProperties(); i++) {
	*File << "# " << Trend->PropertyNameOf(i);
	*File << ": " << Trend->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: Time Counts Sum Mean Deviation" << endl;
    *File << endl;

    for (int Index = 0; Index < Trend->NumberOfPoints(); Index++) {
	*File << Trend->PassedTimeOf(Index) << " ";
	*File << Trend->CountsOf(Index) << " ";
	*File << Trend->SumOf(Index) << " ";
	*File << Trend->MeanOf(Index) << " ";
	*File << Trend->DeviationOf(Index) << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveHistory(const string& Name, TKaspHistory* History) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: History" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << History->Title() << endl;
    *File << "# Depth: " << History->MaxNumberOfSamples() << endl;

    for (unsigned i = 0; i < History->NumberOfProperties(); i++) {
	*File << "# " << History->PropertyNameOf(i);
	*File << ": " << History->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: Time Counts Sum Mean Deviation" << endl;
    *File << endl;

    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	*File << History->PassedTimeOf(Index) << " ";
	*File << History->CountsOf(Index) << " ";
	*File << History->SumOf(Index) << " ";
	*File << History->MeanOf(Index) << " ";
	*File << History->DeviationOf(Index) << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveWave(const string& Name, TKaspWave* Wave) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Wave" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Wave->Title() << endl;

    for (unsigned i = 0; i < Wave->NumberOfProperties(); i++) {
	*File << "# " << Wave->PropertyNameOf(i);
	*File << ": " << Wave->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: Index Value" << endl;
    *File << endl;

    for (int Index = 0; Index < Wave->NumberOfPoints(); Index++) {
	*File << Index << " ";
	*File << Wave->ValueOf(Index) << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveMap(const string& Name, TKaspMap* Map) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Map" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Map->Title() << endl;

    for (unsigned i = 0; i < Map->NumberOfProperties(); i++) {
	*File << "# " << Map->PropertyNameOf(i);
	*File << ": " << Map->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: Address Value" << endl;
    *File << endl;

    for (int Index = 0; Index < Map->NumberOfPoints(); Index++) {
	*File << Map->AddressOf(Index) << " ";
	*File << Map->ValueOf(Index) << endl;
    }

    CloseOutputFile(File);
}

void TKaspPlainTextRepository::SaveTabular(const string& Name, TKaspTabular* Tabular) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ostream* File = OpenOutputFile(Name, Revision);

    *File << "# Type: Tabular" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Date: " << TMushDateTime().AsString() << endl;
    *File << "# Title: " << Tabular->Title() << endl;

    for (unsigned i = 0; i < Tabular->NumberOfProperties(); i++) {
	*File << "# " << Tabular->PropertyNameOf(i);
	*File << ": " << Tabular->PropertyValueOf(i) << endl;
    }

    *File << "# Fields: FieldName Value" << endl;
    *File << endl;

    for (int Index = 0; Index < Tabular->NumberOfFields(); Index++) {
	*File << Tabular->FieldNameOf(Index) << " ";
	*File << Tabular->ValueOf(Index) << endl;
    }

    CloseOutputFile(File);
}

TKaspNtuple* TKaspPlainTextRepository::LoadNtuple(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspNtuple* Ntuple = 0;
    static const int MaxNumberOfColumns = 1024;

    int NumberOfColumns = 0;
    double* Row = new double[MaxNumberOfColumns];

    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istringstream LineStream(Line);
	int ColumnIndex = 0;
	double Value;
	while (ReadDouble(LineStream >> ws, Value)) {
	    if (ColumnIndex >= MaxNumberOfColumns) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadNtuple()",
		    "too many columns"
		);
            }

            if (Ntuple == 0) {
       	        NumberOfColumns++;
            }
            Row[ColumnIndex] = Value;
            ColumnIndex++;
        }
	
	if (Ntuple == 0) {
	    Ntuple = new TKaspNtuple(Name, NumberOfColumns);
	}
	else if (ColumnIndex != NumberOfColumns) {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadNtuple()",
		"inconsistent number of columns: " + Line
	    );
	}

	Ntuple->Fill(Row);
    }

    if (Ntuple == 0) {
	NumberOfColumns = 0;
	Ntuple = new TKaspNtuple(Name, NumberOfColumns);
    }

    Ntuple->SetTitle(Header.ItemValueOf("Title"));
    vector<string> FieldNameList = Header.ItemValueListOf("Fields");
    for (unsigned Index = 0; Index < FieldNameList.size(); Index++) {
	Ntuple->SetColumnName(Index, FieldNameList[Index]);
    }
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Ntuple->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    delete[] Row;

    CloseInputFile(File);

    return Ntuple;
}

TKaspGraph* TKaspPlainTextRepository::LoadGraph(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspGraph* Graph = new TKaspGraph(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    int CurrentSegmentSize = 0;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    if (CurrentSegmentSize > 0) {
		Graph->BreakSegment();
		CurrentSegmentSize = 0;
	    }
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istringstream LineStream(Line);
	double X, Y, XError, YError;
	if (LineStream >> X >> Y) {
	    if (LineStream >> XError >> YError) {
		Graph->Fill(X, Y, XError, YError);
	    }
	    else {
		Graph->Fill(X, Y);
	    }
	    CurrentSegmentSize++;
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadGraph()",
		"invalid data line: " + Line
	    );
	}
    }

    if (CurrentSegmentSize > 0) {
	Graph->BreakSegment();
    }

    Graph->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Graph->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Graph;
}

TKaspHistogram* TKaspPlainTextRepository::LoadHistogram(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspHistogram* Histogram = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (Histogram == 0) {
	    string NumberOfBinsString = Header.ItemValueOf("NumberOfBins");
	    string MinString = Header.ItemValueOf("Min");
	    string MaxString = Header.ItemValueOf("Max");
	    string AccumulationTimeString = Header.ItemValueOf("AccumulationTime");

	    istringstream NumberOfBinsStream(NumberOfBinsString);
	    istringstream MinStream(MinString);
	    istringstream MaxStream(MaxString);
	    istringstream AccumulationTimeStream(AccumulationTimeString);

	    int NumberOfBins;
	    float Min, Max;
	    if (
		! (NumberOfBinsStream >> NumberOfBins) ||
		! (MinStream >> Min) ||
		! (MaxStream >> Max)
	    ){
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadHistogram()",
		    "unable to get histogram parameters"
		);
	    }

	    Histogram = new TKaspHistogram(Name, NumberOfBins, Min, Max);

	    long AccumulationTime;
	    if (AccumulationTimeStream >> AccumulationTime) {
		Histogram->SetAccumulationTime(AccumulationTime);
	    }
	}

	istringstream LineStream(Line);
	double BinCenter, Counts, Error;
	if (LineStream >> BinCenter >> Counts) {
	    if (LineStream >> Error) {
		Histogram->Fill(BinCenter, Counts, Error);
	    }
	    else {
		Histogram->Fill(BinCenter, Counts);
	    }
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistogram()",
		"invalid data line: " + Line
	    );
	}
    }

    Histogram->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Histogram->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Histogram;
}

TKasp2dHistogram* TKaspPlainTextRepository::Load2dHistogram(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKasp2dHistogram* Histogram = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (Histogram == 0) {
	    string NumberOfXBinsString = Header.ItemValueOf("NumberOfXBins");
	    string XMinString = Header.ItemValueOf("XMin");
	    string XMaxString = Header.ItemValueOf("XMax");
	    string NumberOfYBinsString = Header.ItemValueOf("NumberOfYBins");
	    string YMinString = Header.ItemValueOf("YMin");
	    string YMaxString = Header.ItemValueOf("YMax");

	    istringstream NumberOfXBinsStream(NumberOfXBinsString);
	    istringstream XMinStream(XMinString);
	    istringstream XMaxStream(XMaxString);
	    istringstream NumberOfYBinsStream(NumberOfYBinsString);
	    istringstream YMinStream(YMinString);
	    istringstream YMaxStream(YMaxString);

	    int NumberOfXBins, NumberOfYBins;
	    float XMin, XMax, YMin, YMax;
	    if (
		! (NumberOfXBinsStream >> NumberOfXBins) ||
		! (XMinStream >> XMin) ||
		! (XMaxStream >> XMax) ||
		! (NumberOfYBinsStream >> NumberOfYBins) ||
		! (YMinStream >> YMin) ||
		! (YMaxStream >> YMax)
	    ){
		throw TKaspException(
		    "TKaspPlainTextRepository::Load2dHistogram()",
		    "unable to get histogram parameters"
		);
	    }

	    Histogram = new TKasp2dHistogram(
		Name, 
		TKaspHistogramScale(NumberOfXBins, XMin, XMax),
		TKaspHistogramScale(NumberOfYBins, YMin, YMax)
	    );
	}

	istringstream LineStream(Line);
	double XBinCenter, YBinCenter, Counts;
	if (LineStream >> XBinCenter >> YBinCenter >> Counts) {
	    Histogram->Fill(XBinCenter, YBinCenter, Counts);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistogram()",
		"invalid data line: " + Line
	    );
	}
    }

    Histogram->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Histogram->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Histogram;
}

TKaspTrend* TKaspPlainTextRepository::LoadTrend(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspTrend* Trend = 0;
    TKaspPlainTextRepositoryHeader Header;
    long StartTime, TickInterval, WindowLength;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (Trend == 0) {
	    string StartTimeString = Header.ItemValueOf("StartTime");
	    istringstream StartTimeStream(StartTimeString);
	    if (! (StartTimeStream >> StartTime)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadTrend()",
		    "unable to get trend parameters (StartTime)"
		);
	    }

	    string IntervalString = Header.ItemValueOf("TickInterval");
	    istringstream IntervalStream(IntervalString);
	    if (! (IntervalStream >> TickInterval)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadTrend()",
		    "unable to get trend parameters (TickInterval)"
		);
	    }
	    string LengthString = Header.ItemValueOf("WindowLength");
	    istringstream LengthStream(LengthString);
	    if (! (LengthStream >> WindowLength)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadTrend()",
		    "unable to get trend parameters (WindowLength)"
		);
	    }

	    Trend = new TKaspTrend(Name, TickInterval, WindowLength);
	    Trend->Start(StartTime);
	}

	istringstream LineStream(Line);
	long Time; double Counts, Sum, Mean, Deviation;
	if (LineStream >> Time >> Counts >> Sum >> Mean >> Deviation) {
	    Trend->Fill(StartTime + Time, Mean, Counts, Deviation);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadTrend()",
		"invalid data line: " + Line
	    );
	}
    }

    Trend->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Trend->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Trend;
}

TKaspHistory* TKaspPlainTextRepository::LoadHistory(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspHistory* History = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (History == 0) {
	    string DepthString = Header.ItemValueOf("Depth");
	    istringstream DepthStream(DepthString);
	    int MaxNumberOfSamples;
	    if (! (DepthStream >> MaxNumberOfSamples)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadHistory()",
		    "unable to get history parameters"
		);
	    }

	    History = new TKaspHistory(Name, MaxNumberOfSamples);
	}

	istringstream LineStream(Line);
	long Time, Counts; double Sum, Mean, Deviation;
	if (LineStream >> Time >> Counts >> Sum >> Mean >> Deviation) {
	    History->InsertSample(Time, Counts, Sum, Mean, Deviation);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistory()",
		"invalid data line: " + Line
	    );
	}
    }

    History->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	History->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return History;
}

TKaspWave* TKaspPlainTextRepository::LoadWave(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspWave* Wave = new TKaspWave(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istringstream LineStream(Line);
	long Index; double Value;
	if (LineStream >> Index >> Value) {
	    Wave->Fill(Index, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadWave()",
		"invalid data line: " + Line
	    );
	}
    }

    Wave->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Wave->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Wave;
}

TKaspMap* TKaspPlainTextRepository::LoadMap(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspMap* Map = new TKaspMap(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istringstream LineStream(Line);
	long Address; double Value;
	if (LineStream >> Address >> Value) {
	    Map->Fill(Address, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadMap()",
		"invalid data line: " + Line
	    );
	}
    }

    Map->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Map->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Map;
}

TKaspTabular* TKaspPlainTextRepository::LoadTabular(const string& Name, int Revision) throw(TKaspException)
{
    istream* File = OpenInputFile(Name, Revision);

    TKaspTabular* Tabular = new TKaspTabular(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istringstream LineStream(Line);
	string FieldName, Value;
	if (
	    (LineStream >> ws >> FieldName) && 
	    getline(LineStream >> ws, Value)
	){
	    Tabular->Fill(FieldName, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadTabular()",
		"invalid data line: " + Line
	    );
	}
    }

    Tabular->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Tabular->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    CloseInputFile(File);

    return Tabular;
}



TKaspPlainTextRepositoryHeader::TKaspPlainTextRepositoryHeader(void)
{
}

TKaspPlainTextRepositoryHeader::~TKaspPlainTextRepositoryHeader(void)
{
}

unsigned TKaspPlainTextRepositoryHeader::NumberOfItems()
{
    return _ItemList.size();
}

bool TKaspPlainTextRepositoryHeader::ReadLine(const string& HeaderItemLine)
{
    // syntax: ^[\s]*#+[\s]*([\w]*)[\s]*:[\s]*(.*)$, Name = \1, Value = \2

    string Name;
    string Value;
    
    enum TState { 
	State_Initial, State_Header, State_LeadingSpace,
	State_Name, 
	State_PreSeparatorSpace, State_Separator, State_ProSeparatorSpace,
	State_Value
    };

    TState State = State_Initial;

    for (unsigned i = 0; i < HeaderItemLine.size(); i++) {
	char Char = HeaderItemLine[i];

	if (State == State_Initial) {
	    if (isspace(Char)) {
		continue;
	    }
	    else if (Char == '#') {
		State = State_Header;
		continue;
	    }
	    else {
		return false;
	    }
	}
	if (State == State_Header) {
	    if (Char == '#') {
		continue;
	    }
	    else {
		State = State_LeadingSpace;
	    }
	}
	if (State == State_LeadingSpace) {
	    if (isspace(Char)) {
		continue;
	    }
	    else {
		State = State_Name;
	    }
	}
	if (State == State_Name) {
	    if (! isspace(Char) && (Char != ':')) {
		Name += Char;
		continue;	    
	    }
	    else {
		State = State_PreSeparatorSpace;
	    }
	}
	if (State == State_PreSeparatorSpace) {
	    if (isspace(Char)) {
		continue;	    
	    }
	    else {
		State = State_Separator;
	    }
	}
	if (State == State_Separator) {
	    if (Char == ':') {
		State = State_ProSeparatorSpace;
		continue;	    
	    }
	    else {
		return false;
	    }
	}
	if (State == State_ProSeparatorSpace) {
	    if (isspace(Char)) {
		continue;	    
	    }
	    else {
		State = State_Value;
	    }
	}
	if (State == State_Value) {
	    Value += Char;
	    continue;	    
	}
    }

    if (State != State_Value) {
	return false;
    }

    if (_ItemTable.count(Name) == 0) {
	_ItemList.push_back(make_pair(Name, Value));
	_ItemTable[Name] = Value;
    }
    else {
	for (unsigned i = 0; i < _ItemList.size(); i++) {
	    if (_ItemList[i].first == Name) {
		_ItemList[i].second += " " + Value;
		break;
	    }
	}
	_ItemTable[Name] += " " + Value;
    }

    return true;
}

string TKaspPlainTextRepositoryHeader::ItemNameOf(unsigned ItemIndex)
{
    return _ItemList[ItemIndex].first;
}

string TKaspPlainTextRepositoryHeader::ItemValueOf(unsigned ItemIndex)
{
    return _ItemList[ItemIndex].second;
}

string TKaspPlainTextRepositoryHeader::ItemValueOf(const string& ItemName)
{
    if (_ItemTable.count(ItemName) == 0) {
	return string();
    }
    else {
	return _ItemTable[ItemName];
    }
}

vector<string> TKaspPlainTextRepositoryHeader::ItemValueListOf(unsigned ItemIndex)
{
    return SplitValue(_ItemList[ItemIndex].second);
}

vector<string> TKaspPlainTextRepositoryHeader::ItemValueListOf(const string& ItemName)
{
    if (_ItemTable.count(ItemName) == 0) {
	return vector<string>();
    }
    else {
	return SplitValue(_ItemTable[ItemName]);
    }
}

vector<string> TKaspPlainTextRepositoryHeader::SplitValue(const string& Value, const string& SeparatorSet)
{
    vector<string> Result;
    string CurrentText;

    for (unsigned i = 0; i < Value.size(); i++) {
	// if '\' is followd by a separator character, 
	// the character is treated as a normal text character.
	if ((Value[i] == '\\') && (i+1 < Value.size())) {
	    if (SeparatorSet.find_first_of(Value[i+1]) != string::npos) {
		CurrentText += Value[i+1];
		i++;
		continue;
	    }
	}

	if (SeparatorSet.find_first_of(Value[i]) == string::npos) {
	    CurrentText += Value[i];
	}
	else {
	    if (! CurrentText.empty()) {
		Result.push_back(CurrentText);
		CurrentText.erase(CurrentText.begin(), CurrentText.end());
	    }
	}
    }

    if (! CurrentText.empty()) {
	Result.push_back(CurrentText);
	CurrentText.erase(CurrentText.begin(), CurrentText.end());
    }

    return Result;
}
