/* KaspRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 26 June 2002. */


#include <string>
#include <fstream>
#include "KaspDefs.hh"
#include "KaspNtuple.hh"
#include "KaspGraph.hh"
#include "KaspHistogram.hh"
#include "Kasp2dHistogram.hh"
#include "KaspHistory.hh"
#include "KaspWave.hh"
#include "KaspMap.hh"
#include "KaspTabular.hh"
#include "KaspRepository.hh"

using namespace std;


TKaspRepository::TKaspRepository(void)
{
}

TKaspRepository::~TKaspRepository()
{
}


TKaspRepositoryFactory* TKaspRepositoryFactory::_Instance = 0;

TKaspRepositoryFactory::TKaspRepositoryFactory(void)
{
}

TKaspRepositoryFactory::~TKaspRepositoryFactory()
{
}

TKaspRepositoryFactory* TKaspRepositoryFactory::GetInstance(void)
{
    if (_Instance == 0) {
	_Instance = new TKaspRepositoryFactory();
    }

    return _Instance;
}

TKaspRepository* TKaspRepositoryFactory::CreateRepository(const std::string& Name)
{
    if (Name == "Xml") {
	return new TKaspXmlRepository();
    }

#if IS_ROOT_AVAILABLE    
    if (Name == "Root") {
	return new TKaspRootRepository();
    }
#endif

    return 0;
}



TKaspXmlRepository::TKaspXmlRepository(void)
{
    _OutputFile = 0;
}

TKaspXmlRepository::~TKaspXmlRepository()
{
    delete _OutputFile;
}

void TKaspXmlRepository::OpenOutputFile(const string& FileName) throw(TKaspException)
{
    if (_OutputFile == 0) {
	_OutputFile = new ofstream(FileName.c_str());
	if (! *_OutputFile) {
	    delete _OutputFile;
	    _OutputFile = 0;

	    throw TKaspException(
		"TKaspXmlRepository::OpenOutputFile()",
		"unable to open file: " + FileName
	    );
	}
    }

    *_OutputFile << "<?xml version=\"1.0\"?>" << endl << endl;
    *_OutputFile << "<kinoko-view version=\"1.0\">" << endl;
}

void TKaspXmlRepository::CloseOutputFile(void) throw(TKaspException)
{
    if (_OutputFile) {
	*_OutputFile << "</kinoko-view>" << endl;
	delete _OutputFile;
	_OutputFile = 0;
    }
}

void TKaspXmlRepository::SaveNtuple(TKaspNtuple* Ntuple) throw(TKaspException)
{
    //... TODO:
}

void TKaspXmlRepository::SaveGraph(TKaspGraph* Graph) throw(TKaspException)
{
    //... TODO:
}

void TKaspXmlRepository::SaveHistogram(TKaspHistogram* Histogram) throw(TKaspException)
{
    if (_OutputFile == 0) {
	throw TKaspException(
	    "TKaspXmlRepository::SaveHistogram()", "file not opened"
	);
    }

    const TKaspHistogramScale& Scale = Histogram->Scale();

    *_OutputFile << "<histogram";
    *_OutputFile << " name=\"" << Histogram->Name() << "\"";
    *_OutputFile << " title=\"" << Histogram->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<scale";
    *_OutputFile << " title=\"" << Scale.Title() << "\"";
    *_OutputFile << " min=\"" << Scale.Min() << "\"";
    *_OutputFile << " max=\"" << Scale.Max() << "\"";
    *_OutputFile << " number-of-bins=\"" << Scale.NumberOfBins() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<out-of-range-data";
    *_OutputFile << " underflow=\"" << Histogram->UnderflowCounts() << "\"";
    *_OutputFile << " overflow=\"" << Histogram->OverflowCounts() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<statistic name=\"Entries\"";
    *_OutputFile << " value=\"" << Histogram->NumberOfEntries() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<statistic name=\"Mean\"";
    *_OutputFile << " value=\"" << Histogram->Mean() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<data record-type=\"count\" delimiter=\",\">";
    for (int BinIndex = 0; BinIndex < Scale.NumberOfBins(); BinIndex++) {
	*_OutputFile << Histogram->CountsOf(BinIndex) << ",";
    }
    *_OutputFile << "</data>" << endl;

    *_OutputFile << "</histogram>" << endl;
}

void TKaspXmlRepository::Save2dHistogram(TKasp2dHistogram* Histogram) throw(TKaspException)
{
    //... TODO:
}

void TKaspXmlRepository::SaveHistory(TKaspHistory* History) throw(TKaspException)
{
    if (_OutputFile == 0) {
	throw TKaspException(
	    "TKaspXmlRepository::SaveHistory()", "file not opened"
	);
    }

    *_OutputFile << "<history";
    *_OutputFile << " name=\"" << History->Name() << "\"";
    *_OutputFile << " title=\"" << History->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data record-type=\"time-entries-sum-mean-deviation\" delimiter=\",\">";
    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	*_OutputFile << History->PassedTimeOf(Index) << ",";
	*_OutputFile << History->EntryCountOf(Index) << ",";
	*_OutputFile << History->SumOf(Index) << ",";
	*_OutputFile << History->MeanOf(Index) << ",";
	*_OutputFile << History->DeviationOf(Index) << ",";
    }
    *_OutputFile << "</data>" << endl;

    *_OutputFile << "</histogram>" << endl;
}

void TKaspXmlRepository::SaveWave(TKaspWave* Wave) throw(TKaspException)
{
    if (_OutputFile == 0) {
	throw TKaspException(
	    "TKaspXmlRepository::SaveWave()", "file not opened"
	);
    }

    *_OutputFile << "<wave";
    *_OutputFile << " name=\"" << Wave->Name() << "\"";
    *_OutputFile << " title=\"" << Wave->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data record-type=\"index-value\" delimiter=\",\">";
    for (int Index = 0; Index < Wave->NumberOfPoints(); Index++) {
	*_OutputFile << Index << "," << Wave->ValueOf(Index) << ",";
    }
    *_OutputFile << "</data>" << endl;

    *_OutputFile << "</wave>" << endl;
}

void TKaspXmlRepository::SaveMap(TKaspMap* Map) throw(TKaspException)
{
    //... TODO:
}

void TKaspXmlRepository::SaveTabular(TKaspTabular* Tabular) throw(TKaspException)
{
    //... TODO:
}



#if IS_ROOT_AVAILABLE

#include <TFile.h>
#include <TNtuple.h>
#include <TH1.h>
#include <TGraph.h>

TKaspRootRepository::TKaspRootRepository(void)
{
    _File = 0;
}

TKaspRootRepository::~TKaspRootRepository()
{
    delete _File;
}

void TKaspRootRepository::OpenOutputFile(const string& FileName) throw(TKaspException)
{
    _FileName = FileName;
}

void TKaspRootRepository::CloseOutputFile(void) throw(TKaspException)
{
    if (_File) {
	_File->Write();
	delete _File;
	_File = 0;
    }
}

void TKaspRootRepository::SaveNtuple(TKaspNtuple* Ntuple) throw(TKaspException)
{
    if (_File == 0) {
	if (_FileName.empty()) {
	    throw TKaspException(
		"TKaspRootRepository::SaveNtuple()", "file not opened"
	    );
	}
	_File = new TFile(_FileName.c_str(), "RECREATE");
    }

    unsigned NumberOfColumns = Ntuple->NumberOfColumns();

    string ColumnNameList;
    for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	string ColumnName = Ntuple->ColumnNameOf(Column);
	if (Column != 0) {
	    ColumnNameList += ':';
	}
	ColumnNameList += ColumnName;
    }
    
    TNtuple* RootNtuple = new TNtuple(
	Ntuple->Name().c_str(), Ntuple->Title().c_str(), ColumnNameList.c_str()
    );

    float* Data = new float[NumberOfColumns];
    for (unsigned Row = 0; Row < Ntuple->NumberOfRows(); Row++) {
	for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	    Data[Column] = (*Ntuple)[Row][Column];
	}
	RootNtuple->Fill(Data);
    }
    delete[] Data;

    RootNtuple->Write();
    delete RootNtuple;
}

void TKaspRootRepository::SaveGraph(TKaspGraph* Graph) throw(TKaspException)
{
    if (_File == 0) {
	if (_FileName.empty()) {
	    throw TKaspException(
		"TKaspRootRepository::SaveGraph()", "file not opened"
	    );
	}
	_File = new TFile(_FileName.c_str(), "RECREATE");
    }

    int NumberOfPoints = Graph->NumberOfPoints();
    if (NumberOfPoints == 0) {
	// ROOT cannot make a Graph object when NumberOfPoints is 0.
	return;
    }

    TGraph* RootGraph = new TGraph(
	Graph->NumberOfPoints(), Graph->XValueList(), Graph->YValueList()
    );
    RootGraph->SetNameTitle(Graph->Name().c_str(), Graph->Title().c_str());
    RootGraph->Write();

    delete RootGraph;
}

void TKaspRootRepository::SaveHistogram(TKaspHistogram* Histogram) throw(TKaspException)
{
    if (_File == 0) {
	if (_FileName.empty()) {
	    throw TKaspException(
		"TKaspRootRepository::SaveHistogram()", "file not opened"
	    );
	}
	_File = new TFile(_FileName.c_str(), "RECREATE");
    }

    const TKaspHistogramScale& Scale = Histogram->Scale();
    int NumberOfBins = Scale.NumberOfBins();

    TH1* RootHistogram = new TH1D(
	Histogram->Name().c_str(), Histogram->Title().c_str(),
	NumberOfBins, Scale.Min(), Scale.Max()
    );

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double BinCenter = Scale.BinCenterOf(BinIndex);
	double Count = Histogram->CountsOf(BinIndex);
	RootHistogram->Fill(BinCenter, Count);
    }

    RootHistogram->SetBinContent(0, Histogram->UnderflowCounts());
    RootHistogram->SetBinContent(NumberOfBins + 1, Histogram->OverflowCounts());
    RootHistogram->SetEntries(Histogram->NumberOfEntries());

    RootHistogram->Write();

    delete RootHistogram;
}

void TKaspRootRepository::Save2dHistogram(TKasp2dHistogram* Histogram) throw(TKaspException)
{
    //... TODO:
}

void TKaspRootRepository::SaveHistory(TKaspHistory* History) throw(TKaspException)
{
    if (_File == 0) {
	if (_FileName.empty()) {
	    throw TKaspException(
		"TKaspRootRepository::SaveHistory()", "file not opened"
	    );
	}
	_File = new TFile(_FileName.c_str(), "RECREATE");
    }

    TNtuple* RootNtuple = new TNtuple(
	History->Name().c_str(), History->Title().c_str(),
	"time:entries:sum:mean:deviation"
    );

    float Data[5];
    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	Data[0] = History->PassedTimeOf(Index);
	Data[1] = History->EntryCountOf(Index);
	Data[2] = History->SumOf(Index);
	Data[3] = History->MeanOf(Index);
	Data[4] = History->DeviationOf(Index);

	RootNtuple->Fill(Data);
    }

    RootNtuple->Write();
    delete RootNtuple;
}

void TKaspRootRepository::SaveWave(TKaspWave* Wave) throw(TKaspException)
{
    if (_File == 0) {
	if (_FileName.empty()) {
	    throw TKaspException(
		"TKaspRootRepository::SaveWave()", "file not opened"
	    );
	}
	_File = new TFile(_FileName.c_str(), "RECREATE");
    }

    int NumberOfPoints = Wave->NumberOfPoints();
    if (NumberOfPoints == 0) {
	// ROOT cannot make a Graph object when NumberOfPoints is 0.
	return;
    }

    double* XDataList = new double[NumberOfPoints];
    double* YDataList = new double[NumberOfPoints];

    for (int Index = 0; Index < NumberOfPoints; Index++) {
	XDataList[Index] = Index;
	YDataList[Index] = Wave->ValueOf(Index);
    }

    TGraph* RootGraph = new TGraph(NumberOfPoints, XDataList, YDataList);
    RootGraph->SetNameTitle(Wave->Name().c_str(), Wave->Title().c_str());
    RootGraph->Write();

    delete RootGraph;
    delete[] XDataList;
    delete[] YDataList;
}

void TKaspRootRepository::SaveMap(TKaspMap* Map) throw(TKaspException)
{
    //... TODO:
}

void TKaspRootRepository::SaveTabular(TKaspTabular* Tabular) throw(TKaspException)
{
    //... TODO:
}

#endif
