/* KaspXmlRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 26 June 2002. */


#include <string>
#include <fstream>
#include "KaspDefs.hh"
#include "KaspNtuple.hh"
#include "KaspGraph.hh"
#include "KaspHistogram.hh"
#include "Kasp2dHistogram.hh"
#include "KaspTrend.hh"
#include "KaspHistory.hh"
#include "KaspWave.hh"
#include "KaspMap.hh"
#include "KaspTabular.hh"
#include "KaspRepository.hh"
#include "KaspXmlRepository.hh"

using namespace std;


TKaspXmlRepository::TKaspXmlRepository(const string& FileName)
{
    _FileName = FileName;

    _OutputFile = 0;
    _InputFile = 0;
}

TKaspXmlRepository::~TKaspXmlRepository()
{
    Close();

    delete _InputFile;
    delete _OutputFile;
}

void TKaspXmlRepository::OpenToWrite(void) throw(TKaspException)
{
    if (_OutputFile) {
	throw TKaspException(
	    "TKaspXmlRepository::OpenToWrite()", 
	    "file already opened"
	);
    }
    if (_InputFile) {
	throw TKaspException(
	    "TKaspXmlRepository::OpenToWrite()", 
	    "file already opened in read-only mode"
	);
    }

    _OutputFile = new ofstream(_FileName.c_str());
    if (! *_OutputFile) {
	delete _OutputFile;
	_OutputFile = 0;
	
	throw TKaspException(
	    "TKaspXmlRepository::OpenToWrite()",
	    "unable to open file: " + _FileName
	);
    }

    *_OutputFile << "<?xml version=\"1.0\"?>" << endl;
    *_OutputFile << endl;
    *_OutputFile << "<kasp version=\"1.0\">" << endl;
}

void TKaspXmlRepository::OpenToRead(void) throw(TKaspException)
{
    throw TKaspException(
	"TKaspXmlRepository::OpenToRead()", "function not implemented yet"
    );
}

void TKaspXmlRepository::Close(void)
{
    if (_InputFile) {
	delete _InputFile;
	_InputFile = 0;
    }

    if (_OutputFile) {
	*_OutputFile << "</kasp>" << endl;
	delete _OutputFile;
	_OutputFile = 0;
    }
}

void TKaspXmlRepository::SaveNtuple(const std::string& Name, TKaspNtuple* Ntuple) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<ntuple";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Ntuple->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"";
    unsigned NumberOfColumns = Ntuple->NumberOfColumns();
    for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	if (Column != 0) {
	    *_OutputFile << " ";
	}
	*_OutputFile << Ntuple->ColumnNameOf(Column);
    }
    *_OutputFile << "\" delimiter=\",\">";
    *_OutputFile << endl;

    for (unsigned Row = 0; Row < Ntuple->NumberOfRows(); Row++) {
	for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	    *_OutputFile << (*Ntuple)[Row][Column] << ",";
	}
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</graph>" << endl;
}

void TKaspXmlRepository::SaveGraph(const std::string& Name, TKaspGraph* Graph) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<graph";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Graph->Title() << "\"";
    *_OutputFile << ">" << endl;

    if (Graph->HasErrors()) {
	*_OutputFile << "<data fields=\"X Y XError YError\" delimiter=\",\">";
    }
    else {
	*_OutputFile << "<data fields=\"X Y\" delimiter=\",\">";
    }
    *_OutputFile << endl;

    const double* XValueList = Graph->XValueList();
    const double* YValueList = Graph->YValueList();
    const double* XErrorList = Graph->XErrorList();
    const double* YErrorList = Graph->YErrorList();
    unsigned NumberOfPoints = Graph->NumberOfPoints();

    for (unsigned i = 0; i < NumberOfPoints; i++) {
	*_OutputFile << XValueList[i] << "," << YValueList[i] << ",";
	if (Graph->HasErrors()) {
	    *_OutputFile << XErrorList[i] << "," << YErrorList[i] << ",";
	}
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</graph>" << endl;
}

void TKaspXmlRepository::SaveHistogram(const std::string& Name, TKaspHistogram* Histogram) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKaspHistogramScale& Scale = Histogram->Scale();

    *_OutputFile << "<histogram";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Histogram->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<scale";
    *_OutputFile << " title=\"" << Scale.Title() << "\"";
    *_OutputFile << " number-of-bins=\"" << Scale.NumberOfBins() << "\"";
    *_OutputFile << " min=\"" << Scale.Min() << "\"";
    *_OutputFile << " max=\"" << Scale.Max() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<out-of-range-data";
    *_OutputFile << " underflow=\"" << Histogram->UnderflowCounts() << "\"";
    *_OutputFile << " overflow=\"" << Histogram->OverflowCounts() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<statistic name=\"entries\"";
    *_OutputFile << " value=\"" << Histogram->NumberOfEntries() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<statistic name=\"mean\"";
    *_OutputFile << " value=\"" << Histogram->Mean() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<data fields=\"Counts\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int BinIndex = 0; BinIndex < Scale.NumberOfBins(); BinIndex++) {
	*_OutputFile << Histogram->CountsOf(BinIndex) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</histogram>" << endl;
}

void TKaspXmlRepository::Save2dHistogram(const std::string& Name, TKasp2dHistogram* Histogram) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKaspHistogramScale& XScale = Histogram->XScale();
    const TKaspHistogramScale& YScale = Histogram->YScale();

    *_OutputFile << "<histogram2d";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Histogram->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<scale";
    *_OutputFile << " axis=\"x\"";
    *_OutputFile << " title=\"" << XScale.Title() << "\"";
    *_OutputFile << " number-of-bins=\"" << XScale.NumberOfBins() << "\"";
    *_OutputFile << " min=\"" << XScale.Min() << "\"";
    *_OutputFile << " max=\"" << XScale.Max() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<scale";
    *_OutputFile << " axis=\"y\"";
    *_OutputFile << " title=\"" << YScale.Title() << "\"";
    *_OutputFile << " number-of-bins=\"" << YScale.NumberOfBins() << "\"";
    *_OutputFile << " min=\"" << YScale.Min() << "\"";
    *_OutputFile << " max=\"" << YScale.Max() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<statistic name=\"entries\"";
    *_OutputFile << " value=\"" << Histogram->NumberOfEntries() << "\"";
    *_OutputFile << "/>" << endl;

    *_OutputFile << "<data fields=\"Counts\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int YBinIndex = 0; YBinIndex < YScale.NumberOfBins(); YBinIndex++) {
	for (int XBinIndex = 0; XBinIndex < XScale.NumberOfBins(); XBinIndex++) {
	    *_OutputFile << Histogram->CountsOf(XBinIndex, YBinIndex) << ",";
	    *_OutputFile << endl;
	}
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</histogram2d>" << endl;
}

void TKaspXmlRepository::SaveTrend(const std::string& Name, TKaspTrend* Trend) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<trend";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Trend->Title() << "\"";
    *_OutputFile << " start_time=\"" << Trend->StartTime() << "\"";
    *_OutputFile << " tick_interval=\"" << Trend->TickInterval() << "\"";
    *_OutputFile << " window_length=\"" << Trend->WindowLength() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"Time Counts Sum Mean Deviation\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int Index = 0; Index < Trend->NumberOfPoints(); Index++) {
	*_OutputFile << Trend->PassedTimeOf(Index) << ",";
	*_OutputFile << Trend->CountsOf(Index) << ",";
	*_OutputFile << Trend->SumOf(Index) << ",";
	*_OutputFile << Trend->MeanOf(Index) << ",";
	*_OutputFile << Trend->DeviationOf(Index) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</trend>" << endl;
}

void TKaspXmlRepository::SaveHistory(const std::string& Name, TKaspHistory* History) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<history";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << History->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"Time Counts Sum Mean Deviation\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	*_OutputFile << History->PassedTimeOf(Index) << ",";
	*_OutputFile << History->CountsOf(Index) << ",";
	*_OutputFile << History->SumOf(Index) << ",";
	*_OutputFile << History->MeanOf(Index) << ",";
	*_OutputFile << History->DeviationOf(Index) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</history>" << endl;
}

void TKaspXmlRepository::SaveWave(const std::string& Name, TKaspWave* Wave) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<wave";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Wave->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"Index Value\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int Index = 0; Index < Wave->NumberOfPoints(); Index++) {
	*_OutputFile << Index << ",";
	*_OutputFile << Wave->ValueOf(Index) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</wave>" << endl;
}

void TKaspXmlRepository::SaveMap(const std::string& Name, TKaspMap* Map) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<map";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Map->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"Address Value\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int Index = 0; Index < Map->NumberOfPoints(); Index++) {
	*_OutputFile << Map->AddressOf(Index) << ",";
	*_OutputFile << Map->ValueOf(Index) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</map>" << endl;
}

void TKaspXmlRepository::SaveTabular(const std::string& Name, TKaspTabular* Tabular) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    *_OutputFile << "<tabular";
    *_OutputFile << " name=\"" << Name << "\"";
    *_OutputFile << " title=\"" << Tabular->Title() << "\"";
    *_OutputFile << ">" << endl;

    *_OutputFile << "<data fields=\"FieldName Value\" delimiter=\",\">";
    *_OutputFile << endl;

    for (int Index = 0; Index < Tabular->NumberOfFields(); Index++) {
	*_OutputFile << Tabular->FieldNameOf(Index) << ",";
	*_OutputFile << Tabular->ValueOf(Index) << ",";
	*_OutputFile << endl;
    }

    *_OutputFile << "</data>" << endl;
    *_OutputFile << "</tabular>" << endl;
}

TKaspNtuple* TKaspXmlRepository::LoadNtuple(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadNtuple()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspGraph* TKaspXmlRepository::LoadGraph(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadGraph()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspHistogram* TKaspXmlRepository::LoadHistogram(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadHistogram()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKasp2dHistogram* TKaspXmlRepository::Load2dHistogram(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::Load2dHistogram()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspHistory* TKaspXmlRepository::LoadHistory(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadHistory()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspTrend* TKaspXmlRepository::LoadTrend(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadTrend()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspWave* TKaspXmlRepository::LoadWave(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadWave()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspMap* TKaspXmlRepository::LoadMap(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::LoadMap()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspTabular* TKaspXmlRepository::LoadTabular(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspXmlRepository::Tabular()",
	"sorry, function not implemented yet"
    );

    return 0;
}
