/* KcomManager.cc */
/* Created by Enomoto Sanshiro on 28 March 2000. */
/* Updated by Enomoto Sanshiro on 1 August 2001. */
/* Last updated by Enomoto Sanshiro on 19 January 2010. */


#include <iostream>
#include <string>
#include "MushTimer.hh"
#include "MushProcess.hh"
#include "MushNetworkSocket.hh"
#include "MushMisc.hh"
#include "KorbOrb.hh"
#include "KorbBroker.hh"
#include "KorbObjectAdapter.hh"
#include "KorbObjectMessenger.hh"
#include "KorbObjectReference.hh"
#include "KorbNamingContext.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryOrb.hh"
#include "KcomComponentAssembler.hh"
#include "KcomObjectAssembler.hh"
#include "KcomEventProcessor.hh"
#include "KcomManager.hh"

using namespace std;


// If message queue capacity is smaller than this value,
// asynchronous transaction is disabled to avoid possible deadlock.
// See comments in TKcomEventProcessor::EmitEvent().
static const int MinimumAsynchronousQueueCapacity = 10240;


TKcomManager::TKcomManager(void)
{
    _Orb = new TKorbOrb();
    _KorbCommander = new TKorbCommander(_Orb);
    _ObjectAdapter = _Orb->ObjectAdapter();

    _ExchangeCenter = 0;
    _ExchangeCenterMessenger = 0;
    _RegistryProcess = 0;

    _Registry = new TKcomRegistryProxy(_Orb);
}

TKcomManager::~TKcomManager()
{
    delete _Registry;
    delete _ExchangeCenterMessenger;
    delete _ExchangeCenter;

    delete _KorbCommander;
    delete _Orb;
}

void TKcomManager::Initialize(void) throw(TKcomException)
{
    try {
	_KcomPath = TMushEnvironmentVariable("KCOM_PATH").AsString();
	cout << "KCOM_PATH: " << _KcomPath << endl;
    }
    catch (TSystemCallException &e) {
	cout << "KCOM_PATH is not set." << endl;
    }
    
    try {
        _KorbCommander->Initialize(_KcomPath);
	LaunchRegistryServer();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::Initialize()",
	    "korb exception: " + e.Message()
	);
    }

    try {
	Construct(_KorbCommander, _Orb);
    }
    catch (TKcomException &e) {
	TerminateRegistryServer();
	_KorbCommander->Finalize();
	throw e;
    }

    TKcomComponentAssembler* ComponentAssembler = GetComponentAssembler();
    TKcomObjectAssembler* ObjectAssembler = GetObjectAssembler();
    TKcomEventProcessor* EventProcessor = GetEventProcessor();

    _ExchangeCenterMessenger = new TKcomExchangeCenterMessenger(
        _ExchangeCenter = new TKcomExchangeCenter(
	    ObjectAssembler, EventProcessor
	)
    );

    if (_Orb->QueueCapacity() < MinimumAsynchronousQueueCapacity) {
	cout << "Warning: KCOM: ";
	cout << "max message queue size (kernel parameter) too small ";
	cout << "(" << _Orb->QueueCapacity() << " byte): ";
	cout << "asynchronous mode disabled." << endl;
	EventProcessor->DisableAsynchronousMode();
    }

    try {
	_ObjectAdapter->AddMessenger(
	    _ExchangeCenterMessenger, "KcomExchangeCenter"
	);
	ComponentAssembler->DeployComponents();
    }
    catch (TKorbException &e) {
	TerminateRegistryServer();
	try {
	    _KorbCommander->Finalize();
	}
	catch (TKorbException &ee) {
	    ;
	}
	throw TKcomException(
	    "TKcomManager::Initialize()",
	    "korb exception: " + e.Message()
	);
    }

    int NumberOfComponents = ComponentAssembler->NumberOfComponents();
    int NumberOfTries = 0;
    while (_ExchangeCenter->NumberOfAttachedComponents() < NumberOfComponents) {
	if (++NumberOfTries % 1000 == 0) {
	    cerr << NumberOfComponents - _ExchangeCenter->NumberOfAttachedComponents();
	    cerr << " components are not attached. still waiting..." << endl;
	}

	try {
	    _ObjectAdapter->DoNowaitTransaction();
	    DoTransaction();
	}
	catch (TKorbException &e) {
	    throw TKcomException(
		"TKcomManager::Initialize()",
		"korb exception: " + e.Message()
	    );
	}
    }

    for (int i = 0; i < NumberOfComponents; i++) {
	EventProcessor->RegisterComponent((*ComponentAssembler)[i].Name());
    }

    OnStartup();
}

int TKcomManager::Finalize(void) throw(TKcomException)
{
    int ReturnValue = 0;
    try {
	static const int TimeOut = 5;
	long StartTime = TMushDateTime::SecSinceEpoch();
	while (_ExchangeCenter->NumberOfAttachedComponents() > 0) {
	    if (TMushDateTime::SecSince(StartTime) > TimeOut) {
		cerr << _ExchangeCenter->NumberOfAttachedComponents();
		if (_ExchangeCenter->NumberOfAttachedComponents() == 1) {
		    cerr << " component is ";
		}
		else {
		    cerr << " components are ";
		}
		cerr << "still running. terminating..." << endl;
		ReturnValue = -1;
		break;
	    }

	    _ObjectAdapter->DoNowaitTransaction();
	    DoTransaction();
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::Finalize()", "korb exception: " + e.Message()
	);
    }

    TerminateRegistryServer();

    try {
	map<unsigned long, int>::iterator DomainIdEntry;
	for (
	    DomainIdEntry = _DomainIdTable.begin(); 
	    DomainIdEntry != _DomainIdTable.end(); 
	    DomainIdEntry++
	){
	    int DomainId = DomainIdEntry->second;
	    if (DomainId != TKorbBroker::PrimaryDomainId()) {
		_KorbCommander->TerminateBroker(DomainId);
	    }
	}
    
	_KorbCommander->Finalize();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::Finalize()", "korb exception: " + e.Message()
        );
    }

    try {
	OnShutdown();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "KCOM: on shutdown()", "korb exception: " + e.Message()
        );
    }

    return ReturnValue;
}

int TKcomManager::Start(void) throw(TKcomException)
{
    Initialize();

    int ReturnValue = 0;
    try {
	TMushSignalHandler SignalHandler;
	TMushSignalCounter SignalCounter;
	SignalHandler.RegisterClient(SIGINT, &SignalCounter);
	SignalHandler.RegisterClient(SIGTERM, &SignalCounter);
	SignalHandler.StartHandling();

	_IsStopRequested = false;

	while (! _IsStopRequested) {
	    if (SignalCounter.SignalCount() > 0) {
		cerr << "KcomManager: signaled: terminating..." << endl;
		ReturnValue = -1;
		break;
	    }

	    _ObjectAdapter->DoNowaitTransaction();
	    DoTransaction();
	}

	SignalHandler.StartDefaultAction(SIGINT);
	SignalHandler.StartDefaultAction(SIGTERM);
    }
    catch (TKorbException &e) {
	Finalize();
	throw TKcomException(
	    "TKcomManager::Start()", "korb exception: " + e.Message()
	);
    }   

    int FinalizeStatus = Finalize();

    return (ReturnValue < 0) ? ReturnValue : FinalizeStatus;
}

int TKcomManager::Stop(void) throw(TKcomException)
{
    _IsStopRequested = true;

    return 0;
}

int TKcomManager::DoTransaction(void) throw(TKcomException)
{
    TMushRealTimeTimer(0, 10000).Suspend();
    return 0;
}

int TKcomManager::LaunchRemoteBroker(const std::string& HostName, const std::string& RemoteWorkingDirectory) throw(TKcomException)
{
    int DomainId = DomainIdOf(HostName);
    if (DomainId > 0) {
	return DomainId;
    }

    //... BUG: KcomPath on remote domain can be different ...//
    int ParentDomainId = TKorbBroker::PrimaryDomainId();
    try {
	DomainId = _KorbCommander->LaunchRemoteBroker(
	    ParentDomainId, HostName, _KcomPath, RemoteWorkingDirectory
	);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomLaunchar::LaunchRemoteBroker()",
	    "KORB exception: " + e.Message()
	);
    }

    unsigned long HostAddress;
    try {
	HostAddress = TMushNetworkSocket::HostAddressByName(HostName);
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomLaunchar::LaunchRemoteBroker()",
	    "system call exception: " + e.Message()
	);
    }
    _DomainIdTable[HostAddress] = DomainId;

    return DomainId;
}

int TKcomManager::DomainIdOf(const string& HostName) throw(TKcomException)
{
    if (HostName.empty() || HostName == "localhost") {
	return TKorbBroker::PrimaryDomainId();
    }

    if (_DomainIdTable.empty()) {
	string LocalHostName = TMushLocalHost::HostName();
	unsigned long LocalHostAddress;
	try {
	    LocalHostAddress = TMushNetworkSocket::HostAddressByName(
		LocalHostName
	    );
	}
	catch (TSystemCallException &e) {
	    throw TKcomException(
		"TKcomLaunchar::DomainIdOf()",
		"system call exception: " + e.Message()
	    );
	}

	_DomainIdTable[LocalHostAddress] = TKorbBroker::PrimaryDomainId();
    }

    unsigned long HostAddress;
    try {
	HostAddress = TMushNetworkSocket::HostAddressByName(HostName);
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomLaunchar::DomainIdOf()",
	    "system call exception: " + e.Message()
	);
    }

    if (_DomainIdTable.count(HostAddress) == 0) {
	return -1;
    }

    return _DomainIdTable[HostAddress];
}

void TKcomManager::LaunchRegistryServer(void) throw(TKcomException)
{
    string Path = "kcom-registry-server";
    vector<string> ArgumentList;
    ArgumentList.push_back(Path);

    try {
	_RegistryProcess = new TMushChildProcess(Path, ArgumentList);
	_RegistryProcess->Run();
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomManager::LaunchRegistryServer()",
	    "system call exception: " + e.Message()
	);
    }

    string Name = "KcomRegistry";
    TKorbNamingContext* NamingContext = _Orb->NamingContext();    
    TMushRealTimeTimer Timer(0, 100000);
    do {
	Timer.Suspend();
    } while (! NamingContext->LookupObject(Name, _RegistryReference));

    _Registry->Bind(_RegistryReference);

    _Registry->SetLongValue("__meta/time", TMushDateTime().SecSinceEpoch());
    _Registry->SetValue("__meta/calendar_time", TMushDateTime().AsString());
}

void TKcomManager::TerminateRegistryServer(void) throw(TKcomException)
{
    _Orb->TerminateAdapter(_RegistryReference);
    _RegistryProcess->WaitToExit();

    delete _RegistryProcess;
    _RegistryProcess = 0;
}

TKcomRegistry* TKcomManager::GetRegistry(void)
{
    return _Registry;
}
