/* KiscInterpreter.cc */
/* Created by Enomoto Sanshiro on 14 December 2001. */
/* Last updated by Enomoto Sanshiro on 28 May 2008. */


#include <iostream>
#include <fstream>
#include <vector>
#include <cstdlib>
#include <fcntl.h>
#include <MushConsoleStream.hh>
#include <MushArgumentList.hh>
#include <MushMisc.hh>
#include <MushSignal.hh>
#include "KiscScript.hh"
#include "KiscInterpreter.hh"

using namespace std;


TKiscInterpreter::TKiscInterpreter(int argc, char** argv) throw(TScriptException)
{
    _ArgumentList = new TMushArgumentList(argc, argv);

    _Console = new TMushReadlineConsole();
    _Console->SetPrompt("> ");

    _Script = new TKiscScript(argc - 1, argv + 1, _Console);

    _TokenTable = _Script->GetTokenTable();
    _SymbolTable = _Script->GetSymbolTable();
    _Package = _Script->GetPackage();
    _StatementParser = _Script->GetStatementParser();

    string ScriptFileName;
    if (_ArgumentList->NumberOfParameters() > 0) {
	ScriptFileName = (*_ArgumentList)[0];
	if (ScriptFileName == "--") {
	    ScriptFileName = "";
	}
	else {
	    _ScriptFileNameList.push_back(ScriptFileName);
	}
    }

    _IsResultPrintEnabled = ! _ArgumentList->IsOptionSpecified("--noprint");
    _IsInteractive = _ArgumentList->IsOptionSpecified("--interactive", 'i');

    if (ScriptFileName.empty()) {
	_IsInteractive = true;
    }
    else {
	_IsResultPrintEnabled = false;
    }
}

TKiscInterpreter::~TKiscInterpreter()
{
    if (_IsInteractive) {
	cout << endl;
    }

    for (unsigned i = 0; i < _InputStreamList.size(); i++) {
	delete _InputStreamList[i];
    }
    for (unsigned j = 0; j < _EntryList.size(); j++) {
	delete _EntryList[j];
    }

    delete _Script;
    delete _InputStream;
    delete _Tokenizer;

    delete _Console;
    delete _ArgumentList;
}

void TKiscInterpreter::AddStartupFile(const std::string& FileName)
{
    _ScriptFileNameList.push_back(FileName);
}

void TKiscInterpreter::SetHistoryFile(const std::string& FileName)
{
    _HistoryFileName = FileName;
}

void TKiscInterpreter::Initialize(void) throw(TScriptException)
{
    if (_IsInteractive) {
	_InputStream = new TMushInputConsoleStream(_Console);
	_Tokenizer = new TParaTokenizer(*_InputStream, _TokenTable);
    }
    else {
	_InputStream = new ifstream("/dev/null");
	_Tokenizer = new TParaTokenizer(*_InputStream, _TokenTable);
    }

    for (unsigned i = 0; i < _ScriptFileNameList.size(); i++) {
	string FileName = _ScriptFileNameList[i];
	ifstream *InputStream = new ifstream(FileName.c_str());
	if (! *InputStream) {
	    throw TScriptException("unable to open file: " + FileName);
	}
	_InputStreamList.push_back(InputStream);
	_Tokenizer->InputBuffer()->SetChildInput(*InputStream);
    }

    TParaSymbolTable& SpecialSymbolTable = (
	TParaOperatorVariableAccess::LocalSymbolTable()
    );

    _LastResultVariable = SpecialSymbolTable.GetVariable(
	SpecialSymbolTable.RegisterVariable("$", TParaValue(TParaVariant()))
    );

    _ArgumentListVariable = SpecialSymbolTable.GetVariable(
	SpecialSymbolTable.RegisterVariable(
	    "$ARG", TParaValue(TParaListValue())
	)
    );

    _EnvironmentVariableTableVariable = SpecialSymbolTable.GetVariable(
	SpecialSymbolTable.RegisterVariable(
	    "$ENV", TParaValue(TParaListValue())
	)
    );

    const deque<string>& ParameterList = _ArgumentList->ParameterList();
    for (unsigned i = 0; i < ParameterList.size(); i++) {
	_ArgumentListVariable->AsValueList().push_back(
	    TParaValue(ParameterList[i])
	);
    }
    const map<string, string>& OptionTable = _ArgumentList->OptionTable();
    for (
	map<string, string>::const_iterator i = OptionTable.begin();
	i != OptionTable.end();
	i++
    ){
	int Index;
	if (i->second.empty()) {
	    Index = _ArgumentListVariable->AsList().AppendValue(
		TParaValue(true)
	    );
	}
	else {
	    Index = _ArgumentListVariable->AsList().AppendValue(
		TParaValue(i->second)
	    );
	}
	_ArgumentListVariable->AsList().SetKey(Index, i->first);
    }
    const set<char>& SwitchSet = _ArgumentList->SwitchSet();
    for (
	set<char>::const_iterator i = SwitchSet.begin();
	i != SwitchSet.end();
	i++
    ){
	_ArgumentListVariable->AsList().SetKey(
	    _ArgumentListVariable->AsList().AppendValue(TParaValue(true)), 
	    string("-") + *i
	);
    }

    map<string, string> EnvTable;
    TMushEnvironmentVariable::LoadTable(EnvTable);
    for (
	map<string, string>::const_iterator i = EnvTable.begin();
	i != EnvTable.end();
	i++
    ){
	int Index;
	if (i->second.empty()) {
	    Index = _EnvironmentVariableTableVariable->AsList().AppendValue(
		TParaValue(true)
	    );
	}
	else {
	    Index = _EnvironmentVariableTableVariable->AsList().AppendValue(
		TParaValue(i->second)
	    );
	}
	_EnvironmentVariableTableVariable->AsList().SetKey(Index, i->first);
    }
}

void TKiscInterpreter::Start(void) throw(TScriptException)
{
    Initialize();

    if (! _HistoryFileName.empty()) {
	_Console->LoadHistoryFrom(_HistoryFileName);
    }

    _PrintBase = Base_dec;
    _IsStopRequested = false;

    TMushSignalHandler SignalHandler;
    SignalHandler.RegisterClient(SIGINT, this);
    SignalHandler.RegisterClient(SIGTERM, this);
    SignalHandler.StartHandling();

    while ((! _IsStopRequested) && (! _Tokenizer->LookAhead().IsEmpty())) {
	try {
	    if (_Tokenizer->LookAhead().Is(".")) {
		ProcessCommand();
	    }
	    else {
		bool IsThisResultPrintEnabled = _IsResultPrintEnabled;
		if (_Tokenizer->LookAhead().Is("?")) {
		    _Tokenizer->Next();
		    IsThisResultPrintEnabled = true;
		}
		ProcessScript();

		if (IsThisResultPrintEnabled) {
		    PrintResult();
		}
	    }
	}
	catch (TScriptException &e) {
	    cerr << "ERROR: " << e << endl;
	}
    }

    if (! _HistoryFileName.empty()) {
	_Console->SaveHistoryTo(_HistoryFileName);
    }
}

void TKiscInterpreter::OnCatchSignal(int SignalId)
{
    _IsStopRequested = true;
    close(STDIN_FILENO);
}

void TKiscInterpreter::ProcessCommand(void) throw(TScriptException)
{
    _Tokenizer->Next().MustBe(".");
    TParaToken Token = _Tokenizer->Next();

    if (Token.Is("execute") || Token.Is("x")) {
	string FileName = _Tokenizer->SkipWhiteSpace().GetLine().AsString();
	ifstream* InputStream = new ifstream(FileName.c_str());
	_InputStreamList.push_back(InputStream);
	if (! *InputStream) {
	    throw TScriptException("unable to open file: " + FileName);
	}
	else {
	    _Tokenizer->InputBuffer()->SetChildInput(*InputStream);
	}
    }
    else if (Token.Is("quit") || Token.Is("q")) {
	_IsStopRequested = true;
    }
    else if (Token.Is("print")) {
	_IsResultPrintEnabled = true;
    }
    else if (Token.Is("noprint")) {
	_IsResultPrintEnabled = false;
    }
    else if (Token.Is("dec")) {
	_PrintBase = Base_dec;
	PrintResult();
    }
    else if (Token.Is("hex")) {
	_PrintBase = Base_hex;
	PrintResult();
    }
    else {
	throw TScriptException("unknown command: " + Token.AsString());
    }
}

void TKiscInterpreter::ProcessScript(void) throw(TScriptException)
{
    _Console->SetPrompt("? ");

    TParaPackageEntry* Entry = _Package->CreateEntry(_Tokenizer);
    if (Entry != 0) {
	Entry->Parse(_Tokenizer, _StatementParser, _SymbolTable);
	_EntryList.push_back(Entry);
    }
    else {
	TParaStatement* Statement = 0;
	try {
	    TParaStatement::TExecResult ExecResult;

	    Statement = _StatementParser->Parse(_Tokenizer, _SymbolTable);
	    ExecResult = Statement->Execute(_SymbolTable);

	    if (ExecResult.ExecStatus != TParaStatement::esNormal) {
		_IsStopRequested = true;
	    }
	    _LastResultVariable->Assign(ExecResult.ReturnValue);
	}
	catch (TScriptException &e) {
	    delete Statement;	
	    _Tokenizer->InputBuffer()->AbortChildInput();
	    _Console->SetPrompt("> ");
	    throw;
	}
	delete Statement;	
    }

    _Console->SetPrompt("> ");
}

void TKiscInterpreter::PrintResult(void) throw(TScriptException)
{
    if (_LastResultVariable->IsVoid()) {
	;
    }
    else if (_LastResultVariable->IsLong() && (_PrintBase == Base_hex)) {
	cout << hex << "0x" << _LastResultVariable->AsLong() << dec << endl;
    }
    else if (_LastResultVariable->IsString()) {
	cout << '\"' << _LastResultVariable->AsString() << '\"' << endl;
    }
    else {
	cout << _LastResultVariable->AsString() << endl;
    }
}
