/* KiscNtuple.cc */
/* Created by Enomoto Sanshiro on 10 July 2009. */
/* Last updated by Enomoto Sanshiro on 10 July 2009. */


#include <string>
#include <iostream>
#include "MushFileSystem.hh"
#include "ParaParser.hh"
#include "KameNtuple.hh"
#include "KamePlainTextRepository.hh"
#include "KameRootRepository.hh"
#include "KameRepositoryFactory.hh"
#include "KiscNtuple.hh"
#include "KiscScript.hh"


using namespace std;
using namespace kame;



TKameUnknownParaValue::TKameUnknownParaValue(const TParaValue& Value)
: _ParaValue(Value)
{
    _ParaValue.Refer();
}

TKameUnknownParaValue::TKameUnknownParaValue(const TKameUnknownParaValue& Value)
: _ParaValue(Value._ParaValue)
{
    _ParaValue.Refer();
}

TKameUnknownParaValue::~TKameUnknownParaValue(void)
{
    _ParaValue.Unrefer();
}

TKameUnknown* TKameUnknownParaValue::Clone(void) const
{
    return new TKameUnknownParaValue(*this);
}

TParaValue& TKameUnknownParaValue::ParaValue(void)
{
    return _ParaValue;
}



TKiscKameObject::TKiscKameObject(void)
{
    _Parser = 0;
    _HeaderProcessor = new TKiscRepositoryHeaderProcessor();
}

TKiscKameObject::~TKiscKameObject()
{
    delete _HeaderProcessor;

    if (_Parser) {
	for (unsigned i = 0; i < _EntryList.size(); i++) {
	    delete _EntryList[i];
	}
	delete _Parser;
    }
}

void TKiscKameObject::SetDelimiter(const std::string& Delimiter)
{
    _Delimiter = Delimiter;
}

void TKiscKameObject::SetQuote(const std::string& Quote)
{
    _Quote = Quote;
}

void TKiscKameObject::Save(string RepositoryName, string ObjectName) throw(TScriptException)
{
    TKameRepository* Repository;

    if (TMushFileAttribute(RepositoryName).Extension() == "root") {
	Repository = (
	    TKameRootRepositoryFactory().CreateRepository(RepositoryName)
	);
	if (Repository == 0) {
	    throw TScriptException("ROOT is not available");
	}
	if (ObjectName.empty()) {
	    ObjectName = "noname"; //...
	}
    }
    else {
	TKamePlainTextRepository* PlainTextRepository = (
	    new TKamePlainTextRepository(RepositoryName)
	);
	if (TMushFileAttribute(ObjectName).Extension() == "gz") {
	    PlainTextRepository->EnableCompression();
	    ObjectName = TMushFileAttribute(ObjectName).FileRootName();
	}
	if (! _Delimiter.empty()) {
	    PlainTextRepository->SetDelimiter(_Delimiter);
	}
	if (! _Quote.empty()) {
	    PlainTextRepository->SetQuote(_Quote);
	}

	Repository = PlainTextRepository;
    }
    
    SaveTo(Repository, ObjectName);
    delete Repository;
}

void TKiscKameObject::Load(string RepositoryName, string ObjectName) throw(TScriptException)
{
    TKameRepository* Repository;

    if (TMushFileAttribute(RepositoryName).Extension() == "root") {
	Repository = (
	    TKameRootRepositoryFactory().CreateRepository(RepositoryName)
	);
	if (Repository == 0) {
	    throw TScriptException("ROOT is not available");
	}
	if (ObjectName.empty()) {
	    ObjectName = "noname"; //...
	}
    }
    else {
	TKamePlainTextRepository* PlainTextRepository = (
	    new TKamePlainTextRepository(RepositoryName)
	);
	if (TMushFileAttribute(ObjectName).Extension() == "gz") {
	    ObjectName = TMushFileAttribute(ObjectName).FileRootName();
	}
	if (! _Delimiter.empty()) {
	    PlainTextRepository->SetDelimiter(_Delimiter);
	}
	if (! _Quote.empty()) {
	    PlainTextRepository->SetQuote(_Quote);
	}
	delete _HeaderProcessor;
	_HeaderProcessor = new TKiscRepositoryHeaderProcessor();	
	PlainTextRepository->SetHeaderProcessor(_HeaderProcessor);
	Repository = PlainTextRepository;
    }
    
    LoadFrom(Repository, ObjectName);
    delete Repository;
	
    ProcessScript(_HeaderProcessor->ScriptText());
    OnLoad();
}

void TKiscKameObject::TokenizeLine(const std::string& Line, std::vector<std::string>& TokenList) throw(TScriptException)
{
    if (_HeaderProcessor == 0) {
	_HeaderProcessor = new TKiscRepositoryHeaderProcessor();
    }

    try {
	_HeaderProcessor->TokenizeLine(Line, TokenList);
    }
    catch (TKameException &e) {
	throw TScriptException(e.Message());
    }
}

void TKiscKameObject::Construct(void)
{
    if (_Parser != 0) {
	return;
    }

    //_Parser = new TParaStandardParser();
    _Parser = new TKiscScript();

    _Package = _Parser->GetPackage();
    _ExpressionParser = _Parser->GetExpressionParser();
    _StatementParser = _Parser->GetStatementParser();
    _SymbolTable = _Parser->GetSymbolTable();

    _SymbolTable->RegisterVariable("pi", TParaValue(3.141592));
    _SymbolTable->RegisterVariable("e", TParaValue(2.718281828));
}

void TKiscKameObject::ProcessScript(const std::string& ScriptText) throw(TScriptException)
{
    if (ScriptText.empty()) {
	return;
    }

    if (_Parser == 0) {
	Construct();
    }

    istringstream is(ScriptText);
    TParaTokenizer Tokenizer(is, _Parser->GetTokenTable());

    while ((! Tokenizer.LookAhead().IsEmpty())) {
	TParaPackageEntry* Entry = _Package->CreateEntry(&Tokenizer);
	if (Entry != 0) {
	    Entry->Parse(&Tokenizer, _StatementParser, _SymbolTable);
	    _EntryList.push_back(Entry);
	}
	else {
	    TParaStatement* Statement = 0;
	    try {
		Statement = _StatementParser->Parse(&Tokenizer, _SymbolTable);
		Statement->Execute(_SymbolTable);
	    }
	    catch (TScriptException &e) {
		delete Statement;	
		throw;
	    }
	    delete Statement;	
	}
    }
}



TKiscRepositoryHeaderProcessor::TKiscRepositoryHeaderProcessor(void)
{
}

TKiscRepositoryHeaderProcessor::~TKiscRepositoryHeaderProcessor()
{
}

bool TKiscRepositoryHeaderProcessor::ProcessLine(const std::string& Line)
{
    if ((Line.size() < 2) || (Line[1] != '%')) {
	return TKamePlainTextHeaderProcessor::ProcessLine(Line);
    }

    _ScriptText += Line.substr(2) + "\n";

    return true;
}

const std::string& TKiscRepositoryHeaderProcessor::ScriptText(void)
{
    return _ScriptText;
}



TKiscNtuple::TKiscNtuple(void)
{
}

TKiscNtuple::TKiscNtuple(int NumberOfColumns)
: TKameNtuple(NumberOfColumns)
{
}    

TKiscNtuple::~TKiscNtuple()
{
}

void TKiscNtuple::SaveTo(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->SaveNtuple(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Ntuple::SaveTo()", e.Message());
    }
}

void TKiscNtuple::LoadFrom(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->LoadNtuple(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Ntuple::LoadFrom()", e.Message());
    }
}

void TKiscNtuple::OnLoad(void) throw(TScriptException)
{
    EvalElements();
}

void TKiscNtuple::EvalElements(void) throw(TScriptException)
{
    vector<int> ColumnList;
    for (unsigned Column = 0; Column < NumberOfColumns(); Column++) {
	if (ColumnTypeOf(Column) == "eval") {
	    ColumnList.push_back(Column);
	}
    }
    if (ColumnList.empty()) {
	return;
    }

    if (_Parser == 0) {
	Construct();
    }

    for (unsigned Row = 0; Row < NumberOfRows(); Row++) {
	for (unsigned j = 0; j < ColumnList.size(); j++) {
	    int Column = ColumnList[j];
	    istringstream ExpressionStream((*this)[Row][Column]);
	    TParaTokenizer Tokenizer(
		ExpressionStream, _Parser->GetTokenTable()
	    );
	    TParaExpression* Expression = 0;
	    try {
		Expression = _ExpressionParser->Parse(
		    &Tokenizer, _SymbolTable
		);
		TParaValue Value = Expression->Evaluate(_SymbolTable);
		if (Value.IsLong()) {
		    (*this)[Row][Column] = Value.AsLong();
		}
		else if (Value.IsDouble()) {
		    (*this)[Row][Column] = Value.AsDouble();
		}
		else if (Value.IsString()) {
		    (*this)[Row][Column] = Value.AsString();
		}
		else {
		    (*this)[Row][Column] = TKameUnknownParaValue(Value);
		}
	    }
	    catch (TScriptException &e) {
		delete Expression;
		throw TScriptException(
		    "TKiscNtuple::EvalElements()",
		    "invalid data word: " + string((*this)[Row][Column])
		);
	    }
	    delete Expression;
	}
    }
}



TKiscGraph::TKiscGraph(unsigned InitialListSize)
: TKameGraph(InitialListSize)
{
}

TKiscGraph::TKiscGraph(const kame::TKameGraph& Graph)
: TKameGraph(Graph)
{
}

TKiscGraph::~TKiscGraph()
{
}

void TKiscGraph::SaveTo(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->SaveGraph(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Graph::SaveTo()", e.Message());
    }
}

void TKiscGraph::LoadFrom(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->LoadGraph(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Graph::LoadFrom()", e.Message());
    }
}



TKiscHistogram::TKiscHistogram()
{
}

TKiscHistogram::TKiscHistogram(const TKameHistogramScale& Scale)
: TKameHistogram(Scale)
{
}

TKiscHistogram::~TKiscHistogram()
{
}

void TKiscHistogram::SaveTo(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->SaveHistogram(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Histogram::SaveTo()", e.Message());
    }
}

void TKiscHistogram::LoadFrom(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->LoadHistogram(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Histogram::LoadFrom()", e.Message());
    }
}



TKiscHistogram2d::TKiscHistogram2d(void)
{
}

TKiscHistogram2d::TKiscHistogram2d(const kame::TKameHistogramScale& XScale, const kame::TKameHistogramScale& YScale)
: TKameHistogram2d(XScale, YScale)
{
}

TKiscHistogram2d::~TKiscHistogram2d()
{
}

void TKiscHistogram2d::SaveTo(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->SaveHistogram2d(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Histogram2d::SaveTo()", e.Message());
    }
}

void TKiscHistogram2d::LoadFrom(kame::TKameRepository* Repository, const std::string& Name) throw(TScriptException)
{
    try {
	Repository->LoadHistogram2d(*this, Name);
    }
    catch (TKameException &e) {
	throw TScriptException("Histogram2d::LoadFrom()", e.Message());
    }
}
