/* KiscSystemScript.cc */
/* Created by Enomoto Sanshiro on 4 July 2001. */
/* Last updated by Enomoto Sanshiro on 4 July 2001. */


#include <sstream>
#include <iomanip>
#include "MushProcess.hh"
#include "MushSignal.hh"
#include "MushFileSystem.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "MushConsole.hh"
#include "MushRegularExpression.hh"
#include "ParaParser.hh"
#include "KiscSystemScript.hh"

using namespace std;


TKiscSystemScript::TKiscSystemScript(TMushConsole* Console)
{
    if (Console == 0) {
	_MyConsole = new TMushReadlineConsole();
	_Console = _MyConsole;
    }
    else {
        _MyConsole = 0;
	_Console = Console;
    }
}

TKiscSystemScript::~TKiscSystemScript()
{
    delete _MyConsole;
}

TParaTokenTable* TKiscSystemScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable;
    TokenTable = TParaParser::CreateTokenTable();

    TokenTable->AddOperator("#");
    TokenTable->AddOperator("..");
    TokenTable->AddOperator("=~");
    TokenTable->AddOperator("!~");

    return TokenTable;
}

TParaOperatorTable* TKiscSystemScript::CreateOperatorTable(void)
{
    TParaOperatorTable* OperatorTable;
    OperatorTable = TParaParser::CreateOperatorTable();

    OperatorTable->AddPrepositionalOperator(
	new TKiscOperatorBitGenerate()
    );
    OperatorTable->AddOperator(
	new TKiscOperatorBitFill(), TParaOperatorPriority("&", -1)
    );
    
    OperatorTable->AddPostpositionalOperator(
	new TKiscOperatorMatch()
    );
    OperatorTable->AddPostpositionalOperator(
	new TKiscOperatorNotMatch()
    );
    OperatorTable->AddElementaryOperator(
	new TKiscOperatorSplit()
    );

    return OperatorTable;
}

TParaBuiltinFunctionTable* TKiscSystemScript::CreateBuiltinFunctionTable(void)
{
    TParaBuiltinFunctionTable* BuiltinFunctionTable;
    BuiltinFunctionTable = TParaParser::CreateBuiltinFunctionTable();

    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKiscEnvironmentVariableObject()
    );
    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKiscSystemObject(_Console)
    );
    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKiscNumberTextObject()
    );

    return BuiltinFunctionTable;
}

TParaObjectPrototypeTable* TKiscSystemScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TParaParser::CreateObjectPrototypeTable();

    ObjectPrototypeTable->RegisterClass(new TKiscRegularExpressionObject());

    return ObjectPrototypeTable;
}

TParaOperator* TKiscOperatorBitGenerate::Clone(void) const
{
    return new TKiscOperatorBitGenerate();
}

string TKiscOperatorBitGenerate::Symbol(void) const
{
    return string("#");
}

string TKiscOperatorBitGenerate::Name(void) const
{
    return string("BitGenerate");
}

TParaValue& TKiscOperatorBitGenerate::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException) 
{
    long BitPosition = Right.AsLong();
    if (BitPosition < 0) {
	throw TScriptException("natural number is expected");
    }

    return Result = TParaValue((long) (0x0001ul << BitPosition));
} 



TParaOperator* TKiscOperatorBitFill::Clone(void) const
{
    return new TKiscOperatorBitFill();
}

string TKiscOperatorBitFill::Symbol(void) const
{
    return string("..");
}

string TKiscOperatorBitFill::Name(void) const
{
    return string("BitFill");
}

TParaValue& TKiscOperatorBitFill::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException) 
{
    long LongValue;

    if (Left.AsLong() == Right.AsLong()) {
	LongValue = Left.AsLong();
    }
    else {
	LongValue = Left.AsLong() | Right.AsLong();
	bool IsFilled = false;
	for (long Bit = 0x0001; Bit < LongValue; Bit <<= 1) {
	    if (LongValue & Bit) {
		if (! IsFilled) {
		    IsFilled = true;
		}
		else {
		    break;
		}
	    }
	    if (IsFilled) {
		LongValue |= Bit;
	    }
	}
    }

    return Result = TParaValue((long) LongValue);
} 



TKiscSystemObject::TKiscSystemObject(TMushConsole* Console)
: TParaObjectPrototype("KiscSystem")
{
    _Console = Console;
}

TKiscSystemObject::~TKiscSystemObject()
{
}

TParaObjectPrototype* TKiscSystemObject::Clone(void)
{
    return new TKiscSystemObject(_Console);
}

int TKiscSystemObject::MethodIdOf(const string& MethodName)
{
    if ((MethodName == "readLine") || (MethodName == "readln")) {
        return MethodId_ReadLine;
    }
    else if (MethodName == "execute") {
        return MethodId_ExecuteProcess;
    }
    else if (MethodName == "wait") {
        return MethodId_WaitProcess;
    }
    else if (MethodName == "kill") {
        return MethodId_KillProcess;
    }
    else if (MethodName == "sleep") {
        return MethodId_Sleep;
    }
    else if ((MethodName == "makeDirectory") || (MethodName == "mkdir")) {
        return MethodId_MakeDirectory;
    }
    else if ((MethodName == "currentDirectory") || (MethodName == "pwd")) {
        return MethodId_CurrentDirectory;
    }
    else if ((MethodName == "changeDirectory") || (MethodName == "cd")) {
        return MethodId_ChangeDirectory;
    }
    else if (MethodName == "time") {
        return MethodId_Time;
    }
    else if (MethodName == "timeOf") {
        return MethodId_TimeOf;
    }
    else if (MethodName == "localTime") {
        return MethodId_LocalTime;
    }
    else if (MethodName == "localTimeOf") {
        return MethodId_LocalTimeOf;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscSystemObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_ReadLine) {
        ReturnValue = ReadLine(ArgumentList);
    }
    else if (MethodId == MethodId_ExecuteProcess) {
        ReturnValue = ExecuteProcess(ArgumentList);
    }
    else if (MethodId == MethodId_WaitProcess) {
        ReturnValue = WaitProcess(ArgumentList);
    }
    else if (MethodId == MethodId_KillProcess) {
        ReturnValue = KillProcess(ArgumentList);
    }
    else if (MethodId == MethodId_Sleep) {
        ReturnValue = Sleep(ArgumentList);
    }
    else if (MethodId == MethodId_MakeDirectory) {
        ReturnValue = MakeDirectory(ArgumentList);
    }
    else if (MethodId == MethodId_CurrentDirectory) {
        ReturnValue = CurrentDirectory(ArgumentList);
    }
    else if (MethodId == MethodId_ChangeDirectory) {
        ReturnValue = ChangeDirectory(ArgumentList);
    }
    else if (MethodId == MethodId_Time) {
        ReturnValue = Time(ArgumentList);
    }
    else if (MethodId == MethodId_TimeOf) {
        ReturnValue = TimeOf(ArgumentList);
    }
    else if (MethodId == MethodId_LocalTime) {
        ReturnValue = LocalTime(ArgumentList);
    }
    else if (MethodId == MethodId_LocalTimeOf) {
        ReturnValue = LocalTimeOf(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKiscSystemObject::ReadLine(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string Prompt = "";
    if (ArgumentList.size() > 0) {
	Prompt = ArgumentList[0]->AsString();
    }

    string OldPrompt = _Console->SetPrompt(Prompt);

    string Line;
    if (_Console->GetLine(Line)) {
	Line += '\n';
    }

    _Console->SetPrompt(OldPrompt);

    return TParaValue(Line);
}

TParaValue TKiscSystemObject::ExecuteProcess(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "execute()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "execute()", "invalid argument[s]"
	);
    }

    string Line = ArgumentList[0]->AsString();
    istringstream LineStream(Line);

    string Path, Argument;
    vector<string> ProcessArgumentList;
    LineStream >> Path;
    ProcessArgumentList.push_back(Path);
    while (LineStream >> Argument) {
	ProcessArgumentList.push_back(Argument);
    }

    for (unsigned i = 1; i < ArgumentList.size(); i++) {
	ProcessArgumentList.push_back(ArgumentList[i]->AsString());
    }

    long ProcessId;
    TMushChildProcess Process(Path, ProcessArgumentList);
    try {
	Process.Run();
	ProcessId = Process.ProcessId();
    }
    catch (TSystemCallException &e) {
	ProcessId = -1;
    }

    return TParaValue(ProcessId);
}

TParaValue TKiscSystemObject::WaitProcess(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    int ProcessId = -1;  // means to wait for any child process
    int Result = 0;

    if (ArgumentList.size() > 0) {
	if (! ArgumentList[0]->IsLong()) {
	    throw TScriptException(
		"wait(int process_id)", "invalid argument[s]"
	    );
	}
	ProcessId = ArgumentList[0]->AsLong();
    }

    try {
	Result = TMushProcess::Wait(ProcessId);
    }
    catch (TSystemCallException &e) {
	throw TScriptException("wait()", e.Message());
    }

    return TParaValue((long) Result);
}

TParaValue TKiscSystemObject::KillProcess(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    int ProcessId = 0, SignalId = SIGTERM;
    int Result = 0;

    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsLong())) {
	throw TScriptException(
	    "kill(int process_id, int signal_id)", "invalid argument[s]"
	);
    }
    ProcessId = ArgumentList[0]->AsLong();

    if (ArgumentList.size() >= 2) {
	if (! ArgumentList[1]->IsLong()) {
	    throw TScriptException(
		"kill(int process_id, int signal_id)", "invalid argument[s]"
	    );
	}
	SignalId = ArgumentList[1]->AsLong();
    }

    try {
	Result = TMushSignalSender::SendSignal(ProcessId, SignalId);
    }
    catch (TSystemCallException &e) {
	throw TScriptException("kill()", e.Message());
    }

    return TParaValue((long) Result);
}

TParaValue TKiscSystemObject::CurrentDirectory(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string DirectoryName;
    try {
	DirectoryName = TMushFileSystem::CurrentDirectory();
    }
    catch (TSystemCallException &e) {
	throw TScriptException(
	    "changeDirectory", "system call exception: " + e.Message()
	);
    }

    return TParaValue(DirectoryName);
}

TParaValue TKiscSystemObject::MakeDirectory(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("too few arguments: makeDirectory(string name)");
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException("invalid arguments: makeDirectory(string name)");
    }

    int Result;
    string DirectoryName = ArgumentList[0]->AsString();
    try {
	Result = TMushFileSystem::MakeDirectory(DirectoryName);
    }
    catch (TSystemCallException &e) {
	throw TScriptException(
	    "makeDirectory", "system call exception: " + e.Message()
	);
    }

    return TParaValue((long) Result);
}

TParaValue TKiscSystemObject::ChangeDirectory(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("too few arguments: changeDirectory(string name)");
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException("invalid arguments: changeDirectory(string name)");
    }

    string DirectoryName = ArgumentList[0]->AsString();
    try {
	TMushFileSystem::ChangeDirectory(DirectoryName);
    }
    catch (TSystemCallException &e) {
	throw TScriptException(
	    "changeDirectory", "system call exception: " + e.Message()
	);
    }

    return TParaValue((long) 0);
}

TParaValue TKiscSystemObject::Sleep(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsLong())) {
	throw TScriptException(
	    "invalid argument[s]: sleep(int sec, ...)"
	);
    }
    if ((ArgumentList.size() > 1) && (! ArgumentList[1]->IsLong())) {
	throw TScriptException(
	    "invalid argument[s]: sleep(int sec, int usec)"
	);
    }

    int Time_sec = ArgumentList[0]->AsLong();
    int Time_usec = 0;

    if (ArgumentList.size() > 1) {
	Time_usec = ArgumentList[1]->AsLong();
    }
    
    TMushRealTimeTimer(Time_sec, Time_usec).Suspend();

    return TParaValue((long) 0);
}

TParaValue TKiscSystemObject::Time(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    long TimeValue = TMushDateTime::SecSinceEpoch();
    return TParaValue(TimeValue);
}

TParaValue TKiscSystemObject::TimeOf(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 6) {
	throw TScriptException(
	    "timeOf(int year, int month, int day, int hour, int min, int sec)",
	    "invalid argument[s]"
	);
    }

    long Year, Month, Day, Hour, Min, Sec;
    try {
	Year = ArgumentList[0]->AsLong();
	Month = ArgumentList[1]->AsLong();
	Day = ArgumentList[2]->AsLong();
	Hour = ArgumentList[3]->AsLong();
	Min = ArgumentList[4]->AsLong();
	Sec = ArgumentList[5]->AsLong();
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "timeOf(int year, int month, int day, int hour, int min, int sec)",
	    "invalid argument[s]"
	);
    }

    TMushDateTime DateTime(Year, Month, Day, Hour, Min, Sec);

    return TParaValue((long) DateTime);
}

TParaValue TKiscSystemObject::LocalTime(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string LocalTimeString;

    if (ArgumentList.size() > 0) {
	if (! ArgumentList[0]->IsString()) {
	    throw TScriptException(
		"localTime(string time_format = \"\")", "invalid argument[s]"
	    );
	}

	string TimeFormat = ArgumentList[0]->AsString();
	LocalTimeString = TMushDateTime().AsString(TimeFormat);
    }
    else {
	LocalTimeString = TMushDateTime().AsString();
    }
    
    return TParaValue(LocalTimeString);
}

TParaValue TKiscSystemObject::LocalTimeOf(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string LocalTimeString;

    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsLong())) {
	throw TScriptException(
	    "localTime(long time_value, string time_format = \"\")", 
	    "invalid argument[s]"
	);
    }
    long TimeValue = ArgumentList[0]->AsLong();

    if (ArgumentList.size() > 1) {
	if (! ArgumentList[1]->IsString()) {
	    throw TScriptException(
		"localTime(long time_value, string time_format = \"\")", 
		"invalid argument[s]"
	    );
	}

	string TimeFormat = ArgumentList[1]->AsString();
	LocalTimeString = TMushDateTime(TimeValue).AsString(TimeFormat);
    }
    else {
	LocalTimeString = TMushDateTime(TimeValue).AsString();
    }
    
    return TParaValue(LocalTimeString);
}



TKiscEnvironmentVariableObject::TKiscEnvironmentVariableObject(void)
: TParaObjectPrototype("EnvironmentVariable")
{
}

TKiscEnvironmentVariableObject::~TKiscEnvironmentVariableObject()
{
}

TParaObjectPrototype* TKiscEnvironmentVariableObject::Clone(void)
{
    return new TKiscEnvironmentVariableObject();
}

int TKiscEnvironmentVariableObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "getEnvironmentVariable") {
        return MethodId_Get;
    }
    else if (MethodName == "setEnvironmentVariable") {
        return MethodId_Set;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscEnvironmentVariableObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_Get) {
        ReturnValue = Get(ArgumentList);
    }
    else if (MethodId == MethodId_Set) {
        ReturnValue = Set(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKiscEnvironmentVariableObject::Get(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::get()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::get()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = TMushEnvironmentVariable::Get(Name);

    return TParaValue(Value);
}

TParaValue TKiscEnvironmentVariableObject::Set(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    _InternalClassName + "::set()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::set()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    TMushEnvironmentVariable::Set(Name, Value);

    return TParaValue();
}



TKiscRegularExpressionObject::TKiscRegularExpressionObject(void)
: TParaObjectPrototype("RegularExpression")
{
    _RegularExpression = 0;
}

TKiscRegularExpressionObject::~TKiscRegularExpressionObject()
{
    delete _RegularExpression;
}

TParaObjectPrototype* TKiscRegularExpressionObject::Clone(void)
{
    return new TKiscRegularExpressionObject();
}

void TKiscRegularExpressionObject::Construct(const std::string& ClassName, std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "RegularExpression::RegularExpression(string pattern)",
	    "too few argument[s]"
	);
    }
    string Pattern = ArgumentList[0]->AsString();

    if (ArgumentList.size() >= 2) {
	_Substitution = ArgumentList[1]->AsString();
    }

    _RegularExpression = new TMushPosixRegularExpression(Pattern);
}

void TKiscRegularExpressionObject::Destruct(void) throw(TScriptException)
{
    delete _RegularExpression;
    _RegularExpression = 0;
}

int TKiscRegularExpressionObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "match") {
        return MethodId_Match;
    }
    else if (MethodName == "substitute") {
        return MethodId_Substitute;
    }
    else if (MethodName == "substituteAll") {
        return MethodId_SubstituteAll;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscRegularExpressionObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_Match) {
        ReturnValue = Match(ArgumentList);
    }
    else if (MethodId == MethodId_Substitute) {
        ReturnValue = Substitute(ArgumentList);
    }
    else if (MethodId == MethodId_SubstituteAll) {
        ReturnValue = SubstituteAll(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKiscRegularExpressionObject::Match(std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "RegularExpression::Match(string text)",
	    "too few argument[s]"
	);
    }
    string Text = ArgumentList[0]->AsString();

    TParaValue Result = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = Result.AsValueList();

    _RegularExpression->Match(Text);

    int NumberOfSubMatchs = _RegularExpression->NumberOfSubMatchs();
    for (int i = 0; i < NumberOfSubMatchs; i++) {
	ResultList.push_back(
	    TParaValue(_RegularExpression->SubMatchOf(i))
	);
    }

    return Result;
}

TParaValue TKiscRegularExpressionObject::Substitute(std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "RegularExpression::Substitute(string text)",
	    "too few argument[s]"
	);
    }
    string Text = ArgumentList[0]->AsString();

    string Result = _RegularExpression->Substitute(Text, _Substitution);

    return TParaValue(Result);
}

TParaValue TKiscRegularExpressionObject::SubstituteAll(std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "RegularExpression::SubstituteAll(string text)",
	    "too few argument[s]"
	);
    }
    string Text = ArgumentList[0]->AsString();

    string Result = _RegularExpression->SubstituteAll(Text, _Substitution);

    return TParaValue(Result);
}



TKiscRegularExpressionLiteral::TKiscRegularExpressionLiteral(void)
{
}

TKiscRegularExpressionLiteral::~TKiscRegularExpressionLiteral()
{
}

void TKiscRegularExpressionLiteral::Parse(TParaTokenizer* Tokenizer) throw(TScriptException)
{
    _Operator = Operator_Match;
    if (Tokenizer->LookAhead().IsNot("/")) {
	TParaToken Token = Tokenizer->Next();
	if (Token.Is("m")) {
	    ;
	}
	else if (Token.Is("s")) {
	    _Operator = Operator_Substitute;
	}
	else {
	    throw TScriptException(
		"unknown regular expression operator: " + Token.AsString()
	    );
	}
    }
    Tokenizer->Next().MustBe("/");

    _Pattern = ParsePattern(Tokenizer);
    if (_Operator == Operator_Substitute) {
	_Substitution = ParseSubstitution(Tokenizer);
    }

    _Modifier = 0;
    char Character;
    while (Tokenizer->GetChar(Character) && (isalpha(Character))) {
	if (Character == 'g') {
	    _Modifier |= Modifier_Global;
	}
	else if (Character == 'i') {
	    _Modifier |= Modifier_CaseInsensitive;
	}
#if 0
	else if (Character == 'm') {
	    _Modifier |= Modifier_MultipleLine;
	}
	else if (Character == 's') {
	    _Modifier |= Modifier_SingleLine;
	}
#endif
	else {
	    throw TScriptException(
		string("unknown regular expression modifier: ") + Character
	    );
	}
    }
    if (Character != 0) {
	Tokenizer->UngetChar(Character);
    }
}

void TKiscRegularExpressionLiteral::ProcessEscape(char& Character)
{
    switch (Character) {
      case 'a':
	Character = '\a'; break;
      case 'b':
	Character = '\b'; break;
      case 'f':
	Character = '\f'; break;
      case 'n':
	Character = '\n'; break;
      case 'r':
	Character = '\r'; break;
      case 't':
	Character = '\t'; break;
      case 'v':
	Character = '\v'; break;
      default:
	;
    }
}

string TKiscRegularExpressionLiteral::ParsePattern(TParaTokenizer* Tokenizer) throw(TScriptException)
{
    string Result;

    bool IsEscaped = false;
    bool IsSetBox = false;
    char Character;
    while (1) {
	if (! Tokenizer->GetChar(Character) || (Character == '\n')) {
	    throw TScriptException("unexpected end-of-file");
	}

	if (! IsEscaped && ! IsSetBox && (Character == '/')) {
	    break;
	}

	if (IsEscaped) {
	    IsEscaped = false;
	    if (IsSetBox) {
		ProcessEscape(Character);
	    }
	}
	else {
	    if (Character == '\\') {
		IsEscaped = true;
		if (IsSetBox) {
		    continue;
		}
	    }
	    else if (! IsSetBox && (Character == '[')) {
		IsSetBox = true;
	    }
	    else if (IsSetBox && (Character == ']')) {
		IsSetBox = false;
	    }
	}

	Result += Character;
    }

    return Result;
}

string TKiscRegularExpressionLiteral::ParseSubstitution(TParaTokenizer* Tokenizer) throw(TScriptException)
{
    string Result;

    bool IsEscaped = false;
    char Character;
    while (1) {
	if (! Tokenizer->GetChar(Character)) {
	    throw TScriptException("unexpected end-of-file");
	}

	if (! IsEscaped && (Character == '/')) {
	    break;
	}

	if (IsEscaped) {
	    IsEscaped = false;
	    ProcessEscape(Character);
	}
	else if (Character == '\\') {
	    IsEscaped = true;
	    continue;
	}

	Result += Character;
    }

    return Result;
}

const std::string& TKiscRegularExpressionLiteral::Pattern(void) const
{
    return _Pattern;
}

const std::string& TKiscRegularExpressionLiteral::Substitution(void) const
{
    return _Substitution;
}

int TKiscRegularExpressionLiteral::Operator(void) const
{
    return _Operator;
}

int TKiscRegularExpressionLiteral::Modifier(void) const
{
    return _Modifier;
}

bool TKiscRegularExpressionLiteral::IsMatch(void) const
{
    return (_Operator == Operator_Match);
}

bool TKiscRegularExpressionLiteral::IsSubstitution(void) const
{
    return (_Operator == Operator_Substitute);
}

bool TKiscRegularExpressionLiteral::IsGlobal(void) const
{
    return (_Modifier & Modifier_Global);
}

bool TKiscRegularExpressionLiteral::IsCaseInsensitive(void) const
{
    return (_Modifier & Modifier_CaseInsensitive);
}



TKiscOperatorMatch::TKiscOperatorMatch(void)
{
    _RegularExpression = 0;
    _Literal = 0;
}

TKiscOperatorMatch::~TKiscOperatorMatch()
{
    delete _RegularExpression;
    delete _Literal;
}

TParaOperator* TKiscOperatorMatch::Clone(void) const
{
    return new TKiscOperatorMatch();
}

std::string TKiscOperatorMatch::Symbol(void) const
{
    return string("=~");
}

string TKiscOperatorMatch::Name(void) const
{
    return string("Match");
}

void TKiscOperatorMatch::Parse(TParaTokenizer* Tokenizer, TParaExpressionParser* ExpressionParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(Symbol());

    _Literal = new TKiscRegularExpressionLiteral();
    _Literal->Parse(Tokenizer);

    _RegularExpression = new TMushPosixRegularExpression(
	_Literal->Pattern(), _Literal->IsCaseInsensitive()
    );
}

TParaValue& TKiscOperatorMatch::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    string Text = Left.AsString();

    if (_Literal->IsMatch()) {
	Result = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = Result.AsValueList();

	_RegularExpression->Match(Text);

	int NumberOfSubMatchs = _RegularExpression->NumberOfSubMatchs();
	for (int i = 0; i < NumberOfSubMatchs; i++) {
	    ResultList.push_back(
		TParaValue(_RegularExpression->SubMatchOf(i))
	    );
	}
    }
    else if (_Literal->IsSubstitution()) {
	const string& Substitution = _Literal->Substitution();
	if (_Literal->IsGlobal()) {
	    Result = TParaValue(
		_RegularExpression->SubstituteAll(Text, Substitution)
	    );
	}
	else {
	    Result = TParaValue(
		_RegularExpression->Substitute(Text, Substitution)
	    );
	}

	if (Left.IsLeftValue()) {
	    Left.Assign(Result);
	}
    }

    return Result;
}



TKiscOperatorNotMatch::TKiscOperatorNotMatch(void)
{
}

TKiscOperatorNotMatch::~TKiscOperatorNotMatch()
{
}

TParaOperator* TKiscOperatorNotMatch::Clone(void) const
{
    return new TKiscOperatorNotMatch();
}

std::string TKiscOperatorNotMatch::Symbol(void) const
{
    return string("!~");
}

string TKiscOperatorNotMatch::Name(void) const
{
    return string("NotMatch");
}

TParaValue& TKiscOperatorNotMatch::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    Result = TKiscOperatorMatch::Evaluate(Left, Right, SymbolTable, Result);
    return Result = TParaValue(! Result.AsBool());
}



TKiscOperatorSplit::TKiscOperatorSplit(void)
{
    _TextExpression = 0;
    _RegularExpression = 0;
}

TKiscOperatorSplit::~TKiscOperatorSplit()
{
    delete _TextExpression;
    delete _RegularExpression;
}

TParaOperator* TKiscOperatorSplit::Clone(void) const
{
    return new TKiscOperatorSplit();
}

std::string TKiscOperatorSplit::Symbol(void) const
{
    return string("split");
}

string TKiscOperatorSplit::Name(void) const
{
    return string("Split");
}

void TKiscOperatorSplit::Parse(TParaTokenizer* Tokenizer, TParaExpressionParser* ExpressionParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(Symbol());
    Tokenizer->Next().MustBe("(");
    _TextExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);

    if (Tokenizer->LookAhead().IsNot(")")) {
	Tokenizer->Next().MustBe(",");

	TKiscRegularExpressionLiteral Literal;
	Literal.Parse(Tokenizer);
	
	if (! Literal.IsMatch()) {
	    throw TScriptException("split: invalid regular expression");
	}
	_Pattern = Literal.Pattern();
    }
    else {
	_Pattern = "[ \t\n\r]+";
    }
    Tokenizer->Next().MustBe(")");

    _RegularExpression = new TMushPosixRegularExpression(_Pattern);
}

TParaValue& TKiscOperatorSplit::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    string Text = _TextExpression->Evaluate(SymbolTable).AsString();

    Result = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = Result.AsValueList();

    int Offset, Size;
    while (_RegularExpression->FindPattern(Text, Offset, Size)) {
	ResultList.push_back(TParaValue(Text.substr(0, Offset)));
	Text.erase(0, Offset + Size);
    }
    ResultList.push_back(TParaValue(Text));

    return Result;
}



TKiscNumberTextObject::TKiscNumberTextObject(void)
: TParaObjectPrototype("KiscNumberText")
{
}

TKiscNumberTextObject::~TKiscNumberTextObject()
{
}

TParaObjectPrototype* TKiscNumberTextObject::Clone(void)
{
    return new TKiscNumberTextObject();
}

int TKiscNumberTextObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "fixed") {
	return MethodId_Fixed;
    }
    else if (MethodName == "scientific") {
	return MethodId_Scientific;
    }
    else if (MethodName == "hex") {
	return MethodId_Hex;
    }
    else if (MethodName == "dec") {
	return MethodId_Dec;
    }
    else if (MethodName == "bin") {
	return MethodId_Bin;
    }
    else if (MethodName == "math") {
	return MethodId_Math;
    }
    
    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscNumberTextObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() >= 1) && ArgumentList[0]->IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();

	TParaValue Result;
	vector<TParaValue*> InputArgumentList;
	InputArgumentList.push_back(&InputList[0]);
	for (unsigned i = 1; i < ArgumentList.size(); i++) {
	    InputArgumentList.push_back(ArgumentList[i]);
	}

	for (unsigned i = 0; i < InputList.size(); i++) {
	    InputArgumentList[0] = &InputList[i];
	    if (this->InvokeMethod(MethodId, InputArgumentList, Result)) {
		ResultList.push_back(Result);
	    }
	    else {
		break;
	    }
	}
	return ResultList.size();
    }


    if (MethodId == MethodId_Fixed) {
	Fixed(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Scientific) {
	Scientific(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Hex) {
	Hex(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Dec) {
	Dec(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Bin) {
	Bin(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Math) {
	Math(ArgumentList, ReturnValue);
    }
    else {
	return TParaObjectPrototype::InvokeMethod(
	    MethodId, ArgumentList, ReturnValue
	);
    }

    return 1;
}

void TKiscNumberTextObject::Fixed(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "fixed(double value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsDouble()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    double Value = ArgumentList[0]->AsDouble();

    ostringstream Stream;
    Stream.setf(ios::fixed);

    if (ArgumentList.size() > 1) {
	int Precision = ArgumentList[1]->AsLong();
	Stream << setprecision(Precision);
    }

    Stream << Value;

    ReturnValue = TParaValue(Stream.str());
}

void TKiscNumberTextObject::Scientific(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "scientific(double value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsDouble()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    double Value = ArgumentList[0]->AsDouble();

    ostringstream Stream;
    Stream.setf(ios::scientific);

    if (ArgumentList.size() > 1) {
	int Precision = ArgumentList[1]->AsLong();
	Stream << setprecision(Precision);
    }

    Stream << Value;

    ReturnValue = TParaValue(Stream.str());
}

void TKiscNumberTextObject::Hex(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Hex(int value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsLong()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    long Value = ArgumentList[0]->AsLong();

    ostringstream Stream;
    Stream.setf(ios::hex);

    if (ArgumentList.size() > 1) {
	int Width = ArgumentList[1]->AsLong();
	Stream << setfill('0') << setw(Width);
    }

    Stream << hex << Value << setfill(' ') << dec;

    ReturnValue = TParaValue(Stream.str());
}

void TKiscNumberTextObject::Dec(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Dec(int value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsLong()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    long Value = ArgumentList[0]->AsLong();

    ostringstream Stream;
    if (ArgumentList.size() > 1) {
	int Width = ArgumentList[1]->AsLong();
	Stream << setfill('0') << setw(Width);
    }

    Stream << dec << Value << setfill(' ');

    ReturnValue = TParaValue(Stream.str());
}

void TKiscNumberTextObject::Bin(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "bin(int value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsLong()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    unsigned long Value = (unsigned long) ArgumentList[0]->AsLong();

    int Length = 8 * sizeof(long);
    int Width = 0;
    if (ArgumentList.size() > 1) {
	Width = ArgumentList[1]->AsLong();
    }

    string NumberText;
    while (Width > Length) {
	NumberText += '0';
	Width--;
    }
    for (int Bit = Length-1; Bit >= 0; Bit--) {
	if (Value & (0x01ul << Bit)) {
	    if (Width <= Bit) {
		Width = Bit+1;
	    }
	    NumberText += '1';
	}
	else if (Bit < Width) {
	    NumberText += '0';
	}
    }

    ReturnValue = TParaValue(NumberText);
}

void TKiscNumberTextObject::Math(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "math(double value, ...)", "too few argument"
	);
    }
    if (! ArgumentList[0]->IsDouble()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsString());
	return;
    }
    double Value = ArgumentList[0]->AsDouble();

    ostringstream Stream;
    Stream.setf(ios::scientific);

    if (ArgumentList.size() > 1) {
	int Precision = ArgumentList[1]->AsLong();
	Stream << setprecision(Precision);
    }

    Stream << Value;

    string String = Stream.str();
    string Result;
    size_t Length = String.find_first_of('e');
    Result = String.substr(0, Length);
    String.erase(0, Length);
    if (Length != string::npos) {
	String.erase(0, 1);
	if (String[1] == '0') {
	    String.erase(1, 1);
	}
	if (String[0] == '+') {
	    String.erase(0, 1);
	}
	Result += "#times10^{" + String + "}";
    }

    ReturnValue = TParaValue(Result);
}
