/* kisc-interactive.cc */
/* Created by Enomoto Sanshiro on 14 December 2001. */
/* Last updated by Enomoto Sanshiro on 2 June 2002. */


#include <iostream>
#include <vector>
#include <cstdlib>
#include "MushConsoleStream.hh"
#include "KiscScript.hh"


using namespace std;


class TKiscInteractive {
  public:
    TKiscInteractive(int argc, char** argv) throw(TScriptException);
    virtual ~TKiscInteractive();
    virtual void Start(void) throw(TScriptException);
  protected:
    virtual void ProcessCommand(void) throw(TScriptException);
    virtual void ProcessScript(void) throw(TScriptException);
    virtual void PrintResult(void) throw(TScriptException);
  protected:
    TKiscScript* _Script;
    TParaTokenTable* _TokenTable;
    TParaSymbolTable* _SymbolTable;
    TParaPackage* _Package;
    TParaStatementParser* _StatementParser;
    TParaTokenizer* _Tokenizer;
    istream* _InputStream;
    TMushConsole* _Console;
    vector<ifstream*> _InputStreamList;
    vector<TParaPackageEntry*> _EntryList;
    enum TBase { Base_dec, Base_hex } _Base;
    int _Count;
    TParaValue _LastResult;
    bool _IsStopRequested;
};



TKiscInteractive::TKiscInteractive(int argc, char** argv) throw(TScriptException)
{
    _Console = new TMushReadlineConsole();
    _Console->SetPrompt("> ");

    _Script = new TKiscScript(argc - 1, argv + 1, _Console);

    _TokenTable = _Script->GetTokenTable();
    _SymbolTable = _Script->GetSymbolTable();
    _Package = _Script->GetPackage();
    _StatementParser = _Script->GetStatementParser();

    _InputStream = new TMushInputConsoleStream(_Console);
    _Tokenizer = new TParaTokenizer(*_InputStream, _TokenTable);

    if (argc > 1) {
	string FileName = argv[1];
	ifstream *InputStream = new ifstream(FileName.c_str());
	if (! *InputStream) {
	    throw TScriptException("unable to open file: " + FileName);
	}
	_InputStreamList.push_back(InputStream);
	_Tokenizer->InputBuffer()->SetChildInput(*InputStream);
    }

    _Count = 1;
    _Base = Base_dec;
    _IsStopRequested = false;
}

TKiscInteractive::~TKiscInteractive()
{
    for (unsigned i = 0; i < _InputStreamList.size(); i++) {
	delete _InputStreamList[i];
    }
    for (unsigned j = 0; j < _EntryList.size(); j++) {
	delete _EntryList[j];
    }

    delete _Script;
    delete _InputStream;
    delete _Tokenizer;
}

void TKiscInteractive::Start(void) throw(TScriptException)
{
    while (! _IsStopRequested && ! _Tokenizer->LookAhead().IsEmpty()) {
	try {
	    if (_Tokenizer->LookAhead().Is(".")) {
		ProcessCommand();
	    }
	    else {
		bool IsResultPrintingEnabled = false;
		if (_Tokenizer->LookAhead().Is("?")) {
		    _Tokenizer->Next();
		    IsResultPrintingEnabled = true;
		}
		ProcessScript();

		if (IsResultPrintingEnabled) {
		    PrintResult();
		}
	    }
	}
	catch (TScriptException &e) {
	    cerr << "ERROR: " << e << endl;
	}
    }
}

void TKiscInteractive::ProcessCommand(void) throw(TScriptException)
{
    _Tokenizer->Next().MustBe(".");
    TParaToken Token = _Tokenizer->Next();

    if (Token.Is("x")) {
	string FileName = _Tokenizer->SkipWhiteSpace().GetLine().AsString();
	ifstream* InputStream = new ifstream(FileName.c_str());
	_InputStreamList.push_back(InputStream);
	if (! *InputStream) {
	    throw TScriptException("unable to open file: " + FileName);
	}
	else {
	    _Tokenizer->InputBuffer()->SetChildInput(*InputStream);
	}
    }
    else if (Token.Is("quit") || Token.Is("q")) {
	_IsStopRequested = true;
    }
    else if (Token.Is("dec")) {
	_Base = Base_dec;
	PrintResult();
    }
    else if (Token.Is("hex")) {
	_Base = Base_hex;
	PrintResult();
    }
    else {
	throw TScriptException("unknown command: " + Token.AsString());
    }
}

void TKiscInteractive::ProcessScript(void) throw(TScriptException)
{
    _Console->SetPrompt("? ");

    TParaPackageEntry* Entry = _Package->CreateEntry(_Tokenizer);
    if (Entry != 0) {
	Entry->Parse(_Tokenizer, _StatementParser, _SymbolTable);
	_EntryList.push_back(Entry);
    }
    else {
	TParaStatement* Statement = 0;
	try {
	    Statement = _StatementParser->Parse(_Tokenizer, _SymbolTable);
	    _LastResult = Statement->Execute(_SymbolTable).ReturnValue;
	}
	catch (TScriptException &e) {
	    delete Statement;	
	    _Console->SetPrompt("> ");
	    throw;
	}
	delete Statement;	
    }

    _Console->SetPrompt("> ");
}

void TKiscInteractive::PrintResult(void) throw(TScriptException)
{
#if 0
    cout << "[" << _Count++ << "] := ";
#endif

    if (_LastResult.IsLong() && (_Base == Base_hex)) {
	cout << hex << "0x" << _LastResult.AsLong() << dec << endl;
    }
    else {
	cout << _LastResult.AsString() << endl;
    }
}



int main(int argc, char** argv)
{
    try {
	TKiscInteractive(argc, argv).Start();
    }
    catch (TScriptException &e) {
	cerr << "ERROR: " << e << endl;
    }
    cout << endl;

    return 0;
}
