/* KmlcFunctionFitter.cc */
/* Created by Enomoto Sanshiro on 21 January 2001 */
/* Last updated by Enomoto Sanshiro on 26 May 2002. */


#include <complex>
#include <cfloat>
#include <cmath>
#include <algorithm>
#include "KumaDefs.hh"
#include "KumaVector.hh"
#include "KumaPrimitive.hh"
#include "KumaFunction.hh"
#include "KmlcFunctionFitter.hh"

using namespace std;
using namespace kuma;


TKmlcChiSquaredFunction::TKmlcChiSquaredFunction(TKumaParameterizedFunction& Function, TKumaPrimitiveVector& XDataList, TKumaPrimitiveVector& YDataList)
: _Function(Function), _XDataList(XDataList), _YDataList(YDataList)
{
}

TKmlcChiSquaredFunction::~TKmlcChiSquaredFunction(void)
{
}

double TKmlcChiSquaredFunction::ValueOf(const double* InputValueList, int InputListSize)
{
    if (_Function.ParameterList().Size() != InputListSize) {
	throw TKmlcException(
	    "TKmlcChiSquaredFunction::ValueOf()",
	    "inconsistent number of parameters"
	);
    }

    for (int i = 0; i < InputListSize; i++) {
	_Function[i] = InputValueList[i];
    }

    double ChiSquared = 0;
    for (int i = 0; i < _XDataList.Size(); i++) {
	double X = _XDataList[i].Value();
	double Y = _YDataList[i].Value();
	double Variance = _YDataList[i].Variance();

	if (_XDataList[i].Error() != 0) {
	    double dx = 1.0e-5 * max(1.0, fabs(X));
	    double dFdx = (_Function(X+dx) - _Function(X-dx)) / (2*dx);
	    Variance += dFdx * dFdx * _XDataList[i].Variance();
	}

	try {
	    double Distance = Y - _Function(X);
	    ChiSquared += Distance * Distance / Variance;
	}
	catch (TKumaException &e) {
	    ChiSquared = FLT_MAX;
	    break;
	}
    }

    return ChiSquared;
}



TKmlcFunctionFitter::TKmlcFunctionFitter(TKmlcFunctionMinimizer* Minimizer)
: _Minimizer(Minimizer)
{    
}

TKmlcFunctionFitter::~TKmlcFunctionFitter(void)
{
}

double TKmlcFunctionFitter::Fit(TKumaParameterizedFunction& Function, TKumaPrimitiveVector& XDataList, TKumaPrimitiveVector& YDataList) throw(TKmlcException)
{
    TKmlcChiSquaredFunction ChiSquaredFunction(Function, XDataList, YDataList);
    TKumaPrimitiveVector& ParameterList = Function.ParameterList();

    double ChiSquared = _Minimizer->Minimize(ChiSquaredFunction, ParameterList);

    return ChiSquared;
}
