/* KorbNetworkAdapter.cc */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Udated by Enomoto Sanshiro on 6 December 2000. */
/* Last updated by Enomoto Sanshiro on 19 January 2010. */


#include <fstream>
#include <sstream>
#include <cstdlib>
#include "KorbNetworkHub.hh"
#include "KorbNetworkAdapter.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbNetworkAdapter", ios::app);
#endif


TKorbNetworkAdapter::TKorbNetworkAdapter(int AdapterId, int MessageQueueProjectId, const string& KorbPath)
{
    _DomainId = TKorbNetworkHub::DomainId_Unknown;
    _AdapterId = AdapterId;

    _MessageQueueProjectId = MessageQueueProjectId;
    _KorbPath = KorbPath;

    _MessageQueue = 0;
}

TKorbNetworkAdapter::~TKorbNetworkAdapter()
{
    delete _MessageQueue;
}

int TKorbNetworkAdapter::DomainId(void) throw(TKorbException)
{
    if (_MessageQueue == 0) {
	Attach();
    }

    return _DomainId;
}

int TKorbNetworkAdapter::AdapterId(void)
{
    return _AdapterId;
}

void TKorbNetworkAdapter::Attach(void) throw(TKorbException)
{
    if (_MessageQueue != 0) {
	return;
    }

    try {
        _MessageQueue = new TMushClientMessageQueue(
	    _MessageQueueProjectId, _KorbPath
	);
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
            "TKorbNetworkAdapter::Attach()",
	    "system call exception: " + e.Message()
	);
    }

    int MessageQueueAddress = _AdapterId;
    _MessageQueue->SetAddress(MessageQueueAddress);

    InquireDomainId();
}

void TKorbNetworkAdapter::InquireDomainId(void) throw(TKorbException)
{
    if (_DomainId != TKorbNetworkHub::DomainId_Unknown) {
	return;
    }

    TKorbNetworkPacket Packet;
    Packet.SetPacketType(TKorbNetworkPacket::ptEchoRequest);
    Packet.SetSender(_DomainId, _AdapterId);
    Packet.SetReceiver(_DomainId, 0);

    TMushMessageQueuePacket* MessageQueuePacket
        = (TMushMessageQueuePacket*) Packet.ContentsAddress();
    MessageQueuePacket->MessageType = TKorbNetworkHub::MessageQueueAddress;
    int MessageLength = Packet.ContentsSize();
    int MaxMessageLength = Packet.ContentsCapacity() - sizeof(long);

    try {
        _MessageQueue->Send(MessageQueuePacket, MessageLength);

	while (1) {
	    _MessageQueue->Receive(MessageQueuePacket, MaxMessageLength);
	    if (Packet.PacketType() == TKorbNetworkPacket::ptEchoReply) {
		break;
	    }
	    else if (Packet.PacketType() == TKorbNetworkPacket::ptMessage) {
		_MessagePacketQueue.push_back(Packet);
	    }
	    else {
		ostringstream os;
		os << "invalid packet type id:" << endl;
		Packet.Dump(os);
		
		cerr << "TKorbNetworkAdapter::InquireDomainId():" << os.str() << endl;
		abort();
		throw TKorbException(
		    "TKorbNetworkAdapter::InquireDomainId()", os.str()
		);
	    }
	}
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkAdapter::InquireDomainId()",
	    "system call exception: " + e.Message()
        );
    }
    
    _DomainId = Packet.SenderDomainId();
}

int TKorbNetworkAdapter::Send(TKorbNetworkPacket& Packet, int Flag) throw(TKorbException)
{
    if (_MessageQueue == 0) {
        Attach();
    }

    Packet.SetPacketType(TKorbNetworkPacket::ptMessage);

    TMushMessageQueuePacket* MessageQueuePacket
        = (TMushMessageQueuePacket*) Packet.ContentsAddress();
    int MessageLength = Packet.ContentsSize();

    if (
	(_DomainId != TKorbNetworkHub::DomainId_Unknown) &&
	(Packet.ReceiverDomainId() == _DomainId) 
    ){
	MessageQueuePacket->MessageType = Packet.ReceiverAdapterId();
    }
    else {
	MessageQueuePacket->MessageType = TKorbNetworkHub::MessageQueueAddress;
    }

    bool IsNonBlocking = ((Flag & (int) Flag_NonBlocking) != 0);
    int SentLength;
    try {
	SentLength = _MessageQueue->Send(
	    MessageQueuePacket, MessageLength, IsNonBlocking
	);
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkAdapter::Send()",
	    "system call exception: " + e.Message()
        );
    }

    if (IsNonBlocking && (SentLength == 0)) {
	return 0;
    }
    else if (SentLength != MessageLength) {
        throw TKorbException(
	    "TKorbNetworkAdapter::Send()",
	    "system call exception: unable to send message"
        );
    }

#ifdef DEBUG
    dumpfile << "Send" << endl;
    Packet.Dump(dumpfile);
#endif

    return SentLength;
}

int TKorbNetworkAdapter::Receive(TKorbNetworkPacket& Packet, int Flag) throw(TKorbException)
{
    if (_MessageQueue == 0) {
        Attach();
    }

    if (! _MessagePacketQueue.empty()) {
	Packet = _MessagePacketQueue.front();
	_MessagePacketQueue.pop_front();
	return Packet.DataSize();
    }

    TMushMessageQueuePacket* MessageQueuePacket
        = (TMushMessageQueuePacket*) Packet.ContentsAddress();
    int MaxMessageLength = Packet.ContentsCapacity() - sizeof(long);

    int ReceivedLength;
    try {
	bool IsNonBlocking = ((Flag & (int) Flag_NonBlocking) != 0);
        ReceivedLength = _MessageQueue->Receive(
	    MessageQueuePacket, MaxMessageLength, IsNonBlocking
	);
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkAdapter::Receive()",
	    "system call exception: " + e.Message()
        );
    }

    if (ReceivedLength > 0) {
	if (Packet.PacketType() != TKorbNetworkPacket::ptMessage) {
	    ostringstream os;
	    os << "invalid packet type id:" << endl;
	    Packet.Dump(os);

	    cerr << "TKorbNetworkAdapter::Receive():" << os.str() << endl;
	    abort();
	    throw TKorbException(
		"TKorbNetworkAdapter::Receive()", os.str()
	    );
	}

#ifdef DEBUG
        dumpfile << "Receive" << endl;
        Packet.Dump(dumpfile);
#endif
    }

    return ReceivedLength;
}

int TKorbNetworkAdapter::QueueCapacity(void) throw(TKorbException)
{
    try {
	if (_MessageQueue == 0) {
	    Attach();
	}

	return _MessageQueue->QueueCapacity();
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkAdapter::QueueCapacity()", 
	    "system call exception: " + e.Message()
	);
    }
}

int TKorbNetworkAdapter::NumberOfPendingPackets(void) throw(TKorbException)
{
    try {
        if (_MessageQueue == 0) {
	    Attach();
	}

	return _MessageQueue->NumberOfPendingPackets();
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkAdapter::NumberOfPendingPackets()",
	    "System Call Exception: " + e.Message()
        );
    }
}
