/* KumaPrimitive.hh */
/* Created by Enomoto Sanshiro on 11 October 2003. */
/* Last updated by Enomoto Sanshiro on 23 August 2004. */


#ifndef __KumaPrimitive_hh__
#define __KumaPrimitive_hh__

#include <string>
#include <complex>
#include "KumaVector.hh"


namespace kuma {


template<class T = double> class TKumaPrimitiveBase {
  public:
    inline TKumaPrimitiveBase(void);
    inline explicit TKumaPrimitiveBase(const T& Value);
    inline TKumaPrimitiveBase(const T& Value, const T& Error);
    inline TKumaPrimitiveBase(const TKumaPrimitiveBase<T>& Primitive);
    inline ~TKumaPrimitiveBase();
    inline TKumaPrimitiveBase<T>& operator=(const TKumaPrimitiveBase<T>& Primitive);
    inline TKumaPrimitiveBase<T>& operator=(const T& Value);
    inline operator T() const;
    inline T& Value(void);
    inline T& Error(void);
    inline std::string& Name(void);
    inline const T& Value(void) const;
    inline const T& Error(void) const;
    inline const std::string& Name(void) const;
    inline T Variance(void) const;
  protected:
    T _Value, _Error;
    std::string _Name;
};


template<int Dimension, class T = double> class TKumaElementBase {
  public:
    inline TKumaPrimitiveBase<T>& operator[](int Index);
    inline const TKumaPrimitiveBase<T>& operator[](int Index) const;
  protected:
    TKumaPrimitiveBase<T> _PrimitiveArray[Dimension];
};


typedef TKumaPrimitiveBase<double> TKumaPrimitive;
typedef TKumaPrimitiveBase<std::complex<double> > TKumaComplexPrimitive;

typedef TKumaElementBase<2, double> TKuma2dElement;
typedef TKumaElementBase<2, std::complex<double> > TKuma2dComplexElement;

typedef TKumaElementBase<3, double> TKuma3dElement;
typedef TKumaElementBase<3, std::complex<double> > TKuma3dComplexElement;

typedef TKumaVectorBase<TKumaPrimitive> TKumaPrimitiveVector;


template<class T> inline TKumaPrimitiveBase<T>::TKumaPrimitiveBase(void)
{
    _Value = 0;
    _Error = 0;
}

template<class T> inline TKumaPrimitiveBase<T>::TKumaPrimitiveBase(const T& Value)
{
    _Value = Value;
    _Error = 0;
}

template<class T> inline TKumaPrimitiveBase<T>::TKumaPrimitiveBase(const T& Value, const T& Error)
{
    _Value = Value;
    _Error = Error;
}

template<class T> inline TKumaPrimitiveBase<T>::TKumaPrimitiveBase(const TKumaPrimitiveBase<T>& Primitive)
{
    _Value = Primitive._Value;
    _Error = Primitive._Error;
    _Name = Primitive._Name;
}

template<class T> inline TKumaPrimitiveBase<T>& TKumaPrimitiveBase<T>::operator=(const TKumaPrimitiveBase<T>& Primitive)
{
    _Value = Primitive._Value;
    _Error = Primitive._Error;
    _Name = Primitive._Name;

    return *this;
}

template<class T> inline TKumaPrimitiveBase<T>& TKumaPrimitiveBase<T>::operator=(const T& Value)
{
    _Value = Value;
    return *this;
}

template<class T> inline TKumaPrimitiveBase<T>::operator T() const
{
    return _Value;
}

template<class T> inline TKumaPrimitiveBase<T>::~TKumaPrimitiveBase()
{
}

template<class T> inline T& TKumaPrimitiveBase<T>::Value(void)
{
    return _Value;
}

template<class T> inline T& TKumaPrimitiveBase<T>::Error(void)
{
    return _Error;
}

template<class T> inline std::string& TKumaPrimitiveBase<T>::Name(void)
{
    return _Name;
}

template<class T> inline const T& TKumaPrimitiveBase<T>::Value(void) const
{
    return _Value;
}

template<class T> inline const T& TKumaPrimitiveBase<T>::Error(void) const
{
    return _Error;
}

template<class T> inline T TKumaPrimitiveBase<T>::Variance(void) const
{
    return _Error * _Error;
}

template<class T> inline const std::string& TKumaPrimitiveBase<T>::Name(void) const
{
    return _Name;
}


template<int Dimension, class T> inline TKumaPrimitiveBase<T>& TKumaElementBase<Dimension, T>::operator[](int Index) 
{ 
    return _PrimitiveArray[Index]; 
}

template<int Dimension, class T> inline const TKumaPrimitiveBase<T>& TKumaElementBase<Dimension, T>::operator[](int Index) const
{ 
    return _PrimitiveArray[Index]; 
}


}

#endif
