/* KumaSignalProcessing.cc */
/* Created by Enomoto Sanshiro on 10 September 2004. */
/* Last updated by Enomoto Sanshiro on 12 April 2006. */


#include <complex>
#include <cmath>
#include "KumaDefs.hh"
#include "KumaFourierTransform.hh"
#include "KumaDigitalFilter.hh"
#include "KumaSignalProcessing.hh"


using namespace std;
using namespace kuma;



TKumaPowerSpectrum::TKumaPowerSpectrum(void)
{
}

TKumaPowerSpectrum::~TKumaPowerSpectrum()
{
}

double TKumaPowerSpectrum::ValueOf(const double* InputValueList, int InputListSize)
{
    if (InputListSize == 1) {
	return Spectrum(InputValueList[0]);
    }
    else if (InputListSize == 2) {
	return Spectrum(complex<double>(InputValueList[0], InputValueList[1]));
    }

    return 0;
}



TKumaFourierTransformPowerSpectrum::TKumaFourierTransformPowerSpectrum(void)
{
    _DataListSize = 0;
    _FourierSpectrum = 0;
}

TKumaFourierTransformPowerSpectrum::~TKumaFourierTransformPowerSpectrum()
{
    delete[] _FourierSpectrum;
}

void TKumaFourierTransformPowerSpectrum::ReadData(double* DataList, int ListSize) throw(TKumaException)
{
    int m = 2;
    while ((0x0001 << m) < ListSize) {
	m++;
    }
    int n = 0x0001 << m;

    if (_DataListSize < n) {
	delete[] _FourierSpectrum;
	_FourierSpectrum = new complex<double>[n];
    }
    _DataListSize = n;

    int i;
    for (i = 0; i < ListSize; i++) {
	_FourierSpectrum[i] = DataList[i];
    }
    for (; i < n; i++) {
	_FourierSpectrum[i] = 0;
    }

    TKumaFastFourierTransformer().Transform(_FourierSpectrum, _DataListSize);
}

double TKumaFourierTransformPowerSpectrum::Spectrum(double Frequency) throw(TKumaException)
{
    if (Frequency < 0) {
	Frequency += (int) fabs(Frequency) + 1;
    }
    int Index = ((int) (Frequency * _DataListSize + 0.5)) % _DataListSize;

    // normalized for the interval [-0.5, 0.5]
    return sqr(abs(_FourierSpectrum[Index])) / _DataListSize;
}

double TKumaFourierTransformPowerSpectrum::Spectrum(const std::complex<double>& Frequency) throw(TKumaException)
{
    //...
    return Spectrum(real(Frequency));
}



TKumaMaximumEntropyPowerSpectrum::TKumaMaximumEntropyPowerSpectrum(int Order)
{
    _Order = Order;
    _CoefficientList = new long double[_Order+1];
}

TKumaMaximumEntropyPowerSpectrum::~TKumaMaximumEntropyPowerSpectrum()
{
    delete[] _CoefficientList;
}

void TKumaMaximumEntropyPowerSpectrum::ReadData(double* DataList, int ListSize) throw(TKumaException)
{
    const int& N = ListSize;
    const int& M = _Order;

    _CoefficientList[0] = 0;
    for (int j = 0; j < N; j++) {
	_CoefficientList[0] += sqr(DataList[j]);
    }
    _CoefficientList[0] /= N;

    double* Work1 = new double[N];
    double* Work2 = new double[N];
    Work1[0] = DataList[0];
    Work2[N-2] = DataList[N-1];
    for (int j = 1; j < N-1; j++) {
	Work1[j] = DataList[j];
	Work2[j-1] = DataList[j];
    }
    Work1[N-1] = 0;
    Work2[N-1] = 0;

    double* WorkM = new double[M+1];
    for (int i = 0; i <= M; i++) {
	WorkM[i] = 0;
    }

    for (int k = 1; k <= M; k++) {
	double Numerator = 0, Denominator = 0;
	for (int j = 0; j < N-k; j++) {
	    Numerator += Work1[j] * Work2[j];
	    Denominator += sqr(Work1[j]) + sqr(Work2[j]);
	}
	_CoefficientList[k] = 2.0 * Numerator / Denominator;
	_CoefficientList[0] *= (1.0 - sqr(_CoefficientList[k]));

	for (int i = 1; i <= k-1; i++) {
	    _CoefficientList[i] = WorkM[i] - _CoefficientList[k] * WorkM[k-i];
	}
	if (k == M) {
	    goto END;
	}
	for (int i = 1; i <= k; i++) {
	    WorkM[i] = _CoefficientList[i];
	}
	for (int j = 0; j < N-k-1; j++) {
	    Work1[j] = Work1[j] - WorkM[k] * Work2[j];
	    Work2[j] = Work2[j+1] - WorkM[k] * Work1[j+1];
	}
    }

  END:
    delete[] Work1;
    delete[] Work2;
    delete[] WorkM;
}

double TKumaMaximumEntropyPowerSpectrum::Spectrum(double Frequency) throw(TKumaException)
{
    return Spectrum(complex<double>(Frequency));
}

double TKumaMaximumEntropyPowerSpectrum::Spectrum(const std::complex<double>& Frequency) throw(TKumaException)
{
    complex<long double> Z = exp(2*M_PI*Frequency * complex<double>(0, 1));

    complex<long double> Sum = 0;
    complex<long double> Zk = 1.0;
    for (int k = 1; k <= _Order; k++) {
	Zk *= Z;
	Sum += _CoefficientList[k] * Zk;
    }

    return _CoefficientList[0] / sqr(abs(1.0L - Sum));  // ???
}

const long double* TKumaMaximumEntropyPowerSpectrum::CoefficientList(void) const
{
    return _CoefficientList;
}


class TKumaTrendFunction: public TKumaFunction {
  public:
    TKumaTrendFunction(double X0, double Y0, double X1, double Y1)
	: _X0(X0), _Y0(Y0), _X1(X1), _Y1(Y1) {}
    virtual ~TKumaTrendFunction() {}
    virtual double ValueOf(const double* InputValueList, int InputListSize) {
	double R = (InputValueList[0] - _X0) / (_X1 - _X0);
	return _Y0 + R * (_Y1 - _Y0);
    }
  protected:
    double _X0, _Y0, _X1, _Y1;
};


class TKumaZeroFunction: public TKumaFunction {
  public:
    TKumaZeroFunction(void) {}
    virtual ~TKumaZeroFunction() {}
    virtual double ValueOf(const double* InputValueList, int InputListSize) {
	return 0;
    }
};


TKumaShannonInterpolatedFunction::TKumaShannonInterpolatedFunction(void)
{
    _StartTime = 0;
    _SampleInterval = 1;

    _SampleList = 0;
    _ListSize = 0;

    _SampleListBuffer = 0;

    _TrendFunction = new TKumaZeroFunction();
}

TKumaShannonInterpolatedFunction::~TKumaShannonInterpolatedFunction(void)
{
    delete[] _SampleListBuffer;
    delete _TrendFunction;
}

void TKumaShannonInterpolatedFunction::SetData(const double* SampleList, int ListSize)
{
    if (ListSize < 2) {
	throw TKumaException(
	    "TKumaShannonInterpolatedFunction::ReadData()",
	    "too few data points (must be more than 1)"
	);
    }

    _SampleList = SampleList;
    _ListSize = ListSize;
}

void TKumaShannonInterpolatedFunction::ReadData(const double* SampleList, int ListSize)
{
    delete[] _SampleListBuffer;
    _SampleListBuffer = new double[ListSize];
    
    memcpy(_SampleListBuffer, SampleList, sizeof(double) * ListSize);

    SetData(_SampleListBuffer, ListSize);
}

void TKumaShannonInterpolatedFunction::SetTimeLabel(double StartTime, double SampleInterval)
{
    _StartTime = StartTime;
    _SampleInterval = SampleInterval;
}

double TKumaShannonInterpolatedFunction::ValueOf(const double* InputValueList, int InputListSize)
{
    double T = (InputValueList[0] - _StartTime) / _SampleInterval;
    
    int To = (int) round(T);
    if (To < 0) {
	To = 0;
    }
    if (To >= _ListSize) {
	To = _ListSize-1;
    }

    double Value = 0;
    for (int i = _ListSize; i > 0; i--) {
	int j = To - i;
	int k = To + i;
	if (j >= 0) {
	    Value += (_SampleList[j] - (*_TrendFunction)(j)) * sinc((T-j));
	}
	if (k < _ListSize) {
	    Value += (_SampleList[k] - (*_TrendFunction)(k)) * sinc((T-k));
	}
    }
    Value += (_SampleList[To] - (*_TrendFunction)(To)) * sinc((T-To));

    return Value + (*_TrendFunction)(T);
}

void TKumaShannonInterpolatedFunction::RemoveTrend(void)
{
    delete _TrendFunction;
    _TrendFunction = new TKumaTrendFunction(
	0, _SampleList[0], _ListSize, _SampleList[_ListSize-1]
    );
}
