/* KumaVector.hh */
/* Created by Enomoto Sanshiro on 18 January 2001 */
/* Last updated by Enomoto Sanshiro on 18 January 2001. */


#ifndef __KumaVector_hh__
#define __KumaVector_hh__


#include <complex>

namespace kuma {


template<class T = double> class TKumaVectorBase {
  public:
    inline TKumaVectorBase(int Size = 0, int InitialCapacity = 32);
    inline TKumaVectorBase(const TKumaVectorBase<T>& Vector);
    inline ~TKumaVectorBase();
    inline TKumaVectorBase<T>& operator= (const TKumaVectorBase<T>& Vector);
    inline int Add(const T& Value);
    inline int Size(void) const;
    inline void Clear(void);
    inline T* RawArray(void);
    inline const T* RawArray(void) const;
  public:
    inline T& operator[] (int Index);
    inline const T& operator[] (int Index) const;
    inline operator T*();
    inline operator const T*() const;
  protected:
    int _Capacity;
    int _Size;
    T* _RawArray;
};


typedef TKumaVectorBase<double> TKumaVector;
typedef TKumaVectorBase<std::complex<double> > TKumaComplexVector;


template<class T> inline TKumaVectorBase<T>::TKumaVectorBase(int Size, int InitialCapacity)
{
    _Size = Size;
    _Capacity = InitialCapacity;
    if (_Size > _Capacity) {
	_Capacity = _Size;
    }

    _RawArray = new T[_Capacity];
}

template<class T> inline TKumaVectorBase<T>::TKumaVectorBase(const TKumaVectorBase<T>& Vector)
{
    _Capacity = Vector._Size;
    _Size = Vector._Size;
    _RawArray = new T[_Capacity];

    for (unsigned i = 0; i < _Size; i++) {
	_RawArray[i] = Vector._RawArray[i];
    }
}

template<class T> inline TKumaVectorBase<T>::~TKumaVectorBase()
{
    delete[] _RawArray;
}

template<class T> inline TKumaVectorBase<T>& TKumaVectorBase<T>::operator= (const TKumaVectorBase<T>& Vector)
{
    if (_Capacity < Vector._Size) {
	delete [] _RawArray;
	_Capacity = Vector._Size;
	_RawArray = new T[_Capacity];
    }
    _Size = Vector._Size;

    for (int i = 0; i < _Size; i++) {
	_RawArray[i] = Vector._RawArray[i];
    }

    return *this;
}

template<class T> inline int TKumaVectorBase<T>::Add(const T& Value)
{
    if (_Size + 1 > _Capacity) {
	T* OldArray = _RawArray;
	_Capacity *= 2;
	_RawArray = new T[_Capacity];
	for (int i = 0; i < _Size; i++) {
	    _RawArray[i] = OldArray[i];
	}
	delete[] OldArray;
    }

    _RawArray[_Size++] = Value;

    return _Size - 1;
}

template<class T> inline int TKumaVectorBase<T>::Size(void) const
{
    return _Size;
}

template<class T> inline void TKumaVectorBase<T>::Clear(void)
{
    _Size = 0;
}

template<class T> inline T* TKumaVectorBase<T>::RawArray(void)
{
    return _RawArray;
}

template<class T> inline const T* TKumaVectorBase<T>::RawArray(void) const
{
    return _RawArray;
}

template<class T> inline T& TKumaVectorBase<T>::operator[] (int Index)
{
    return _RawArray[Index];
}

template<class T> inline const T& TKumaVectorBase<T>::operator[] (int Index) const
{
    return _RawArray[Index];
}

template<class T> inline TKumaVectorBase<T>::operator T*()
{
    return _RawArray;
}

template<class T> inline TKumaVectorBase<T>::operator const T*() const
{
    return _RawArray;
}

}

#endif
