/* MushArgumentList.cc */
/* Created by Enomoto Sanshiro on 8 July 2000. */
/* Last updated by Enomoto Sanshiro on 8 July 2000. */


#include <iostream>
#include <sstream>
#include <string>
#include <deque>
#include <map>
#include <set>
#include <cctype>
#include "MushDefs.hh"
#include "MushArgumentList.hh"


using namespace std;


TMushArgumentList::TMushArgumentList(int argc, char** argv)
{
    _argc = argc;
    _argv = argv;

    _IsParsed = false;
}

TMushArgumentList::~TMushArgumentList()
{
}

const string& TMushArgumentList::ProgramName(void) const
{
    if (! _IsParsed) {
	Parse();
    }

    return _ProgramName;
}

const deque<string>& TMushArgumentList::ParameterList(void) const
{
    if (! _IsParsed) {
	Parse();
    }

    return _ParameterList;
}

const set<char>& TMushArgumentList::SwitchSet(void) const
{
    if (! _IsParsed) {
	Parse();
    }

    return _SwitchSet;
}

const map<string, string>& TMushArgumentList::OptionTable(void) const
{
    if (! _IsParsed) {
	Parse();
    }

    return _OptionTable;
}

int TMushArgumentList::NumberOfParameters(void) const
{
    if (! _IsParsed) {
	Parse();
    }

    return _ParameterList.size();
}

void TMushArgumentList::ExpectParameters(int NumberOfExpectedParameters) const throw(TSystemCallException)
{
    if (NumberOfParameters() < NumberOfExpectedParameters) {
	throw TSystemCallException("too few argument[s]", "");
    }
}

string TMushArgumentList::ParameterOf(int Index) const
{
    if (! _IsParsed) {
	Parse();
    }

    if (Index < (int) _ParameterList.size()) {
        return _ParameterList[Index];
    }
    else {
        return "";
    }
}

string TMushArgumentList::operator[] (int Index) const
{
    return ParameterOf(Index);
}

//... The following method should be implemented as a template method
long TMushArgumentList::IntParameterOf(int Index) const throw(TSystemCallException)
{
    if (! _IsParsed) {
	Parse();
    }
    
    string StringValue = ParameterOf(Index);

    long LongValue;
    if (
	(StringValue.size() > 2) && 
	(StringValue[0] == '0') && (StringValue[1] == 'x')
    ){
	istringstream ValueStream(StringValue.substr(2, string::npos));
	if (! (ValueStream >> hex >> LongValue)) {
	    throw TSystemCallException(
		"invalid parameter: " + StringValue, "integer is expected"
	    );
	}
	string Remaining;
	getline(ValueStream, Remaining);
	if (! Remaining.empty()) {
	    throw TSystemCallException("invalid parameter: " + StringValue, "");
	}
    }
    else {
	istringstream ValueStream(StringValue);
	if (! (ValueStream >> LongValue)) {
	    throw TSystemCallException(
		"invalid parameter: " + StringValue, "integer is expected"
	    );
	}
	string Remaining;
	getline(ValueStream, Remaining);
	if (! Remaining.empty()) {
	    throw TSystemCallException("invalid parameter: " + StringValue, "");
	}
    }

    return LongValue;
}

double TMushArgumentList::FloatParameterOf(int Index) const throw(TSystemCallException)
{
    if (! _IsParsed) {
	Parse();
    }
    
    string StringValue = ParameterOf(Index);
    istringstream ValueStream(StringValue);
    
    double DoubleValue;
    if (! (ValueStream >> DoubleValue)) {
	throw TSystemCallException(
	    "invalid parameter: " + StringValue, "floating number is expected"
	);
    }

    string Remaining;
    getline(ValueStream, Remaining);
    if (! Remaining.empty()) {
	throw TSystemCallException("invalid parameter: " + StringValue, "");
    }

    return DoubleValue;
}

string TMushArgumentList::RemoveParameter(int Index, int Count)
{
    string Parameter = ParameterOf(Index);
    if (! Parameter.empty()) {
	deque<string>::iterator Begin = _ParameterList.begin() + Index;
	deque<string>::iterator End = _ParameterList.end();
	if (Index + Count < (int) _ParameterList.size()) {
	    End = _ParameterList.begin() + Count;
        }

	_ParameterList.erase(Begin, End);
    }
    
    return Parameter;
}

void TMushArgumentList::InsertParameter(const string& Parameter, int Index)
{
    deque<string>::iterator Position = _ParameterList.end();
    if ((Index >= 0) && (Index < (int) _ParameterList.size())) {
	Position = _ParameterList.begin() + Index;
    }

    _ParameterList.insert(Position, Parameter);
}

string TMushArgumentList::RemoveOption(const string& Name)
{
    string Value;

    map<string, string>::iterator Position;
    Position = _OptionTable.find(Name);
    if (Position != _OptionTable.end()) {
	Value = (*Position).second;
	_OptionTable.erase(Position);
    }

    return Value;
}

void TMushArgumentList::AddOption(const string& Name, const string& Value)
{
    _OptionTable[Name] = Value;
}

bool TMushArgumentList::IsSwitchSpecified(char SwitchCharacter) const
{
    return _SwitchSet.count(SwitchCharacter) > 0;
}

bool TMushArgumentList::IsOptionSpecified(const string& OptionName, char EquivalentSwitchCharacter) const
{
    if (! _IsParsed) {
	Parse();
    }

    bool Result = (_OptionTable.count(OptionName) > 0);
    if (! Result) {
	return IsSwitchSpecified(EquivalentSwitchCharacter);
    }

    return Result;
}

string TMushArgumentList::OptionValueOf(const string& Name, const string& DefaultValue) const
{
    if (! _IsParsed) {
	Parse();
    }

    if (_OptionTable.count(Name) > 0) {
        return _OptionTable[Name];
    } 
    else {
        return DefaultValue;
    }
}

string TMushArgumentList::operator[] (const string& Name) const
{
    return OptionValueOf(Name);
}

//... The following method should be implemented as a template method
long TMushArgumentList::IntOptionValueOf(const string& Name) const throw(TSystemCallException)
{
    if (! _IsParsed) {
	Parse();
    }

    string StringValue = OptionValueOf(Name);

    long LongValue;
    if (
	(StringValue.size() > 2) && 
	(StringValue[0] == '0') && (StringValue[1] == 'x')
    ){
	istringstream ValueStream(StringValue.substr(2, string::npos));
	if (! (ValueStream >> hex >> LongValue)) {
	    throw TSystemCallException(
		"invalid option value: " + Name, "integer is expected"
	    );
	}
	string Remaining;
	getline(ValueStream, Remaining);
	if (! Remaining.empty()) {
	    throw TSystemCallException("invalid parameter: " + StringValue, "");
	}
    }
    else {
	istringstream ValueStream(StringValue);
	if (! (ValueStream >> LongValue)) {
	    throw TSystemCallException(
		"invalid option value: " + Name, "integer is expected");
	}
	string Remaining;
	getline(ValueStream, Remaining);
	if (! Remaining.empty()) {
	    throw TSystemCallException("invalid parameter: " + StringValue, "");
	}
    }

    return LongValue;
}

double TMushArgumentList::FloatOptionValueOf(const string& Name) const throw(TSystemCallException)
{
    if (! _IsParsed) {
	Parse();
    }

    string StringValue = OptionValueOf(Name);
    istringstream ValueStream(StringValue);

    double DoubleValue;
    if (! (ValueStream >> DoubleValue)) {
	throw TSystemCallException(
	    "invalid option value: " + Name, "floating number is expected"
	);
    }

    string Remaining;
    getline(ValueStream, Remaining);
    if (! Remaining.empty()) {
	throw TSystemCallException("invalid parameter: " + StringValue, "");
    }

    return DoubleValue;
}

//... The following method should be implemented as a template method
long TMushArgumentList::IntOptionValueOf(const string& Name, long DefaultValue) const throw(TSystemCallException)
{
    if (! IsOptionSpecified(Name)) {
	return DefaultValue;
    }

    return IntOptionValueOf(Name);
}

double TMushArgumentList::FloatOptionValueOf(const string& Name, double DefaultValue) const throw(TSystemCallException)
{
    if (! IsOptionSpecified(Name)) {
	return DefaultValue;
    }

    return FloatOptionValueOf(Name);
}

void TMushArgumentList::Parse(void) const
{
    if (_argc > 0) {
	_ProgramName = _argv[0];
    }

    for (int i = 1; i < _argc; i++) {
	string Argument = _argv[i];
	if ((Argument[0] != '-') || (Argument == "-") || (Argument == "--")) {
	    _ParameterList.push_back(Argument);
	}
	else if ((Argument.size() > 1) && (isdigit(Argument[1]))) {
	    // negative number parameter //
	    _ParameterList.push_back(Argument);
	}
	else {
	    size_t NameLength = Argument.find_first_of('=');
	    string Name = Argument.substr(0, NameLength);
	    string Value = "";

	    if (NameLength != string::npos) {
		Value = Argument.substr(NameLength + 1, Argument.size());
	    }
	    
	    if (Value.empty() && (i+1 < _argc)) {
		if (NameLength != string::npos) {
		    // already contains the assign operator// 
		    i++;
		    Value = _argv[i];
		}
		else if (_argv[i+1][0] == '=') {
		    i++;
		    if (_argv[i][1] != '\0') {
			Value = (_argv[i] + 1);
		    }
		    else if (i+1 < _argc) {
			i++;
			Value = _argv[i];
		    }
		}
	    }

	    _OptionTable[Name] = Value;

	    if ((Argument.size() > 1) && (Argument[1] != '-')) {
		for (unsigned j = 1; j < Argument.size(); j++) {
		    _SwitchSet.insert(Argument[j]);
		}
	    }
	}
    }

    _IsParsed = true;
}

void TMushArgumentList::Dump(ostream& os) const
{
    if (! _IsParsed) {
	Parse();
    }

    os << "ProgramName: " << _ProgramName << endl;

    os << "Parameters:" << endl;
    for (unsigned i = 0; i < _ParameterList.size(); i++) {
	os << "    " << _ParameterList[i] << endl;
    }
    
    os << "Options:" << endl;
    map<string, string, less<string> >::iterator OptionIterator;
    for (
	OptionIterator = _OptionTable.begin();
	OptionIterator != _OptionTable.end();
	OptionIterator++
    ){
	os << "    " << (*OptionIterator).first;
	os << ": " << (*OptionIterator).second << endl;
    }

    os << "Switchs:" << endl;
    set<char>::iterator SwitchIterator;
    for (
	SwitchIterator = _SwitchSet.begin();
	SwitchIterator != _SwitchSet.end();
	SwitchIterator++
    ){
	os << "    " << *SwitchIterator << endl;
    }
}
