/* MushChainedFile.cc */
/* Created by Enomoto Sanshiro on 18 November 2000. */
/* Last updated by Enomoto Sanshiro on 30 April 2002. */


#include <iostream>
#include <sstream>
#include <string>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushChainedFile.hh"


using namespace std;
using namespace mush;


TMushChainedFile::TMushChainedFile(const string& FileName, unsigned MaxFileSize, unsigned MinimumTailPacketSize)
: TMushFile(FileName)
{
    _MaxFileSize = MaxFileSize;
    _MinimumTailPacketSize = MinimumTailPacketSize;

    _ThisFileIndex = 0;
    _ThisFileHeadOffset = 0;
    _PositionInThisFile = 0;
    _LastPacketSize = 0;
}

TMushChainedFile::~TMushChainedFile()
{
}

void TMushChainedFile::OpenReadMode(void) throw(TSystemCallException)
{
    if (IsOpened()) {
	return;
    }

    TMushFile::OpenReadMode();
    _IsReadMode = true;
}

void TMushChainedFile::OpenWriteMode(unsigned AccessMode, bool IsExclusive) throw(TSystemCallException)
{
    if (IsOpened()) {
	return;
    }

    if (TMushFileAttribute(_Name).IsWritable()) {
	try {
	    Unlink();
	}
	catch (TSystemCallException &e) {
	    /* The file might be /dev/null */
	}
    }

    TMushFile::OpenWriteMode(AccessMode, IsExclusive);
    _IsReadMode = false;
}

void TMushChainedFile::OpenAppendMode(unsigned AccessMode) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushChainedFile::Open()", 
	"opening in 'append' mode is not implemented yet"
    ); 
}

void TMushChainedFile::OpenReadWriteMode(unsigned AccessMode) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushChainedFile::Open()", 
	"opening in 'read-write' mode is not implemented yet"
    ); 
}

void TMushChainedFile::Open(int OpenFlag, unsigned AccessMode) throw(TSystemCallException)
{
    TMushFile::Open(OpenFlag, AccessMode);

    _OpenFlag = OpenFlag;
    _AccessMode = AccessMode;

    _ThisFileIndex = 0;
    _ThisFileHeadOffset = 0;
    _PositionInThisFile = 0;
    _LastPacketSize = 0;
}

int TMushChainedFile::Write(void *Buffer, size_t Length) throw(TSystemCallException)
{
    if (
	(_PositionInThisFile + Length > _MaxFileSize) &&
	(_LastPacketSize >= _MinimumTailPacketSize)
    ){
	if (MoveToNext() < 0) {
	    throw TSystemCallException(
		"TMushChainedFile::Write()", "unable to open new file"
	    ); 
	}
    }

    int WrittenLength = TMushFile::Write(Buffer, Length);
    _PositionInThisFile += WrittenLength;
    _LastPacketSize = WrittenLength;

    return WrittenLength;
}

int TMushChainedFile::Read(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int ReadLength;
    ReadLength = TMushFile::Read(Buffer, Length);
    if (ReadLength == 0) {
	if (MoveToNext() >= 0) {
	    ReadLength = TMushFile::Read(Buffer, Length);
	}
	else {
	    // really end of file[s].
	    ReadLength = 0;
	}
    }

    _PositionInThisFile += ReadLength;

    return ReadLength;
}

Int64 TMushChainedFile::Position(void) throw(TSystemCallException)
{
    return _ThisFileHeadOffset + _PositionInThisFile;
}

void TMushChainedFile::SeekTo(Int64 Offset) throw(TSystemCallException)
{
    if (Offset < _ThisFileHeadOffset) {
	if (Offset > _ThisFileHeadOffset / 2) {
	    // find file backward from the current file
	    if (MoveToPrevious() < 0) {
		throw TSystemCallException(
		    "TMushChainedFile::SeekTo()", 
		    "unable to find file: " + FileNameOf(_ThisFileIndex - 1)
		); 
	    }		
	}
	else if (Offset > 0) {
	    // find file foreward from the chain head
	    if (MoveToFirst() < 0) {
		throw TSystemCallException(
		    "TMushChainedFile::SeekTo()", 
		    "unable to find file: " + FileNameOf(0)
		); 
	    }		
	}
	else {
	    throw TSystemCallException(
		"TMushChainedFile::SeekTo()", "bad file offset"
	    ); 
	}
	
	SeekTo(Offset);
    }
    else if (Offset < _ThisFileHeadOffset + FileSizeOf(_ThisFileIndex)) {
	// Offset is in this file
	_PositionInThisFile = Offset - _ThisFileHeadOffset;
	TMushFile::SeekTo(_PositionInThisFile);
    }
    else {
	if (! _IsReadMode) {
	    _PositionInThisFile = Offset - _ThisFileHeadOffset;
	    TMushFile::SeekTo(_PositionInThisFile);
	}
	else {
	    // find file foreward from the current file
	    if (MoveToNext() < 0) {
		throw TSystemCallException(
		    "TMushChainedFile::Seek()", 
		    "invalid file offset (too large offset or missing file)"
		); 
	    }
	    SeekTo(Offset);
	}
    }
}

void TMushChainedFile::SeekBy(Int64 Offset) throw(TSystemCallException)
{
    SeekTo(Position() + Offset);
}

void TMushChainedFile::Unlink(void) throw(TSystemCallException)
{
    TMushFile::Unlink();
    
    for (int FileIndex = 1; ; FileIndex++) {
	if (unlink(FileNameOf(FileIndex).c_str()) < 0) {
	    break;
	}
    }
}

long TMushChainedFile::Size(void) const throw(TSystemCallException)
{
    return TMushChainedFileAttribute(_Name).Size();
}

string TMushChainedFile::FileNameOf(unsigned FileIndex)
{
    string FileName = _FileNameCacheTable[FileIndex];

    if (FileName.empty()) {
	if (FileIndex == 0) {
	    FileName =_Name;
	}
	else {
	    ostringstream FileNameStream;
	    FileNameStream << _Name << '.' << FileIndex;
	    FileName = FileNameStream.str();
	}
	
	_FileNameCacheTable[FileIndex] = FileName;
    }

    return FileName;
}

Int64 TMushChainedFile::FileSizeOf(unsigned FileIndex) throw(TSystemCallException)
{
    Int64 Size = _FileSizeCacheTable[FileIndex];
    if (Size == 0) {
	try {
	    Size = TMushFileAttribute(FileNameOf(FileIndex)).Size();
	}
	catch (TSystemCallException &e) {
	    TSystemCallException(
		"TMushChainedFile::FileSizeOf(): " + FileNameOf(FileIndex), 
		e.Message()
	    );
	}

	_FileSizeCacheTable[FileIndex];
    }

    return Size;
}

int TMushChainedFile::MoveTo(unsigned FileIndex, Int64 FileHeadOffset) throw(TSystemCallException)
{
    string NewFileName = FileNameOf(FileIndex);
    if (_IsReadMode && (access(NewFileName.c_str(), R_OK) < 0)) {
	return -1;
    }

    int NewFileDescriptor = open(NewFileName.c_str(), _OpenFlag, _AccessMode);
    if (NewFileDescriptor < 0) {
	return -1;
    }
    
    close(_FileDescriptor);
    _FileDescriptor = NewFileDescriptor;

    _ThisFileIndex = FileIndex;
    _ThisFileHeadOffset = FileHeadOffset;
    _PositionInThisFile = 0;

    return _ThisFileIndex;
}

int TMushChainedFile::MoveToNext(void) throw(TSystemCallException)
{
    return MoveTo(_ThisFileIndex + 1, _ThisFileHeadOffset + FileSizeOf(_ThisFileIndex));
}

int TMushChainedFile::MoveToPrevious(void) throw(TSystemCallException)
{
    return MoveTo(_ThisFileIndex - 1, _ThisFileHeadOffset - FileSizeOf(_ThisFileIndex - 1));
}

int TMushChainedFile::MoveToFirst(void) throw(TSystemCallException)
{
    return MoveTo(0, 0);
}



TMushChainedFileAttribute::TMushChainedFileAttribute(const std::string& FilePathName)
: TMushFileAttribute(FilePathName)
{
}

TMushChainedFileAttribute::~TMushChainedFileAttribute()
{
}

long TMushChainedFileAttribute::Size(void) throw(TSystemCallException)
{
    long Size = TMushFileAttribute::Size();

    int SubfileIndex = 1;
    while (1) {
	ostringstream FileNameStream;
	FileNameStream << _FilePathName << '.' << SubfileIndex;

	TMushFileAttribute SubfileAttribute(FileNameStream.str());
	if (! SubfileAttribute.Exists()) {
	    break;
	}

	Size += SubfileAttribute.Size();
	SubfileIndex++;
    }

    return Size;
}
