/* MushConsole.cc */
/* Created by Enomoto Sanshiro on 29 November 2000. */
/* Last updated by Enomoto Sanshiro on 29 November 2000. */


#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <string>
#include "MushDefs.hh"
#include "MushConfig.hh"
#include "MushConsole.hh"

#if USE_READLINE
#undef HAVE_CONFIG_H
extern "C" {
#include <readline/readline.h>
#include <readline/history.h>
};
#endif

using namespace std;



TMushConsole::TMushConsole(void)
{
    _Prompt = 0;

    _HistoryItemListCapacity = 128;
    _NumberOfHistoryItems = 0;

    _HistoryItemList = new string[_HistoryItemListCapacity];
}

TMushConsole::~TMushConsole()
{
    delete[] _Prompt;
    delete[] _HistoryItemList;
}

string TMushConsole::SetPrompt(const string& Prompt)
{
    string OldPrompt;

    if (_Prompt != 0) {
	OldPrompt = _Prompt;
	delete[] _Prompt;
    }
    _Prompt = new char[Prompt.size() + 1];
    strcpy(_Prompt, Prompt.c_str());

    return OldPrompt;
}

bool TMushConsole::GetLine(string& Line)
{
    if (_Prompt != 0) {
	cout << _Prompt << flush;
    }

    return (bool) getline(cin, Line);
}

bool TMushConsole::GetLineTo(char* Buffer, size_t MaxSize)
{
    if (_Prompt != 0) {
	cout << _Prompt << flush;
    }

    return (bool) cin.getline(Buffer, MaxSize);
}

void TMushConsole::Put(const char* Buffer, size_t Size)
{
    cout.write(Buffer, Size);
}

void TMushConsole::AddHistory(const std::string& Line)
{
    if (Line == _LastHistoryItem) {
	return;
    }

    _HistoryItemList[_NumberOfHistoryItems % _HistoryItemListCapacity] = Line;
    _NumberOfHistoryItems++;
    _LastHistoryItem = Line;
}

void TMushConsole::LoadHistoryFrom(const std::string& FileName)
{
    ifstream File(FileName.c_str());
    if (File) {
	string Line;
	while (getline(File, Line)) {
	    AddHistory(Line);
	}
    }
}

void TMushConsole::SaveHistoryTo(const std::string& FileName)
{
    ofstream File(FileName.c_str());
    if (File) {
	int StartIndex = 0, NumberOfSavedItems = _NumberOfHistoryItems;
	if (_NumberOfHistoryItems > _HistoryItemListCapacity) {
	    StartIndex = _NumberOfHistoryItems % _HistoryItemListCapacity;
	    NumberOfSavedItems = _HistoryItemListCapacity;
	}

	for (int i = 0; i < NumberOfSavedItems; i++) {
	    int Index = (StartIndex + i) % _HistoryItemListCapacity;
	    File << _HistoryItemList[Index] << endl;
	}
    }
}



TMushReadlineConsole::TMushReadlineConsole(void)
{
#if USE_READLINE
    // disables <TAB> completion //
    //rl_bind_key('\t', (Function*) rl_insert); 
#endif
}

TMushReadlineConsole::~TMushReadlineConsole()
{
}

bool TMushReadlineConsole::GetLine(string& Line)
{
#if USE_READLINE
    char* Buffer = readline(_Prompt);     // EOF returns NULL
    if (Buffer != 0) {
	Line = Buffer;
	free(Buffer);
	if (! Line.empty()) {
	    AddHistory(Line);
	}
    }

    return (Buffer != 0);

#else
    return TMushConsole::GetLine(Line);
#endif
}

bool TMushReadlineConsole::GetLineTo(char* Buffer, size_t MaxSize)
{
#if USE_READLINE
    char* Line = readline(_Prompt);     // EOF returns NULL
    if (Line != 0) {
	if (Line[0] != '\0') {
	    AddHistory(Line);
	}
	strncpy(Buffer, Line, MaxSize - 1);
	free(Line);
    }

    return (Line != 0);
#else
    return TMushConsole::GetLineTo(Buffer, MaxSize);
#endif
}

void TMushReadlineConsole::AddHistory(const std::string& Line)
{
    TMushConsole::AddHistory(Line);

#if USE_READLINE
    add_history((char*) Line.c_str());
#endif
}
