/* MushDecoratedSocket.cc */
/* Created by Enomoto Sanshiro on 6 June 1999. */
/* Last updated by Enomoto Sanshiro on 4 December 2000. */


#include "MushDecoratedSocket.hh"



TMushDecoratedSocket::TMushDecoratedSocket(TMushSocket *ComponentSocket)
{
    _ComponentSocket = ComponentSocket;
}

TMushDecoratedSocket::~TMushDecoratedSocket()
{
    delete _ComponentSocket;
}
     
TMushSocket::TSocketStatus TMushDecoratedSocket::Status(void) const
{
    return _ComponentSocket->Status();
}

void TMushDecoratedSocket::Open(void) throw(TSystemCallException)
{
    _ComponentSocket->Open();
}

void TMushDecoratedSocket::Close(void)
{
    _ComponentSocket->Close();
}

int TMushDecoratedSocket::Send(void *Buffer, size_t Length) throw(TSystemCallException)
{
    return _ComponentSocket->Send(Buffer, Length);
}

int TMushDecoratedSocket::Receive(void *Buffer, size_t Length) throw(TSystemCallException)
{
    return _ComponentSocket->Receive(Buffer, Length);
}

bool TMushDecoratedSocket::IsServer(void) const
{
    return _ComponentSocket->IsServer();
}

int TMushDecoratedSocket::FileDescriptor(void) const
{
    return _ComponentSocket->FileDescriptor();
}



TMushFramedSocket::TMushFramedSocket(TMushSocket *ComponentSocket)
: TMushDecoratedSocket(ComponentSocket)
{
    _NextPacketSize = -1;
}

TMushFramedSocket::~TMushFramedSocket()
{
}

int TMushFramedSocket::Send(void *Buffer, size_t Length) throw(TSystemCallException)
{
    // send size header
    static unsigned char Header[_HeaderSize];
    for (int Byte = 0; Byte < _HeaderSize; Byte++) {
	Header[Byte] = (Length >> (8 * Byte)) & 0x000000ff;
    }
    if (_ComponentSocket->Send(Header, _HeaderSize) != _HeaderSize) {
	throw TSystemCallException(
	    "TMushFramedSocket::Send()", "could not send data header."
	);
    }

    // send data
    int SentLength;
    int RemainingLength = Length;
    char *DataPtr = (char *) Buffer;
    while (RemainingLength > 0) {
	SentLength = _ComponentSocket->Send(DataPtr, RemainingLength);

	DataPtr += SentLength;
	RemainingLength -= SentLength;
    }

    return Length;
}

int TMushFramedSocket::ReceiveBlock(TMushSocket* Socket, void *Buffer, size_t Length) throw(TSystemCallException)
{
    char* DataPtr = (char *) Buffer;
    char* EndPtr = DataPtr + Length;
    int RemainingLength, ThisLength;

    while ((RemainingLength = EndPtr - DataPtr) > 0) {
	ThisLength = Socket->Receive(DataPtr, RemainingLength);
	if (ThisLength > 0) {
	    DataPtr += ThisLength;
	}
	else if (ThisLength == 0) {
	    // end of file //
	    break;
	}
	else {
	    // interrupted etc. //
	    continue;
	}
    }

    return Length - RemainingLength;
}

int TMushFramedSocket::NextPacketSize(void) throw(TSystemCallException)
{
    if (_NextPacketSize >= 0) {
	return _NextPacketSize;
    }

    static unsigned char Header[_HeaderSize];
    int ReceivedLength = ReceiveBlock(_ComponentSocket, Header, _HeaderSize);

    if (ReceivedLength == 0) {
	_NextPacketSize = -1;
	return 0;
    }
    else if (ReceivedLength != _HeaderSize) {
	throw TSystemCallException(
	    "TMushFramedSocket::NextPacketSize()", "unexpected end-of-file"
	);
    }
    else {
	_NextPacketSize = 0;
	for (int Byte = 0; Byte < _HeaderSize; Byte++) {
	    _NextPacketSize += Header[Byte] << (8 * Byte);
	}
    }

    return _NextPacketSize;
}

int TMushFramedSocket::Receive(void *Buffer, size_t Length) throw(TSystemCallException)
{
    if (_NextPacketSize < 0) {
	NextPacketSize();
    }

    if (_NextPacketSize <= 0) {
	_NextPacketSize = -1;
	return 0;
    }
    else if (_NextPacketSize > (int) Length) {
	throw TSystemCallException(
	    "TMushFramedSocket::Receive()", "buffer is too small"
	);
    }

    int ReceivedLength = ReceiveBlock(_ComponentSocket, Buffer, _NextPacketSize);
    if (ReceivedLength != _NextPacketSize) {
	throw TSystemCallException(
	    "TMushFramedSocket::Receive()", "unexpected end-of-file"
	);
    }
    _NextPacketSize = -1;

    return ReceivedLength;
}



TMushEncodedSocket::TMushEncodedSocket(TMushSocket *ComponentSocket, TMushDataEncoder* DataEncoder)
: TMushDecoratedSocket(ComponentSocket)
{
    _DataEncoder = DataEncoder;

    _WorkAreaSize = 0;
    _WorkArea = 0;
}

TMushEncodedSocket::~TMushEncodedSocket()
{
    delete[] _WorkArea;
}

int TMushEncodedSocket::Send(void *Buffer, size_t Length) throw(TSystemCallException)
{
    if (_WorkAreaSize < Length) {
        delete[] _WorkArea;
	_WorkAreaSize = Length + Length / 2;
	_WorkArea = new char[_WorkAreaSize];
    }

    void* EncodedData;
    size_t EncodedDataSize;
    _DataEncoder->Encode(Buffer, Length, EncodedData, EncodedDataSize);

    _ComponentSocket->Send(EncodedData, EncodedDataSize);

    return Length;
}

int TMushEncodedSocket::Receive(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int ReceivedLength = _ComponentSocket->Receive(Buffer, Length);
    if (ReceivedLength <= 0) {
	return ReceivedLength;
    }

    void* DecodedData;
    size_t DecodedDataSize;
    _DataEncoder->Decode(Buffer, ReceivedLength, DecodedData, DecodedDataSize);

    return DecodedDataSize;
}
