/* MushRegularExpression.cc */
/* Created by Enomoto Sanshiro on 19 December 2000. */
/* Last updated by Enomoto Sanshiro on 10 June 2002. */


#include <iostream>
#include <string>
#include <sys/types.h>
#include <regex.h>
#include "MushDefs.hh"
#include "MushRegularExpression.hh"


using namespace std;


TMushPosixRegularExpression::TMushPosixRegularExpression(const string& Pattern, bool IsCaseInsensitive)
{
    _Pattern = Pattern;
    _IsCaseInsensitive = IsCaseInsensitive;

    _CompiledPattern = 0;
    _LastMatchResult = false;

    _SubMatchBuffer = 0;
}

TMushPosixRegularExpression::~TMushPosixRegularExpression()
{
    if (_CompiledPattern != 0) {
	regfree(_CompiledPattern);
	delete _CompiledPattern;
    }

    if (_SubMatchBuffer) {
	delete[] _SubMatchBuffer;
    }
}

void TMushPosixRegularExpression::CompilePattern(void) throw(TSystemCallException)
{
    _CompiledPattern = new regex_t();

    // Flags: 
    //   REG_EXTENDED: use POSIX extended regular expression syntax.
    //   REG_ICASE: do not differentiate case.

    int Flags = REG_EXTENDED;
    if (_IsCaseInsensitive) {
	Flags |= REG_ICASE;
    }

    int ErrorCode;
    if ((ErrorCode = regcomp(_CompiledPattern, _Pattern.c_str(), Flags)) < 0) {
	char Message[64];
	regerror(ErrorCode, _CompiledPattern, Message, sizeof(Message));
	regfree(_CompiledPattern);
	delete _CompiledPattern;
	_CompiledPattern = 0;

	throw TSystemCallException(
	    "TMushPosixRegularExpression::CompilePattern()", 
	    string("invalid regular expression: ") + Message + ": " + _Pattern
	);
    }

    // Note that SubMatch[0] is the whole region of each match.
    _NumberOfSubMatchs = _CompiledPattern->re_nsub + 1;
    _SubMatchBuffer = new regmatch_t[_NumberOfSubMatchs];
}

bool TMushPosixRegularExpression::Match(const string& String) throw(TSystemCallException)
{
    if (_CompiledPattern == 0) {
	CompilePattern();
    }

    _CurrentString = String;

    // Flags: 
    //   REG_NOTBOL: match-beginning-of-line operator always fails to match
    //   REG_NOTEOL: match-end-of-line operator always fauls to match

    int Flags = 0;
    int Result = regexec(
	_CompiledPattern, _CurrentString.c_str(), 
	_NumberOfSubMatchs, _SubMatchBuffer,
	Flags
    );

    _LastMatchResult = (Result == 0);

    return _LastMatchResult;
}

int TMushPosixRegularExpression::NumberOfSubMatchs(void)
{
    if (_LastMatchResult) {
	return _NumberOfSubMatchs;
    }
    else {
	return 0;
    }
}

string TMushPosixRegularExpression::SubMatchOf(int Index)
{
    if ((Index < 0) || (Index >= _NumberOfSubMatchs)) {
	return string("");
    }

    int Offset = _SubMatchBuffer[Index].rm_so;
    int Size = _SubMatchBuffer[Index].rm_eo - Offset;

    if (Offset < 0) {
	return string("");
    }

    return _CurrentString.substr(Offset, Size);
}

bool TMushPosixRegularExpression::FindPattern(const std::string& String, int& Offset, int& Size) throw(TSystemCallException)
{
    if (Match(String)) {
	Offset = _SubMatchBuffer[0].rm_so;
	Size = _SubMatchBuffer[0].rm_eo - Offset;
	return true;
    }
    else {
	Offset = String.size();
	Size = 0;
	return false;
    }
}

string TMushPosixRegularExpression::Substitute(const string& String, const string& SubstitutionString) throw(TSystemCallException)
{
    int Offset, Size;
    if (! FindPattern(String, Offset, Size)) {
	return String;
    }
    else {
	return (
	    String.substr(0, Offset) + 
	    SubstitutionString + 
	    String.substr(Offset + Size, string::npos)
	);
    }
}

string TMushPosixRegularExpression::SubstituteAll(const string& String, const string& SubstitutionString) throw(TSystemCallException)
{
    string Result;
    string Buffer = String;

    int Offset, Size;
    while (FindPattern(Buffer, Offset, Size)) {
	Result += Buffer.substr(0, Offset) + SubstitutionString;
	Buffer.erase(0, Offset + Size);
    }
    Result += Buffer;

    return Result;
}
