/* MushThread.cc */
/* Created by Enomoto Sanshiro on 13 August 1998. */
/* Last updated by Enomoto Sanshiro on 13 October 2001. */


#include <iostream>
#include <cerrno>
#include <pthread.h>
#include "MushThread.hh"

using namespace std;


void* g_ThreadEntry(void* Runnable);



TMushRunnable::TMushRunnable(void)
{
}

TMushRunnable::~TMushRunnable()
{
}

void TMushRunnable::Exit(void* ExitStatus)
{
    pthread_exit(ExitStatus);
}

void TMushRunnable::TestCancel(void)
{
    pthread_testcancel();
}

void TMushRunnable::EnableCancel(void)
{
    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
}

void TMushRunnable::DisableCancel(void)
{
    pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
}

void TMushRunnable::EnableAsynchronousCancel(void)
{
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
}

void TMushRunnable::DisableAsynchronousCancel(void)
{
    pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
}



TMushThread::TMushThread(TMushRunnable* Runnable)
{
    _Runnable = Runnable;
    pthread_attr_init(&_Attribute);
}

TMushThread::TMushThread(TMushRunnable& Runnable)
{
    _Runnable = &Runnable;
    pthread_attr_init(&_Attribute);
}

TMushThread::TMushThread(void)
{
    _Runnable = this;
    pthread_attr_init(&_Attribute);
}

TMushThread::~TMushThread()
{
}

void TMushThread::Start(void) throw(TSystemCallException)
{
    int Status = pthread_create(
	&_ThreadObject, &_Attribute, g_ThreadEntry, _Runnable
    );
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThread::Start(): pthread_create(3T)", Status
	);
    }
}

long TMushThread::Join(void) throw(TSystemCallException)
{
    void *ExitStatus;

    int Status = pthread_join(_ThreadObject, &ExitStatus);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThread::Join(): pthread_join(3T)", Status
	);
    }

    return (long) ExitStatus;
}

void TMushThread::Detach(void) throw(TSystemCallException)
{
    int Status = pthread_detach(_ThreadObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThread::Detach(): pthread_detach(3T)", Status
	);
    }
}

void TMushThread::Cancel(void) throw(TSystemCallException)
{
    int Status = pthread_cancel(_ThreadObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThread::Cancel(): pthread_detach(3T)", Status
	);
    }
}

void *g_ThreadEntry(void *Runnable)
{
    ((TMushRunnable *) Runnable)->Run();

    return 0;
}



TMushThreadMutex::TMushThreadMutex(void)
{
    _MutexObject = new pthread_mutex_t;
    if (pthread_mutex_init(_MutexObject, NULL) != 0) {
	_MutexObject = 0;
    }
}

TMushThreadMutex::TMushThreadMutex(const TMushThreadMutex& Mutex)
{
    throw TSystemCallException(
	"TMushThreadMutex::TMushThreadMutex()",
	"copying ThreadMutex object is prohibited"
    );
}

TMushThreadMutex& TMushThreadMutex::operator=(const TMushThreadMutex& Mutex)
{
    throw TSystemCallException(
	"TMushThreadMutex::operator=()",
	"copying ThreadMutex object is prohibited"
    );

    return *this;
}

TMushThreadMutex::~TMushThreadMutex()
{
    if (_MutexObject != 0) {
	pthread_mutex_destroy(_MutexObject);
	delete _MutexObject;
    }
}

void TMushThreadMutex::Lock(void) throw(TSystemCallException)
{
    if (_MutexObject == 0) {
	throw TSystemCallException(
	    "TMushThreadMutex::Lock()", "Mutex is not initialized"
	);
    }

    int Status = pthread_mutex_lock(_MutexObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadMutex::Lock(): pthread_mutex_lock(3T)", Status
	);
    }
}

bool TMushThreadMutex::TryLock(void) throw(TSystemCallException)
{
    if (_MutexObject == 0) {
	throw TSystemCallException(
	    "TMushThreadMutex::TryLock()", "Mutex is not initialized"
	);
    }

    int Status = pthread_mutex_trylock(_MutexObject);
    if ((Status != 0) && (Status != EBUSY)) {
	throw TSystemCallException(
	    "TMushThreadMutex::TryLock(): pthread_mutex_trylock(3T)", Status
	);
    }

    return (Status == 0);
}

void TMushThreadMutex::Unlock(void) throw(TSystemCallException)
{
    if (_MutexObject == 0) {
	throw TSystemCallException(
	    "TMushThreadMutex::Unock()", "Mutex is not initialized"
	);
    }

    int Status = pthread_mutex_unlock(_MutexObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadMutex::Unlock(): pthread_mutex_unlock(3T)", Status
	);
    }
}



TMushThreadCondition::TMushThreadCondition(void)
{
    _ConditionObject = new pthread_cond_t;
    if (pthread_cond_init(_ConditionObject, NULL) != 0) {
	_ConditionObject = 0;
    }
}

TMushThreadCondition::TMushThreadCondition(const TMushThreadCondition& Condition)
: TMushThreadMutex(Condition)
{
    throw TSystemCallException(
	"TMushThreadCondition::TMushThreadCondition()",
	"copying ThreadCondition object is prohibited"
    );
}

TMushThreadCondition& TMushThreadCondition::operator=(const TMushThreadCondition& Condition)
{
    throw TSystemCallException(
	"TMushThreadCondition::operator=()",
	"copying ThreadCondition object is prohibited"
    );

    return *this;
}

TMushThreadCondition::~TMushThreadCondition()
{
    if (_ConditionObject != 0) {
	pthread_cond_destroy(_ConditionObject);
	delete _ConditionObject;
    }
}

void TMushThreadCondition::Wait(void) throw(TSystemCallException)
{
    if (_ConditionObject == 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::Wait()", "Condition is not initialized"
	);
    }

    int Status = pthread_cond_wait(_ConditionObject, _MutexObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::Wait(): pthread_cond_wait(3T)", Status
	);
    }
}

void TMushThreadCondition::TimedWait(long Sec, long NSec) throw(TSystemCallException)
{
    if (_ConditionObject == 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::TimedWait()", "Condition is not initialized"
	);
    }

    struct timespec TimeOut;
    TimeOut.tv_sec = Sec;
    TimeOut.tv_nsec = NSec;

    int Status = pthread_cond_timedwait(_ConditionObject, _MutexObject, &TimeOut);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::TmiedWait(): pthread_cond_tmiedwait(3T)", 
	    Status
	);
    }
}

void TMushThreadCondition::Signal(void) throw(TSystemCallException)
{
    if (_ConditionObject == 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::Signal()", "Condition is not initialized"
	);
    }

    int Status = pthread_cond_signal(_ConditionObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::(): pthread_cond_signal(3T)", Status
	);
    }
}

void TMushThreadCondition::Broadcast(void) throw(TSystemCallException)
{
    if (_ConditionObject == 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::Broadcast()", "Condition is not initialized"
	);
    }

    int Status = pthread_cond_broadcast(_ConditionObject);
    if (Status != 0) {
	throw TSystemCallException(
	    "TMushThreadCondition::(): pthread_cond_broadcast(3T)", Status
	);
    }
}
