/* MushTimer.cc */
/* Created by Enomoto Sanshiro on 6 May 1998. */
/* Last updated by Enomoto Sanshiro on 6 October 1999. */


#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include "MushTimer.hh"

using namespace std;


TMushSignalClient *TMushIntervalTimer::_SignalClient[NSIG];

TMushIntervalTimer::TMushIntervalTimer(int TimerSignalId, unsigned long Time_sec, unsigned long Time_usec, TMushSignalClient *SignalClient)
{
    _SignalId = TimerSignalId;
    _SignalClient[_SignalId] = SignalClient;

    sigemptyset(&_EmptySignalSet);
    sigfillset(&_AllSignalSet);
    sigfillset(&_NonAlarmSignalSet);
    sigdelset(&_NonAlarmSignalSet, _SignalId);

    SetTimeValue(Time_sec, Time_usec);

    struct sigaction SignalAction;
    SignalAction.sa_handler = Handler;
    SignalAction.sa_mask = _AllSignalSet;
    SignalAction.sa_flags = 0;

    sigaction(_SignalId, &SignalAction, &_OldSignalAction);
}

TMushIntervalTimer::~TMushIntervalTimer() 
{
    sigaction(_SignalId, &_OldSignalAction, 0);
    _SignalClient[_SignalId] = 0;
}

void TMushIntervalTimer::SetTimeValue(unsigned long Time_sec, unsigned long Time_usec)
{
    _IntervalTimerValue.it_value.tv_sec = Time_sec + (Time_usec / 1000000);
    _IntervalTimerValue.it_value.tv_usec = Time_usec % 1000000;
    _IntervalTimerValue.it_interval.tv_sec = 0;
    _IntervalTimerValue.it_interval.tv_usec = 0;
}

void TMushIntervalTimer::GetTimeValue(unsigned long &Time_sec, unsigned long &Time_usec)
{
    Time_sec = _IntervalTimerValue.it_value.tv_sec;
    Time_usec = _IntervalTimerValue.it_value.tv_usec;
}

void TMushIntervalTimer::Start(void)
{
    _IntervalTimerValue.it_interval.tv_sec = 0;
    _IntervalTimerValue.it_interval.tv_usec = 0;

    SetTimer(_IntervalTimerValue);
}

void TMushIntervalTimer::StartRepetitive(void)
{
    _IntervalTimerValue.it_interval.tv_sec = _IntervalTimerValue.it_value.tv_sec;
    _IntervalTimerValue.it_interval.tv_usec = _IntervalTimerValue.it_value.tv_usec;

    SetTimer(_IntervalTimerValue);
}

void TMushIntervalTimer::Stop(void)
{
    struct itimerval IntervalTimerValue;

    IntervalTimerValue.it_value.tv_sec = 0;
    IntervalTimerValue.it_value.tv_usec = 0;
    IntervalTimerValue.it_interval.tv_sec = 0;
    IntervalTimerValue.it_interval.tv_usec = 0;

    SetTimer(IntervalTimerValue);
}

void TMushIntervalTimer::Read(unsigned long &Time_sec, unsigned long &Time_usec)
{
    struct itimerval CurrentIntervalTimerValue;

    GetTimer(CurrentIntervalTimerValue);

    Time_sec = CurrentIntervalTimerValue.it_value.tv_sec;
    Time_usec = CurrentIntervalTimerValue.it_value.tv_usec;
}

void TMushIntervalTimer::InterruptibleSuspend(void)
{
    static sigset_t OldSignalSet;
    sigprocmask(SIG_BLOCK, &_AllSignalSet, &OldSignalSet);

    SetTimer(_IntervalTimerValue);
    sigsuspend(&OldSignalSet);

    sigprocmask(SIG_SETMASK, &OldSignalSet, 0);
}

void TMushIntervalTimer::Suspend(void)
{
    static sigset_t OldSignalSet;
    sigprocmask(SIG_BLOCK, &_AllSignalSet, &OldSignalSet);

    SetTimer(_IntervalTimerValue);
    sigsuspend(&_NonAlarmSignalSet);

    sigprocmask(SIG_SETMASK, &OldSignalSet, 0);
}

void TMushIntervalTimer::Handler(int SignalId)
{
    if (_SignalClient[SignalId] != 0) {
	_SignalClient[SignalId]->OnCatchSignal(SignalId);
    }
}


void TMushRealTimeTimer::SetTimer(struct itimerval& TimerValue)
{
    setitimer(ITIMER_REAL, &TimerValue, 0);
}

void TMushRealTimeTimer::GetTimer(struct itimerval& TimerValue)
{
    getitimer(ITIMER_REAL, &TimerValue);
}

void TMushVirtualTimeTimer::SetTimer(struct itimerval& TimerValue)
{
    setitimer(ITIMER_VIRTUAL, &TimerValue, 0);
}

void TMushVirtualTimeTimer::GetTimer(struct itimerval& TimerValue)
{
    getitimer(ITIMER_VIRTUAL, &TimerValue);
}

void TMushProfileTimeTimer::SetTimer(struct itimerval& TimerValue)
{
    setitimer(ITIMER_PROF, &TimerValue, 0);
}

void TMushProfileTimeTimer::GetTimer(struct itimerval& TimerValue)
{
    getitimer(ITIMER_PROF, &TimerValue);
}
