/* DeviceFactory.cc */
/* Created by Enomoto Sanshiro on 7 May 1999. */
/* Updated by Enomoto Sanshiro on 7 July 2001. */
/* Updated by Enomoto Sanshiro on 24 June 2009. */


#include <string>
#include <map>
#include "RoomModule.hh"
#include "RoomVmeAccess.hh"
#include "RoomCamacAccess.hh"
#include "RoomNetworkAccess.hh"
#include "RoomVmeBridge.hh"
#include "RoomNetworkBridge.hh"
#include "RoomSoftwareDevice.hh"
#include "RoomDeviceFactory.hh"

using namespace std;


TRoomDeviceFactory* TRoomDeviceFactory::_Instance = 0;

TRoomDeviceFactory::TRoomDeviceFactory()
{
    _VmeControllerTable["Null"] = new TRoomNullVmeController();
    _CamacControllerTable["Null"] = new TRoomNullCamacController();
}

TRoomDeviceFactory::~TRoomDeviceFactory()
{
}

TRoomDeviceFactory* TRoomDeviceFactory::GetInstance(void)
{
    if (_Instance == 0) {
        _Instance = new TRoomDeviceFactory();
    }

    return _Instance;
}

string TRoomDeviceFactory::RegulatedNameOf(string Name) const
{
    // replace '-' or ' ' with '_'
    for (unsigned i = 0; i < Name.size(); i++) {
	if ((Name[i] == '-') || (Name[i] == ' ')) {
	    Name[i] = '_';
	}
    }

    return Name;
}

void TRoomDeviceFactory::RegisterVmeController(const string& Name, TRoomVmeController* Controller)
{
    _VmeControllerTable[Name] = Controller;
}

void TRoomDeviceFactory::RegisterCamacController(const string& Name, TRoomCamacController* Controller)
{
    _CamacControllerTable[Name] = Controller;
}

void TRoomDeviceFactory::RegisterVmeCamacBridge(const string& Name, TRoomVmeCamacBridge* Bridge)
{
    _VmeCamacBridgeTable[Name] = Bridge;
}

void TRoomDeviceFactory::RegisterNetworkCamacBridge(const string& Name, TRoomNetworkCamacBridge* Bridge)
{
    _NetworkCamacBridgeTable[Name] = Bridge;
}

void TRoomDeviceFactory::RegisterVmeModule(const string& Name, TRoomVmeModule* Module)
{
    _VmeModuleTable[Name] = Module;
}

void TRoomDeviceFactory::RegisterCamacModule(const string& Name, TRoomCamacModule* Module)
{
    _CamacModuleTable[Name] = Module;
}

void TRoomDeviceFactory::RegisterNetworkModule(const string& Name, TRoomNetworkModule* Module)
{
    _NetworkModuleTable[Name] = Module;
}

void TRoomDeviceFactory::RegisterSoftwareModule(const string& Name, TRoomSoftwareModule* Module)
{
    _SoftwareModuleTable[Name] = Module;
}

TRoomVmeController* TRoomDeviceFactory::CreateVmeController(const string& Name) throw(THardwareException)
{
    TRoomVmeController* ControllerPrototype;

    string RegulatedName = RegulatedNameOf(Name);
    if (_VmeControllerTable.count(RegulatedName) > 0) {
	ControllerPrototype = _VmeControllerTable[RegulatedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateVmeController()",
	    "No such VME controller: " + Name
	);
    }

    return ControllerPrototype->CloneController();
}

TRoomCamacController* TRoomDeviceFactory::CreateCamacController(const string& Name) throw(THardwareException)
{
    TRoomCamacController* ControllerPrototype = 0;

    string RegulatedName = RegulatedNameOf(Name);
    if (_CamacControllerTable.count(RegulatedName) > 0) {
	ControllerPrototype = _CamacControllerTable[RegulatedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateCamacController()",
	    "No such CAMAC controller: " + Name
	);
    }

    return ControllerPrototype->CloneController();
}

TRoomVmeCamacBridge* TRoomDeviceFactory::CreateVmeCamacBridge(const string& Name) throw(THardwareException)
{
    TRoomVmeCamacBridge* Prototype = 0;

    string RegulatedName = RegulatedNameOf(Name);
    if (_VmeCamacBridgeTable.count(RegulatedName) > 0) {
	Prototype = _VmeCamacBridgeTable[RegulatedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateVmeCamacBridge()",
	    "No such VME-CAMAC bridge: " + Name
	);
    }

    return Prototype->CloneBridge();
}

TRoomNetworkCamacBridge* TRoomDeviceFactory::CreateNetworkCamacBridge(const string& Name) throw(THardwareException)
{
    TRoomNetworkCamacBridge* Prototype = 0;

    string RegulatedName = RegulatedNameOf(Name);
    if (_NetworkCamacBridgeTable.count(RegulatedName) > 0) {
	Prototype = _NetworkCamacBridgeTable[RegulatedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateVmeCamacBridge()",
	    "No such Network-CAMAC bridge: " + Name
	);
    }

    return Prototype->CloneBridge();
}

TRoomVmeCrate* TRoomDeviceFactory::CreateVmeCrate(const string& Name) throw(THardwareException)
{
    return new TRoomVmeCrate();
}

TRoomCamacCrate* TRoomDeviceFactory::CreateCamacCrate(const string& Name) throw(THardwareException)
{
    return new TRoomCamacCrate();
}

TRoomVmeModule* TRoomDeviceFactory::CreateVmeModule(const string& Name) throw(THardwareException)
{
    string RegulatedName = RegulatedNameOf(Name);

    if (_VmeModuleTable.count(RegulatedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateVmeModule()",
            "No such VME module: " + Name
        );
    }

    return _VmeModuleTable[RegulatedName]->Clone();
}

TRoomCamacModule* TRoomDeviceFactory::CreateCamacModule(const string& Name) throw(THardwareException)
{
    string RegulatedName = RegulatedNameOf(Name);

    if (_CamacModuleTable.count(RegulatedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateCamacModule()",
            "No such CAMAC module: " + Name
        );
    }

    return _CamacModuleTable[RegulatedName]->Clone();
}

TRoomNetworkModule* TRoomDeviceFactory::CreateNetworkModule(const string& Name) throw(THardwareException)
{
    string RegulatedName = RegulatedNameOf(Name);

    if (_NetworkModuleTable.count(RegulatedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateNetworkModule()",
            "No such network module: " + Name
        );
    }

    return _NetworkModuleTable[RegulatedName]->Clone();
}

TRoomSoftwareModule* TRoomDeviceFactory::CreateSoftwareModule(const string& Name) throw(THardwareException)
{
    string RegulatedName = RegulatedNameOf(Name);

    if (_SoftwareModuleTable.count(RegulatedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateSoftwareModule()",
            "No such software module: " + Name
        );
    }

    return _SoftwareModuleTable[RegulatedName]->Clone();
}

void TRoomDeviceFactory::ShowDeviceList(std::ostream& os) const
{
    cout << "VME Controllers:" << endl;
    map<string, TRoomVmeController*>::const_iterator VmeController;
    for (
	 VmeController = _VmeControllerTable.begin();
	 VmeController != _VmeControllerTable.end();
	 VmeController++
    ){
        os << "    " << VmeController->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "CAMAC Controllers:" << endl;
    map<string, TRoomCamacController*>::const_iterator CamacController;
    for (
	 CamacController = _CamacControllerTable.begin();
	 CamacController != _CamacControllerTable.end();
	 CamacController++
    ){
        os << "    " << CamacController->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "VME-CAMAC Bridges:" << endl;
    map<string, TRoomVmeCamacBridge*>::const_iterator VmeCamacBridge;
    for (
	 VmeCamacBridge = _VmeCamacBridgeTable.begin();
	 VmeCamacBridge != _VmeCamacBridgeTable.end();
	 VmeCamacBridge++
    ){
        os << "    " << VmeCamacBridge->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "Network-CAMAC Bridges:" << endl;
    map<string, TRoomNetworkCamacBridge*>::const_iterator NetworkCamacBridge;
    for (
	 NetworkCamacBridge = _NetworkCamacBridgeTable.begin();
	 NetworkCamacBridge != _NetworkCamacBridgeTable.end();
	 NetworkCamacBridge++
    ){
        os << "    " << NetworkCamacBridge->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "VME Modules:" << endl;
    map<string, TRoomVmeModule*>::const_iterator VmeModule;
    for (
	 VmeModule = _VmeModuleTable.begin();
	 VmeModule != _VmeModuleTable.end();
	 VmeModule++
    ){
        os << "    " << VmeModule->first << ": ";
	os << VmeModule->second->ModelName();
	os << endl;
    }
    os << endl;

    cout << "CAMAC Modules:" << endl;
    map<string, TRoomCamacModule*>::const_iterator CamacModule;
    for (
	 CamacModule = _CamacModuleTable.begin();
	 CamacModule != _CamacModuleTable.end();
	 CamacModule++
    ){
        os << "    " << CamacModule->first << ": ";
	os << CamacModule->second->ModelName();
	os << endl;
    }
    os << endl;

    cout << "Network Modules:" << endl;
    map<string, TRoomNetworkModule*>::const_iterator NetworkModule;
    for (
	 NetworkModule = _NetworkModuleTable.begin();
	 NetworkModule != _NetworkModuleTable.end();
	 NetworkModule++
    ){
        os << "    " << NetworkModule->first << ": ";
	os << NetworkModule->second->ModelName();
	os << endl;
    }
    os << endl;

    cout << "Software Modules:" << endl;
    map<string, TRoomSoftwareModule*>::const_iterator SoftwareModule;
    for (
	 SoftwareModule = _SoftwareModuleTable.begin();
	 SoftwareModule != _SoftwareModuleTable.end();
	 SoftwareModule++
    ){
        os << "    " << SoftwareModule->first << ": ";
	os << SoftwareModule->second->ModelName();
	os << endl;
    }
    os << endl;
}



TRoomVmeControllerCreator::TRoomVmeControllerCreator(const string& Name, TRoomVmeController* Controller)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeController(Name, Controller);
}

TRoomCamacControllerCreator::TRoomCamacControllerCreator(const string& Name, TRoomCamacController* Controller)
{
    TRoomDeviceFactory::GetInstance()->RegisterCamacController(Name, Controller);
}

TRoomVmeCamacBridgeCreator::TRoomVmeCamacBridgeCreator(const string& Name, TRoomVmeCamacBridge* Bridge)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeCamacBridge(Name, Bridge);
}

TRoomNetworkCamacBridgeCreator::TRoomNetworkCamacBridgeCreator(const string& Name, TRoomNetworkCamacBridge* Bridge)
{
    TRoomDeviceFactory::GetInstance()->RegisterNetworkCamacBridge(Name, Bridge);
}

TRoomVmeModuleCreator::TRoomVmeModuleCreator(const string& Name, TRoomVmeModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeModule(Name, Module);
}

TRoomCamacModuleCreator::TRoomCamacModuleCreator(const string& Name, TRoomCamacModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterCamacModule(Name, Module);
}

TRoomNetworkModuleCreator::TRoomNetworkModuleCreator(const string& Name, TRoomNetworkModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterNetworkModule(Name, Module);
}

TRoomSoftwareModuleCreator::TRoomSoftwareModuleCreator(const string& Name, TRoomSoftwareModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterSoftwareModule(Name, Module);
}
