/* RoomNetworkAccess.cc */
/* Created by Enomoto Sanshiro on 16 July 2009. */
/* Last updated by Enomoto Sanshiro on 16 July 2009. */


#include <string>
#include <errno.h>

#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <poll.h>

#include "RoomNetworkAccess.hh"


using namespace std;


TRoomNetworkConnection::TRoomNetworkConnection(void)
{
    _Socket = 0;
}

TRoomNetworkConnection::~TRoomNetworkConnection()
{
    if (_Socket >= 0) {
	static const int ShutdownMethod = 2;  // 2: shutdown send and receive
	shutdown(_Socket, ShutdownMethod);
	close(_Socket);
    }
}

int TRoomNetworkConnection::Connect(const string& RemoteHostName, int PortNumber) throw(THardwareException)
{
    // Obtain Remote Host Address //
    struct hostent *RemoteHost = gethostbyname(RemoteHostName.c_str());
    if (RemoteHost == 0) {
	unsigned long Address = inet_addr(RemoteHostName.c_str());
	if ((signed long) Address > 0) {
            RemoteHost = gethostbyaddr(
		(char *) &Address, sizeof(Address), AF_INET
	    );
	}
    }
    if (RemoteHost == 0) {
	throw THardwareException(
	    "TRoomNetworkConnection::Connect()",
	    "unable to find host: " + RemoteHostName
        );
    }
    unsigned long RemoteAddress = (
	* (unsigned long *) RemoteHost->h_addr_list[0]
    );

    // Create Client Socket //
    _Socket = socket(AF_INET, SOCK_STREAM, 0);
    if (_Socket < 0) {
	_Socket = 0;
	throw THardwareException(
	    "TRoomNetworkConnection::Connect()", 
	    "socket(2): " + string(strerror(errno))
	);
    }

    // Connect Socket //
    struct sockaddr_in RemoteSocketAddress;
    RemoteSocketAddress.sin_family = AF_INET;
    RemoteSocketAddress.sin_port = htons(PortNumber);
    RemoteSocketAddress.sin_addr.s_addr = RemoteAddress;
    if (
        connect(
	    _Socket,
	    (struct sockaddr *) &RemoteSocketAddress, 
            sizeof(RemoteSocketAddress)
	) < 0
    ){
	close(_Socket);
	_Socket = 0;
	throw THardwareException(
	    "TRoomNetworkConnection::Connect()",
	    "connect(2): " +string(strerror(errno))
	);
    }

    return _Socket;
}

bool TRoomNetworkConnection::Probe(void) throw(THardwareException)
{
    if (_Socket <= 0) {
	throw THardwareException(
	    "TRoomNetworkConnection::Probe()",
	    "network not connected"
	);
    }
    
    return true;
}

int TRoomNetworkConnection::Send(const void* Data, size_t Size) throw(THardwareException)
{
    int SentSize = send(_Socket, (const char *) Data, Size, MSG_DONTWAIT);

    if ((SentSize < 0) && (errno == EAGAIN)) {
	SentSize = send(_Socket, (const char *) Data, Size, 0);
    }

    if (SentSize < 0) {
	if (errno == EINTR) {
	    SentSize = 0;
	}
	else {
	    throw THardwareException(
		"TRoomNetworkConnection::Send()",
		"send(2): " + string(strerror(errno))
	    );
	}
    }

    return SentSize;
}

int TRoomNetworkConnection::Receive(void* Data, size_t MaxSize) throw(THardwareException)
{
    int ReceivedSize = recv(_Socket, (char *) Data, MaxSize, 0);

    if (ReceivedSize < 0) {
	if (errno == EINTR) {
	    ReceivedSize = 0;
	}
	else {
	    throw THardwareException(
		"TRoomNetworkConnection::Send()",
		"recv(2): " + string(strerror(errno))
	    );
	}
    }

    return ReceivedSize;
}

int TRoomNetworkConnection::SendBlock(const void* Data, size_t Size) throw(THardwareException)
{
    int RemainingSize = Size;
    while (RemainingSize > 0) {
	int SentSize = Send(Data, RemainingSize);
	if (SentSize <= 0) {
	    break;
	}
	RemainingSize -= SentSize;
	Data = (const char*) Data + SentSize;
    }

    return Size - RemainingSize;
}

int TRoomNetworkConnection::ReceiveBlock(void* Data, size_t Size) throw(THardwareException)
{
    int RemainingSize = Size;
    while (RemainingSize > 0) {
	int ReceivedSize = Receive(Data, RemainingSize);
	if (ReceivedSize <= 0) {
	    break;
	}
	RemainingSize -= ReceivedSize;
	Data = (char*) Data + ReceivedSize;
    }

    return Size - RemainingSize;
}

bool TRoomNetworkConnection::WaitArrival(int TimeOut_ms) throw(THardwareException)
{
    pollfd PollFd;
    PollFd.fd = _Socket;
    PollFd.events = POLLIN;

    int NumberOfEvents = poll(&PollFd, 1, TimeOut_ms);
    if (NumberOfEvents < 0) {
	if (errno == EINTR) {
	    return false;
	}
	else {
	    throw THardwareException(
		"TRoomNetworkConnection::WaitArrival()",
		"poll(2): " + string(strerror(errno))
	    );
	}
    }

    if (NumberOfEvents == 0) {
	return false;
    }

    return (PollFd.revents & POLLIN) != 0;
}



TRoomNetworkModule::TRoomNetworkModule(const string& ModuleType, const string& ModelName)
: TRoomModule(ModuleType, ModelName)
{
}

TRoomNetworkModule::~TRoomNetworkModule()
{
}

