/* RoomServiceRequester.cc */
/* Created by Enomoto Sanshiro on 7 July 2001. */
/* Last updated by Enomoto Sanshiro on 29 March 2002. */


#include <unistd.h>
#include <signal.h>
#include "RoomConfig.hh"
#include "RoomDefs.hh"
#include "RoomServiceRequester.hh"

using namespace std;


#define DEBUG(x)
//#define DEBUG(x) (x)


static const int DefaultPollingInterval_usec = 10;


TRoomServiceRequester::TRoomServiceRequester(void)
{
}

TRoomServiceRequester::~TRoomServiceRequester()
{
}

bool TRoomServiceRequester::WaitForServiceRequest(int TimeOut_sec) throw(THardwareException)
{
    throw THardwareException(
        "TRoomServiceRequester::WaitForServiceRequest()", 
	"function is not available"
    );

    return false;
}

int TRoomServiceRequester::RequestingServiceNumber(void) throw(THardwareException)
{
    return 0;
}

void TRoomServiceRequester::EnableServiceRequest(void) throw(THardwareException)
{
}

void TRoomServiceRequester::DisableServiceRequest(void) throw(THardwareException)
{
}

void TRoomServiceRequester::ClearServiceRequest(void) throw(THardwareException)
{
    throw THardwareException(
        "TRoomServiceRequester::ClearServiceRequest()", 
	"function is not available"
    );
}

bool TRoomServiceRequester::IsRequestingService(void) throw(THardwareException)
{
    throw THardwareException(
        "TRoomServiceRequester::IsRequestingService()", 
	"function is not available"
    );

    return 0;
}

bool TRoomServiceRequester::IsSignalOnServiceRequestAvailable(void)
{
    return false;
}

void TRoomServiceRequester::EnableSignalOnServiceRequest(int SignalId) throw(THardwareException)
{
}

void TRoomServiceRequester::DisableSignalOnServiceRequest(void) throw(THardwareException)
{
}



TRoomServiceRequestHandler::TRoomServiceRequestHandler(void)
{
    _RequesterListCapacity = 32;
    _NumberOfRequesters = 0;

    _RequesterList = new TRoomServiceRequester* [_RequesterListCapacity];
    _RequesterIdList = new int[_RequesterListCapacity];

    _IsSignalAvailable = false;
    _IsPollingForced = false;
    _PollingInterval_usec = DefaultPollingInterval_usec;

    _RequestingRequester = 0;
}

TRoomServiceRequestHandler::~TRoomServiceRequestHandler()
{
    delete[] _RequesterIdList;
    delete[] _RequesterList;
}

void TRoomServiceRequestHandler::SetPollingInterval(int Interval_usec)
{
    _PollingInterval_usec = Interval_usec;
}

void TRoomServiceRequestHandler::ForcePolling(void)
{
    _IsPollingForced = true;
}

void TRoomServiceRequestHandler::RegisterRequester(TRoomServiceRequester* Requester, int RequesterId)
{
    if (_NumberOfRequesters + 1 > _RequesterListCapacity) {
	TRoomServiceRequester** OldRequesterList = _RequesterList;
	int* OldRequesterIdList = _RequesterIdList;

	_RequesterListCapacity *= 2;
	_RequesterList = new TRoomServiceRequester* [_RequesterListCapacity];
	_RequesterIdList = new int[_RequesterListCapacity];

	for (int i = 0; i < _NumberOfRequesters; i++) {
	    _RequesterList[i] = OldRequesterList[i];
	    _RequesterIdList[i] = OldRequesterIdList[i];
	}

	delete[] OldRequesterIdList;
	delete[] OldRequesterList;
    }

    _RequesterList[_NumberOfRequesters] = Requester;
    _RequesterIdList[_NumberOfRequesters] = RequesterId;
    _NumberOfRequesters++;

    if (_NumberOfRequesters == 1) {
	_IsSignalAvailable = Requester->IsSignalOnServiceRequestAvailable();
    }
    else {
	_IsSignalAvailable &= Requester->IsSignalOnServiceRequestAvailable();
    }
}

void TRoomServiceRequestHandler::Initialize(void) throw(THardwareException)
{
    if (_IsSignalAvailable) {
	StartSignalHandling();
	for (int i = 0; i < _NumberOfRequesters; i++) {
	    _RequesterList[i]->EnableSignalOnServiceRequest(_SignalId);
	}
    }

    for (int j = 0; j < _NumberOfRequesters; j++) {
	_RequesterList[j]->EnableServiceRequest();
    }

    _RequestingRequester = 0;
}

void TRoomServiceRequestHandler::Finalize(void) throw(THardwareException)
{
    for (int j = 0; j < _NumberOfRequesters; j++) {
	_RequesterList[j]->DisableServiceRequest();
    }

    if (_IsSignalAvailable) {
	for (int i = 0; i < _NumberOfRequesters; i++) {
	    _RequesterList[i]->DisableSignalOnServiceRequest();
	}
	StopSignalHandling();
    }
}

bool TRoomServiceRequestHandler::WaitForServiceRequest(int& RequesterId, int& RequestNumber, int TimeOut_sec) throw(THardwareException)
{
    int RequesterIndex;

    if (_IsPollingForced) {
	RequesterIndex = PollRequest(TimeOut_sec);
    }
    else if (_NumberOfRequesters == 1) {
	if (_RequesterList[0]->IsRequestingService()) {
	    RequesterIndex = 0;
	}
        else if (_RequesterList[0]->WaitForServiceRequest(TimeOut_sec)) {
	    RequesterIndex = 0;
	}
	else {
	    RequesterIndex = -1;
	}
    }
    else if (_NumberOfRequesters > 1) {
	if (_IsSignalAvailable) {
	    RequesterIndex = LookForRequestSignal(TimeOut_sec);
	}
	else {
	    RequesterIndex = PollRequest(TimeOut_sec);
	}
    }
    else {
	if (TimeOut_sec > 0) {
	    Suspend(TimeOut_sec, 0);
	}
	RequesterIndex = -1;
    }
    
    if (RequesterIndex >= 0) {
	TRoomServiceRequester* Requester = _RequesterList[RequesterIndex];
	RequesterId = _RequesterIdList[RequesterIndex];
	RequestNumber = Requester->RequestingServiceNumber();
	_RequestingRequester = Requester;
    }
    else {
	_RequestingRequester = 0;
    }

    return (RequesterIndex >= 0);
}

void TRoomServiceRequestHandler::ClearServiceRequest(void) throw(THardwareException)
{
    if (_RequestingRequester) {
	_RequestingRequester->ClearServiceRequest();
    }
}

int TRoomServiceRequestHandler::LookForRequestSignal(int TimeOut_sec) throw(THardwareException)
{
    DEBUG(cerr << "waiting signal..." << flush);

    TimeOut_sec = WaitForSignal(TimeOut_sec);
    if (TimeOut_sec == 0) {
	DEBUG(cerr << "timedout." << endl);
	return -1;
    }
    else {
	DEBUG(cerr << "ok." << endl);
    }

    static int RequesterIndex = -1;
    int PollingCount = 0;

    while (1) {
	RequesterIndex++;
	if (RequesterIndex == _NumberOfRequesters) {
	    RequesterIndex = 0;
	}

	if (_RequesterList[RequesterIndex]->IsRequestingService()) {
	    break;
	}

	PollingCount++;
	if (PollingCount == _NumberOfRequesters) {
	    return -1;
	}
    }

    return RequesterIndex;
}

int TRoomServiceRequestHandler::PollRequest(int TimeOut_sec) throw(THardwareException)
{
    static int RequesterIndex = -1;
    int PollingCount = 0;

    DEBUG(cerr << "polling request" << flush);
    while (1) {
	RequesterIndex++;
	if (RequesterIndex == _NumberOfRequesters) {
	    RequesterIndex = 0;
	}

	if (_RequesterList[RequesterIndex]->IsRequestingService()) {
	    break;
	}

	PollingCount++;
	if (PollingCount == _NumberOfRequesters) {
	    if (_PollingInterval_usec > 0) {
		Suspend(0, _PollingInterval_usec);
	    }
	    TimeOut_sec -= _PollingInterval_usec;
	    DEBUG(cerr << "." << flush);

	    if (TimeOut_sec <= 0) {
		DEBUG(cerr << "timed out" << endl);
		return -1;
	    }
	}
    }
    DEBUG(cerr << "ok" << endl);

    return RequesterIndex;
}



const int TRoomServiceRequestHandler::_SignalId = SIGPOLL;
int TRoomServiceRequestHandler::_SignalCount = 0;
void (*TRoomServiceRequestHandler::_OldSignalHandler)(int) = 0;

void TRoomServiceRequestHandler::SignalHandler(int SignalId)
{
    if (SignalId == _SignalId) {
	_SignalCount++;
    }
}

void TRoomServiceRequestHandler::StartSignalHandling(void)
{
    _SignalCount = 0;    
    _OldSignalHandler = signal(_SignalId, SignalHandler);
}

void TRoomServiceRequestHandler::StopSignalHandling(void)
{
    if (_OldSignalHandler != 0) {
	signal(_SignalId, _OldSignalHandler);
    }
}

int TRoomServiceRequestHandler::WaitForSignal(int TimeOut_sec)
{
    if (_SignalCount > 0) {
	_SignalCount--;
	return TimeOut_sec;
    }

    TimeOut_sec = sleep(TimeOut_sec);
    _SignalCount = 0;

    return TimeOut_sec;
}
