/* module-CAEN_V775.cc */
/* Created by Enomoto Sanshiro on 24 April 2005 */
/* Last updated by Enomoto Sanshiro on 9 November 2005. */


#include <iostream>
#include <iomanip>

#include "RoomDeviceFactory.hh"
#include "RoomVmeAccess.hh"
#include "module-CAEN_V775.hh"

using namespace std;


static TRoomVmeModuleCreator Creator1("CAEN_V775", new TVmeTDC_CAEN_V775(32));
static TRoomVmeModuleCreator Creator2("CAEN_V775N", new TVmeTDC_CAEN_V775(16));


TVmeTDC_CAEN_V775::TVmeTDC_CAEN_V775(int NumberOfChannels)
: TRoomVmeModule("VmeTDC", "CAEN_V775", (TRoomVmeTransferMode) _TransferMode, (size_t) _MapSize)
{
    _NumberOfChannels = NumberOfChannels;

    _AddressBitLength = 0;
    while (NumberOfChannels > 1) {
	_AddressBitLength++;
	NumberOfChannels /= 2;
    }

    _InterruptLevel = 0;
    _InterruptVector = 0;
    _EventTriggerThreshold = 0;
    _IsEventTriggerSet = false;

    _SingleReadBuffer = new U32bit[_NumberOfChannels];
    _SingleReadFlags = ~0;
    _EventCounterValue = 0;
}

TVmeTDC_CAEN_V775::~TVmeTDC_CAEN_V775()
{
    delete[] _SingleReadBuffer;
}

TRoomVmeModule* TVmeTDC_CAEN_V775::Clone(void)
{
    return new TVmeTDC_CAEN_V775(_NumberOfChannels);
}

int TVmeTDC_CAEN_V775::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}

int TVmeTDC_CAEN_V775::AddressBitLength(void)
{
    return _AddressBitLength;
}

int TVmeTDC_CAEN_V775::DataBitLength(void)
{
    return _DataBitLength;
}

bool TVmeTDC_CAEN_V775::Probe(void) throw(THardwareException)
{
    unsigned BoardId = 0;
    BoardId += (WordAt(regBoardIdMSB) & 0xff) << 16;
    BoardId += (WordAt(regBoardId) & 0xff) << 8;
    BoardId += (WordAt(regBoardIdLSB) & 0xff) << 0;

    return BoardId == 0x000307;
}

int TVmeTDC_CAEN_V775::Initialize(int InitialState) throw(THardwareException)
{
    WordAt(regCrateSelect) = 0x0000;
    WordAt(regGeoAddress) = 0x0000;

    unsigned RegisterAddress;
    for (int Channel = 0; Channel < _NumberOfChannels; Channel++) {
        if (_NumberOfChannels == 16) {
	    // model V775N //
            RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Channel;
	}
	else {
	    // model V775 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Channel;
	}
	WordAt(RegisterAddress) = 0x0000;
    }
    
    WordAt(regBitSet1) = bitSoftwareReset;
    WordAt(regBitClear1) = bitSoftwareReset;

    // select common-start mode //
    WordAt(regBitClear2) = bitStartStop;

    WordAt(regBitSet2) = bitClearData;
    WordAt(regBitClear2) = bitClearData;

    _SingleReadFlags = ~0;
    _EventCounterValue = 0;
    
    return 1;
}

int TVmeTDC_CAEN_V775::Finalize(int FinalState) throw(THardwareException)
{
    return 1;
}

int TVmeTDC_CAEN_V775::Enable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	for (int i = 0; i < _NumberOfChannels; i++) {
	    Enable(i);
	}
    }
    else if (Address < _NumberOfChannels) {
        unsigned RegisterAddress;
        if (_NumberOfChannels == 16) {
	    // model V775N //
	    RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Address;
	}
	else {
	    // model V775 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Address;
	}
	WordAt(RegisterAddress) &= ~bitKill;
    }

    return 1;
}

int TVmeTDC_CAEN_V775::Disable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	for (int i = 0; i < _NumberOfChannels; i++) {
	    Disable(i);
	}
    }
    else if (Address < _NumberOfChannels) {
        unsigned RegisterAddress;
        if (_NumberOfChannels == 16) {
	    // model V775N //
	    RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Address;
	}
	else {
	    // model V775 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Address;
	}
	WordAt(RegisterAddress) |= bitKill;
    }

    return 1;
}

int TVmeTDC_CAEN_V775::Clear(int Address) throw(THardwareException)
{
    WordAt(regBitSet2) = bitClearData;
    WordAt(regBitClear2) = bitClearData;

    _SingleReadFlags = ~0;

    return 1;
}

bool TVmeTDC_CAEN_V775::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    return TRoomModule::WaitData(TimeOut_sec);
}

bool TVmeTDC_CAEN_V775::HasData(int Address) throw(THardwareException)
{
    if (_IsEventTriggerSet) {
	return (WordAt(regStatus1) & bitEventReady) != 0;
    }
    else {
	return (WordAt(regStatus1) & bitDataReady) != 0;
    }
}

int TVmeTDC_CAEN_V775::Read(int Address, int &Data) throw(THardwareException)
{
    unsigned AddressBit = Bit(Address);
    U32bit Word;

    // second readout to the same channel triggers new data block readout //
    if (_SingleReadFlags & AddressBit) {
	if ((WordAt(regStatus1) & bitDataReady) == 0) {
	    return 0;
	}

	// read the header word //
	Word = DoubleWordAt(0x0000);
	if (((Word >> 24) & 0x07) != 0x02) {
	    throw THardwareException(
		_ModelName + "::Read()", "invalid data word (header)"
	    );
	}
	// read data elements //
	_NumberOfSingleReadWords = ((Word >> 8) & 0x3f);
	for (int Index = 0; Index < _NumberOfSingleReadWords; Index++) {
	    _SingleReadBuffer[Index] = DoubleWordAt(0x0000);
	}
	// read the trailer word //
	Word = DoubleWordAt(0x0000);
	if (((Word >> 24) & 0x07) != 0x04) {
	    throw THardwareException(
		_ModelName + "::Read()", "invalid data word (trailer)"
	    );
	}
	_EventCounterValue = Word & 0x00ffffff;

	_SingleReadFlags = 0;
    }
    _SingleReadFlags |= AddressBit;

    if (Address >= _NumberOfChannels) {
	if (Address == _NumberOfChannels) {
	    Data = (_EventCounterValue >> 0) & 0x0fff;
	}
	else if (Address == _NumberOfChannels+1) {
	    Data = (_EventCounterValue >> 12) & 0x0fff;
	}
	else {
	    Data = 0;
	}
	return 1;
    }

    if (_NumberOfChannels == 16) {
	// model V775N //
	Address <<= 1;
    }

    Data = 0;
    for (int Index = 0; Index < _NumberOfSingleReadWords; Index++) {
	Word = _SingleReadBuffer[Index];
	if (((Word >> 16) & 0x00ff) == (unsigned) Address) {
            if (Word & 0x1000) {
	        // overflow //
	        Data = 0x0fff;
	    }
            else if ((Word & 0x4000) == 0) {
	        // not-valid //
                Data = 0xffff;
	    }
	    else {
	        Data = (Word & 0x0fff);
	    }
	    break;
	}
    }
    
    return 1;
}

int TVmeTDC_CAEN_V775::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _EventBufferSize;
}

int TVmeTDC_CAEN_V775::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    int WordIndex = 0;
    while ((WordAt(regStatus2) & bitBufferEmpty) == 0) {
	((U32bit*) Data)[WordIndex++] = DoubleWordAt(0x0000);
    }

    return WordIndex * sizeof(U32bit);
}

int TVmeTDC_CAEN_V775::WriteRegister(int Address, int Data) throw(THardwareException)
{
    WordAt(Address) = Data;
    return 1;
}

int TVmeTDC_CAEN_V775::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    Data = WordAt(Address) & 0xffff;
    return 1;
}

int TVmeTDC_CAEN_V775::EnableInterrupt(int SignalId) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    _InterruptLevel = VmeAccessProperty()->InterruptNumber;
    _InterruptVector = VmeAccessProperty()->InterruptVector;

    if (_InterruptLevel == 0) {
        return 0;
    }

    TRoomVmeModule::EnableInterrupt(SignalId);

    WordAt(regInterruptVector) = _InterruptVector & 0x00ff;
    WordAt(regInterruptLevel) = _InterruptLevel & 0x0007;
#endif

    return 0;
}

int TVmeTDC_CAEN_V775::DisableInterrupt(void) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    if (_InterruptLevel == 0) {
        return 0;
    }

    WordAt(regInterruptLevel) = 0;
    TRoomVmeModule::DisableInterrupt();
#endif

    return 0;
}

int TVmeTDC_CAEN_V775::ClearInterrupt(void) throw(THardwareException)
{
    return 0;
}

void TVmeTDC_CAEN_V775::EnableServiceRequest(void) throw(THardwareException)
{
    if (! _IsEventTriggerSet) {
	_IsEventTriggerSet = true;
	if (_EventTriggerThreshold == 0) {
	    _EventTriggerThreshold = 1;
	}
	WordAt(regEventTrigger) = _EventTriggerThreshold;
    }
}

void TVmeTDC_CAEN_V775::DisableServiceRequest(void) throw(THardwareException)
{
    WordAt(regEventTrigger) = 0;
    _IsEventTriggerSet = false;
}

void TVmeTDC_CAEN_V775::ClearServiceRequest(void) throw(THardwareException)
{
}

bool TVmeTDC_CAEN_V775::IsRequestingService(void) throw(THardwareException)
{
    return (WordAt(regStatus1) & bitEventReady) != 0;
}

bool TVmeTDC_CAEN_V775::IsSignalOnServiceRequestAvailable(void)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    return true;
#else
    return false;
#endif
}

int TVmeTDC_CAEN_V775::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;

    if (CommandName == "readEventCounter") {
	ControlId = ControlId_ReadEventCounter;
    }
    else if (CommandName == "readEventCounterOfThisRead") {
	ControlId = ControlId_ReadEventCounterOfThisRead;
    }
    else if (CommandName == "resetEventCounter") {
	ControlId = ControlId_ResetEventCounter;
    }
    else if (CommandName == "setThreshold") {
	ControlId = ControlId_SetThreshold;
    }
    else if (CommandName == "setEventTrigger") {
	ControlId = ControlId_SetEventTrigger;
    }
    else if (CommandName == "setFastClearWindow") {
	ControlId = ControlId_SetFastClearWindow;
    }
    else if (CommandName == "setFullScaleRange") {
	ControlId = ControlId_SetFullScaleRange;
    }
    else if (CommandName == "selectCommonStartMode") {
	ControlId = ControlId_SelectCommonStartMode;
    }
    else if (CommandName == "selectCommonStopMode") {
	ControlId = ControlId_SelectCommonStopMode;
    }
    else if (CommandName == "enableZeroSuppression") {
	ControlId = ControlId_EnableZeroSuppression;
    }
    else if (CommandName == "disableZeroSuppression") {
	ControlId = ControlId_DisableZeroSuppression;
    }
    else if (CommandName == "enableOverflowSuppression") {
	ControlId = ControlId_EnableOverflowSuppression;
    }
    else if (CommandName == "disableOverflowSuppression") {
	ControlId = ControlId_DisableOverflowSuppression;
    }
    else if (CommandName == "enableNotValidDataSuppression") {
	ControlId = ControlId_EnableNotValidDataSuppression;
    }
    else if (CommandName == "disableNotValidDataSuppression") {
	ControlId = ControlId_DisableNotValidDataSuppression;
    }
    else if (CommandName == "enableEmptyEventRecording") {
	ControlId = ControlId_EnableEmptyEventRecording;
    }
    else if (CommandName == "disableEmptyEventRecording") {
	ControlId = ControlId_DisableEmptyEventRecording;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TVmeTDC_CAEN_V775::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    int Result = 1;
    
    switch (ControlId) {
      case ControlId_ReadEventCounter:
	if (NumberOfArguments > 0) {
	    ArgumentList[0] = (WordAt(regEventCounterHigh) & 0x00ff) << 16;
	    ArgumentList[0] |= (WordAt(regEventCounterLow) & 0xffff) << 0;
	}
	break;
      case ControlId_ReadEventCounterOfThisRead:
	if (NumberOfArguments > 0) {
	    ArgumentList[0] = _EventCounterValue & 0x00ffffff;
	}
	break;
      case ControlId_ResetEventCounter:
        WordAt(regEventCounterReset) = 0x0001;
	break;
      case ControlId_SetThreshold:
	Result = SetThreshold(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetEventTrigger:
	Result = SetEventTrigger(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetFastClearWindow:
	Result = SetFastClearWindow(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetFullScaleRange:
	Result = SetFullScaleRange(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SelectCommonStartMode:
	WordAt(regBitClear2) = bitStartStop;
	break;
      case ControlId_SelectCommonStopMode:
	WordAt(regBitSet2) = bitStartStop;
	break;
      case ControlId_EnableZeroSuppression:
	WordAt(regBitClear2) = bitLowThresholdEnable;
	break;
      case ControlId_DisableZeroSuppression:
	WordAt(regBitSet2) = bitLowThresholdEnable;
	break;
      case ControlId_EnableOverflowSuppression:
	WordAt(regBitClear2) = bitOverRangeEnable;
	break;
      case ControlId_DisableOverflowSuppression:
	WordAt(regBitSet2) = bitOverRangeEnable;
	break;
      case ControlId_EnableNotValidDataSuppression:
	WordAt(regBitClear2) = bitValidControl;
	break;
      case ControlId_DisableNotValidDataSuppression:
	WordAt(regBitSet2) = bitValidControl;
	break;
      case ControlId_EnableEmptyEventRecording:
	WordAt(regBitSet2) = bitEmptyEnable;
	break;
      case ControlId_DisableEmptyEventRecording:
	WordAt(regBitClear2) = bitEmptyEnable;
	break;
      default:
	Result = 0;
    }

    return Result;
}

int TVmeTDC_CAEN_V775::SetThreshold(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 2) {
	throw THardwareException(
	    _ModelName + "::setThreshold(address, value)", 
	    "too few argument[s]"
	);
    }
    int Channel = ArgumentList[0];
    if ((Channel < 0) || (Channel >= _NumberOfChannels)) {
        return 0;
    }

    unsigned RegisterAddress;
    if (_NumberOfChannels == 16) {
	// model V775N //
	RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Channel;
    }
    else {
	// model V775 //
	RegisterAddress = regThresholdBase + sizeof(U16bit) * Channel;
    }

    unsigned Value = ((unsigned) ArgumentList[1] >> 4) & 0x00ff;

    WordAt(RegisterAddress) |= Value;

    return 1;
}

int TVmeTDC_CAEN_V775::SetEventTrigger(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setEventTrigger(number_of_events)", 
	    "too few argument[s]"
	);
    }

    if (ArgumentList[0] == 0) {
	_EventTriggerThreshold = 0;
	WordAt(regEventTrigger) = 0;
        _IsEventTriggerSet = false;
    }
    else if (ArgumentList[0] <= _MaxNumberOfEvents) {
	_EventTriggerThreshold = ArgumentList[0];
	WordAt(regEventTrigger) = _EventTriggerThreshold;
        _IsEventTriggerSet = true;
    }
    else {
	return 0;
    }

    return 1;
}

int TVmeTDC_CAEN_V775::SetFastClearWindow(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setFastClearWindow(length)", 
	    "too few argument[s]"
	);
    }

    if ((ArgumentList[0] > 0) && (ArgumentList[0] <= 0x3f0)) {
	WordAt(regFastClearWindow) = ArgumentList[0] & 0x3ff;
	return 1;
    }
    else {
	return 0;
    }
}

int TVmeTDC_CAEN_V775::SetFullScaleRange(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setFullScaleRange(full_scale_range)", 
	    "too few argument[s]"
	);
    }

    if ((ArgumentList[0] > 0) && (ArgumentList[0] <= 0xff)) {
	WordAt(regFullScaleRange) = ArgumentList[0] & 0xff;
	return 1;
    }
    else {
	return 0;
    }
}


#if 0
datasource VmeTdc
{
    int base_address = 0x07750000;
    int readout_channels = #0..#3;

    VmeCrate crate;
    VmeController controller("SBS-620");
    VmeModule tdc("CAEN-V775N");
    
    crate.installController(controller);
    crate.installModule(tdc, base_address);

    on run_begin {
	// default mode is COMMON START //
        //tdc.selectCommonStopMode();

	tdc.disable(~readout_channels);  


        tdc.disableZeroSuppression();
        tdc.disableOverflowSuppression();
	tdc.disableNotValidDataSuppression();
        //tdc.setThreshold(0, 200);
	//tdc.setEventTrigger(16);
    };

    on trigger(tdc) {
        tdc.read(readout_channels);
	//tdc.blockRead();
    }
}
#endif
