/* module-CAEN_V792.cc */
/* Created by Enomoto Sanshiro on 24 April 2005 */
/* Last updated by Enomoto Sanshiro on 9 November 2005. */


#include <iostream>
#include <iomanip>

#include "RoomDeviceFactory.hh"
#include "RoomVmeAccess.hh"
#include "module-CAEN_V792.hh"

using namespace std;


static TRoomVmeModuleCreator Creator1("CAEN_V792", new TVmeQADC_CAEN_V792(32));
static TRoomVmeModuleCreator Creator2("CAEN_V792N", new TVmeQADC_CAEN_V792(16));


TVmeQADC_CAEN_V792::TVmeQADC_CAEN_V792(int NumberOfChannels)
: TRoomVmeModule("VmeQADC", "CAEN_V792", (TRoomVmeTransferMode) _TransferMode, (size_t) _MapSize)
{
    _NumberOfChannels = NumberOfChannels;

    _AddressBitLength = 0;
    while (NumberOfChannels > 1) {
	_AddressBitLength++;
	NumberOfChannels /= 2;
    }

    _InterruptLevel = 0;
    _InterruptVector = 0;
    _EventTriggerThreshold = 0;
    _IsEventTriggerSet = false;

    _SingleReadBuffer = new U32bit[_NumberOfChannels];
    _SingleReadFlags = ~0;
    _EventCounterValue = 0;
}

TVmeQADC_CAEN_V792::~TVmeQADC_CAEN_V792()
{
    delete[] _SingleReadBuffer;
}

TRoomVmeModule* TVmeQADC_CAEN_V792::Clone(void)
{
    return new TVmeQADC_CAEN_V792(_NumberOfChannels);
}

int TVmeQADC_CAEN_V792::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}

int TVmeQADC_CAEN_V792::AddressBitLength(void)
{
    return _AddressBitLength;
}

int TVmeQADC_CAEN_V792::DataBitLength(void)
{
    return _DataBitLength;
}

bool TVmeQADC_CAEN_V792::Probe(void) throw(THardwareException)
{
    unsigned BoardId = 0;
    BoardId += (WordAt(regBoardIdMSB) & 0xff) << 16;
    BoardId += (WordAt(regBoardId) & 0xff) << 8;
    BoardId += (WordAt(regBoardIdLSB) & 0xff) << 0;

    return BoardId == 0x000318;
}

int TVmeQADC_CAEN_V792::Initialize(int InitialState) throw(THardwareException)
{
    WordAt(regCrateSelect) = 0x0000;
    WordAt(regGeoAddress) = 0x0000;

    unsigned RegisterAddress;
    for (int Channel = 0; Channel < _NumberOfChannels; Channel++) {
        if (_NumberOfChannels == 16) {
	    // model V792N //
            RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Channel;
	}
	else {
	    // model V792 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Channel;
	}
	WordAt(RegisterAddress) = 0x0000;
    }
    
    WordAt(regBitSet1) = bitSoftwareReset;
    WordAt(regBitClear1) = bitSoftwareReset;

    _SingleReadFlags = ~0;
    _EventCounterValue = 0;
    
    return 1;
}

int TVmeQADC_CAEN_V792::Finalize(int FinalState) throw(THardwareException)
{
    return 1;
}

int TVmeQADC_CAEN_V792::Enable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	for (int i = 0; i < _NumberOfChannels; i++) {
	    Enable(i);
	}
    }
    else if (Address < _NumberOfChannels) {
        unsigned RegisterAddress;
        if (_NumberOfChannels == 16) {
	    // model V792N //
	    RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Address;
	}
	else {
	    // model V792 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Address;
	}
	WordAt(RegisterAddress) &= ~bitKill;
    }

    return 1;
}

int TVmeQADC_CAEN_V792::Disable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	for (int i = 0; i < _NumberOfChannels; i++) {
	    Disable(i);
	}
    }
    else if (Address < _NumberOfChannels) {
        unsigned RegisterAddress;
        if (_NumberOfChannels == 16) {
	    // model V792N //
	    RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Address;
	}
	else {
	    // model V792 //
	    RegisterAddress = regThresholdBase + sizeof(U16bit) * Address;
	}
	WordAt(RegisterAddress) |= bitKill;
    }

    return 1;
}

int TVmeQADC_CAEN_V792::Clear(int Address) throw(THardwareException)
{
    WordAt(regBitSet2) = bitClearData;
    WordAt(regBitClear2) = bitClearData;

    _SingleReadFlags = ~0;

    return 1;
}

bool TVmeQADC_CAEN_V792::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    return TRoomModule::WaitData(TimeOut_sec);
}

bool TVmeQADC_CAEN_V792::HasData(int Address) throw(THardwareException)
{
    if (_IsEventTriggerSet) {
	return (WordAt(regStatus1) & bitEventReady) != 0;
    }
    else {
	return (WordAt(regStatus1) & bitDataReady) != 0;
    }
}

int TVmeQADC_CAEN_V792::Read(int Address, int &Data) throw(THardwareException)
{
    unsigned AddressBit = Bit(Address);
    U32bit Word;

    // second readout to the same channel triggers new data block readout //
    if (_SingleReadFlags & AddressBit) {
	if ((WordAt(regStatus1) & bitDataReady) == 0) {
	    return 0;
	}

	// read the header word //
	Word = DoubleWordAt(0x0000);
	if (((Word >> 24) & 0x07) != 0x02) {
	    throw THardwareException(
		_ModelName + "::Read()", "invalid data word (header)"
	    );
	}
	// read data elements //
	_NumberOfSingleReadWords = ((Word >> 8) & 0x3f);
	for (int Index = 0; Index < _NumberOfSingleReadWords; Index++) {
	    _SingleReadBuffer[Index] = DoubleWordAt(0x0000);
	}
	// read the trailer word //
	Word = DoubleWordAt(0x0000);
	if (((Word >> 24) & 0x07) != 0x04) {
	    throw THardwareException(
		_ModelName + "::Read()", "invalid data word (trailer)"
	    );
	}
	_EventCounterValue = Word & 0x00ffffff;

	_SingleReadFlags = 0;
    }
    _SingleReadFlags |= AddressBit;

    if (Address >= _NumberOfChannels) {
	if (Address == _NumberOfChannels) {
	    Data = (_EventCounterValue >> 0) & 0x0fff;
	}
	else if (Address == _NumberOfChannels+1) {
	    Data = (_EventCounterValue >> 12) & 0x0fff;
	}
	else {
	    Data = 0;
	}
	return 1;
    }

    if (_NumberOfChannels == 16) {
	// model V792N //
	Address <<= 1;
    }

    Data = 0;
    for (int Index = 0; Index < _NumberOfSingleReadWords; Index++) {
	Word = _SingleReadBuffer[Index];
	if (((Word >> 16) & 0x00ff) == (unsigned) Address) {
            if (Word & 0x1000) {
	        // overflow //
	        Data = 0x0fff;
	    }
	    else {
	        Data = (Word & 0x0fff);
	    }
	    break;
	}
    }
    
    return 1;
}

int TVmeQADC_CAEN_V792::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _EventBufferSize;
}

int TVmeQADC_CAEN_V792::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    int WordIndex = 0;
    while ((WordAt(regStatus2) & bitBufferEmpty) == 0) {
	((U32bit*) Data)[WordIndex++] = DoubleWordAt(0x0000);
    }

    return WordIndex * sizeof(U32bit);
}

int TVmeQADC_CAEN_V792::WriteRegister(int Address, int Data) throw(THardwareException)
{
    WordAt(Address) = Data;
    return 1;
}

int TVmeQADC_CAEN_V792::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    Data = WordAt(Address) & 0xffff;
    return 1;
}

int TVmeQADC_CAEN_V792::EnableInterrupt(int SignalId) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    _InterruptLevel = VmeAccessProperty()->InterruptNumber;
    _InterruptVector = VmeAccessProperty()->InterruptVector;

    if (_InterruptLevel == 0) {
        return 0;
    }

    TRoomVmeModule::EnableInterrupt(SignalId);

    WordAt(regInterruptVector) = _InterruptVector & 0x00ff;
    WordAt(regInterruptLevel) = _InterruptLevel & 0x0007;
#endif

    return 0;
}

int TVmeQADC_CAEN_V792::DisableInterrupt(void) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    if (_InterruptLevel == 0) {
        return 0;
    }

    WordAt(regInterruptLevel) = 0;
    TRoomVmeModule::DisableInterrupt();
#endif

    return 0;
}

int TVmeQADC_CAEN_V792::ClearInterrupt(void) throw(THardwareException)
{
    return 0;
}

void TVmeQADC_CAEN_V792::EnableServiceRequest(void) throw(THardwareException)
{
    if (! _IsEventTriggerSet) {
	_IsEventTriggerSet = true;
	if (_EventTriggerThreshold == 0) {
	    _EventTriggerThreshold = 1;
	}
	WordAt(regEventTrigger) = _EventTriggerThreshold;
    }
}

void TVmeQADC_CAEN_V792::DisableServiceRequest(void) throw(THardwareException)
{
    WordAt(regEventTrigger) = 0;
    _IsEventTriggerSet = false;
}

void TVmeQADC_CAEN_V792::ClearServiceRequest(void) throw(THardwareException)
{
}

bool TVmeQADC_CAEN_V792::IsRequestingService(void) throw(THardwareException)
{
    return (WordAt(regStatus1) & bitEventReady) != 0;
}

bool TVmeQADC_CAEN_V792::IsSignalOnServiceRequestAvailable(void)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    return true;
#else
    return false;
#endif
}

int TVmeQADC_CAEN_V792::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;

    if (CommandName == "readEventCounter") {
	ControlId = ControlId_ReadEventCounter;
    }
    else if (CommandName == "readEventCounterOfThisRead") {
	ControlId = ControlId_ReadEventCounterOfThisRead;
    }
    else if (CommandName == "resetEventCounter") {
	ControlId = ControlId_ResetEventCounter;
    }
    else if (CommandName == "setThreshold") {
	ControlId = ControlId_SetThreshold;
    }
    else if (CommandName == "setEventTrigger") {
	ControlId = ControlId_SetEventTrigger;
    }
    else if (CommandName == "setFastClearWindow") {
	ControlId = ControlId_SetFastClearWindow;
    }
    else if (CommandName == "setPedestalCurrent") {
	ControlId = ControlId_SetPedestalCurrent;
    }
    else if (CommandName == "enableZeroSuppression") {
	ControlId = ControlId_EnableZeroSuppression;
    }
    else if (CommandName == "disableZeroSuppression") {
	ControlId = ControlId_DisableZeroSuppression;
    }
    else if (CommandName == "enableOverflowSuppression") {
	ControlId = ControlId_EnableOverflowSuppression;
    }
    else if (CommandName == "disableOverflowSuppression") {
	ControlId = ControlId_DisableOverflowSuppression;
    }
    else if (CommandName == "enableEmptyEventRecording") {
	ControlId = ControlId_EnableEmptyEventRecording;
    }
    else if (CommandName == "disableEmptyEventRecording") {
	ControlId = ControlId_DisableEmptyEventRecording;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TVmeQADC_CAEN_V792::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    int Result = 1;
    
    switch (ControlId) {
      case ControlId_ReadEventCounter:
	if (NumberOfArguments > 0) {
	    ArgumentList[0] = (WordAt(regEventCounterHigh) & 0x00ff) << 16;
	    ArgumentList[0] |= (WordAt(regEventCounterLow) & 0xffff) << 0;
	}
	break;
      case ControlId_ReadEventCounterOfThisRead:
	if (NumberOfArguments > 0) {
	    ArgumentList[0] = _EventCounterValue & 0x00ffffff;
	}
	break;
      case ControlId_ResetEventCounter:
        WordAt(regEventCounterReset) = 0x0001;
	break;
      case ControlId_SetThreshold:
	Result = SetThreshold(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetEventTrigger:
	Result = SetEventTrigger(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetFastClearWindow:
	Result = SetFastClearWindow(ArgumentList, NumberOfArguments);
	break;
      case ControlId_SetPedestalCurrent:
	Result = SetPedestalCurrent(ArgumentList, NumberOfArguments);
	break;
      case ControlId_EnableZeroSuppression:
	WordAt(regBitClear2) = bitLowThresholdEnable;
	break;
      case ControlId_DisableZeroSuppression:
	WordAt(regBitSet2) = bitLowThresholdEnable;
	break;
      case ControlId_EnableOverflowSuppression:
	WordAt(regBitClear2) = bitOverRangeEnable;
	break;
      case ControlId_DisableOverflowSuppression:
	WordAt(regBitSet2) = bitOverRangeEnable;
	break;
      case ControlId_EnableEmptyEventRecording:
	WordAt(regBitSet2) = bitEmptyEnable;
	break;
      case ControlId_DisableEmptyEventRecording:
	WordAt(regBitClear2) = bitEmptyEnable;
	break;
      default:
	Result = 0;
    }

    return Result;
}

int TVmeQADC_CAEN_V792::SetThreshold(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 2) {
	throw THardwareException(
	    _ModelName + "::setThreshold(address, value)", 
	    "too few argument[s]"
	);
    }
    int Channel = ArgumentList[0];
    if ((Channel < 0) || (Channel >= _NumberOfChannels)) {
        return 0;
    }

    unsigned RegisterAddress;
    if (_NumberOfChannels == 16) {
	// model V792N //
	RegisterAddress = regThresholdBase + 2 * sizeof(U16bit) * Channel;
    }
    else {
	// model V792 //
	RegisterAddress = regThresholdBase + sizeof(U16bit) * Channel;
    }

    unsigned Value = ((unsigned) ArgumentList[1] >> 4) & 0x00ff;

    WordAt(RegisterAddress) |= Value;

    return 1;
}

int TVmeQADC_CAEN_V792::SetEventTrigger(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setEventTrigger(number_of_events)", 
	    "too few argument[s]"
	);
    }

    if (ArgumentList[0] == 0) {
	_EventTriggerThreshold = 0;
	WordAt(regEventTrigger) = 0;
        _IsEventTriggerSet = false;
    }
    else if (ArgumentList[0] <= _MaxNumberOfEvents) {
	_EventTriggerThreshold = ArgumentList[0];
	WordAt(regEventTrigger) = _EventTriggerThreshold;
        _IsEventTriggerSet = true;
    }
    else {
	return 0;
    }

    return 1;
}

int TVmeQADC_CAEN_V792::SetFastClearWindow(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setFastClearWindow(length)", 
	    "too few argument[s]"
	);
    }

    if ((ArgumentList[0] > 0) && (ArgumentList[0] <= 0x3f0)) {
	WordAt(regFastClearWindow) = ArgumentList[0] & 0x3ff;
	return 1;
    }
    else {
	return 0;
    }
}

int TVmeQADC_CAEN_V792::SetPedestalCurrent(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setPedestalCurrent(pedestal_current)", 
	    "too few argument[s]"
	);
    }

    if ((ArgumentList[0] > 0) && (ArgumentList[0] <= 0xff)) {
	WordAt(regPedestalCurrent) = ArgumentList[0] & 0xff;
	return 1;
    }
    else {
	return 0;
    }
}



#if 0
datasource VmeAdc
{
    int base_address = 0x07920000;
    int readout_channels = #0..#3;

    VmeCrate crate;
    VmeController controller("SBS-620");
    VmeModule adc("CAEN-V792N");
    
    crate.installController(controller);
    crate.installModule(adc, base_address);

    on run_begin {
	adc.disable(~readout_channels);  

        adc.disableZeroSuppression();
        adc.disableOverflowSuppression();
    };

    on trigger(adc) {
        adc.read(readout_channels);
	//adc.blockRead();
    }
}
#endif
