/* module-CAEN_V830.cc */
/* Created by Enomoto Sanshiro on 22 December 2005 */
/* Last updated by Enomoto Sanshiro on 28 December 2005. */


#include <iostream>
#include <iomanip>

#include "RoomDeviceFactory.hh"
#include "RoomVmeAccess.hh"
#include "module-CAEN_V830.hh"

using namespace std;


static TRoomVmeModuleCreator Creator1(
    "CAEN_V830", new TVmeScaler_CAEN_V830(TVmeScaler_CAEN_V830::Model_V830)
);
static TRoomVmeModuleCreator Creator2(
    "CAEN_V820", new TVmeScaler_CAEN_V830(TVmeScaler_CAEN_V830::Model_V820)
);


TVmeScaler_CAEN_V830::TVmeScaler_CAEN_V830(int ModelNumber)
: TRoomVmeModule("VmeScaler", "CAEN_V830", (TRoomVmeTransferMode) _TransferMode, (size_t) _MapSize)
{
    _InterruptLevel = 0;
    _InterruptVector = 0;

    _SingleReadBuffer = new U32bit[_NumberOfChannels];
    _SingleReadFlags = 0xffffffff;;

    _EnabledChannelBitList = 0xffffffff;;
    _IsHeaderEnabled = false;
    _Is32bitFormatSelected = true;

    _ModelNumber = ModelNumber;
    _IsDirectReadSelected = (_ModelNumber == Model_V820);

    _AlmostFullLevel = _BufferWindowSize / sizeof(U32bit);
    _LocalBuffer = new U32bit[_AlmostFullLevel];
    _NumberOfLocalBufferWords = 0;
}

TVmeScaler_CAEN_V830::~TVmeScaler_CAEN_V830()
{
    delete[] _SingleReadBuffer;
    delete[] _LocalBuffer;
}

TRoomVmeModule* TVmeScaler_CAEN_V830::Clone(void)
{
    return new TVmeScaler_CAEN_V830(_ModelNumber);
}

int TVmeScaler_CAEN_V830::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}

int TVmeScaler_CAEN_V830::AddressBitLength(void)
{
    return _AddressBitLength;
}

int TVmeScaler_CAEN_V830::DataBitLength(void)
{
    return _DataBitLength;
}

bool TVmeScaler_CAEN_V830::Probe(void) throw(THardwareException)
{
    unsigned BoardId = 0;
    BoardId += (WordAt(regBoardIdMSB) & 0xff) << 16;
    BoardId += (WordAt(regBoardId) & 0xff) << 8;
    BoardId += (WordAt(regBoardIdLSB) & 0xff) << 0;

    return (BoardId == 0x000334) || (BoardId == 0x00033e);
}

int TVmeScaler_CAEN_V830::Initialize(int InitialState) throw(THardwareException)
{
    _AlmostFullLevel = _BufferWindowSize / sizeof(U32bit);
    WordAt(regAlmostFullLevel) = _AlmostFullLevel;
    _NumberOfLocalBufferWords = 0;

    _EnabledChannelBitList = 0xfffffffful;
    DoubleWordAt(regChannelEnable) = _EnabledChannelBitList;

    _IsHeaderEnabled = false;
    _Is32bitFormatSelected = true;
    _IsDirectReadSelected = (_ModelNumber == Model_V820);

    WordAt(regControl) = bitAcqMode_Disabled | bitDataFormat_32bit;

    _SingleReadFlags = 0xffffffff;;
    
    WordAt(regSoftwareClear) = 1;

    return 1;
}

int TVmeScaler_CAEN_V830::Finalize(int FinalState) throw(THardwareException)
{
    return 1;
}

int TVmeScaler_CAEN_V830::Enable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	_EnabledChannelBitList = 0xfffffffful;
    }
    else if (Address < _NumberOfChannels) {
	_EnabledChannelBitList |= Bit(Address);
    }

    DoubleWordAt(regChannelEnable) = _EnabledChannelBitList;

    return 1;
}

int TVmeScaler_CAEN_V830::Disable(int Address) throw(THardwareException)
{
    if (Address < 0) {
	_EnabledChannelBitList = 0;
    }
    else if (Address < _NumberOfChannels) {
	_EnabledChannelBitList &= ~Bit(Address);
    }

    DoubleWordAt(regChannelEnable) = _EnabledChannelBitList;

    return 1;
}

int TVmeScaler_CAEN_V830::Clear(int Address) throw(THardwareException)
{
    WordAt(regSoftwareClear) = 1;
    _SingleReadFlags = 0xffffffff;;
    _NumberOfLocalBufferWords = 0;

    return 1;
}

bool TVmeScaler_CAEN_V830::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    return TRoomModule::WaitData(TimeOut_sec);
}

bool TVmeScaler_CAEN_V830::HasData(int Address) throw(THardwareException)
{
    return (WordAt(regStatus) & bitDataReady) != 0;
}

int TVmeScaler_CAEN_V830::Read(int Address, int &Data) throw(THardwareException)
{
    if ((Address < 0)  || (Address >= _NumberOfChannels)) {
	return 0;
    }

    if ((_ModelNumber == Model_V820) || _IsDirectReadSelected) {
	Data = DoubleWordAt(regCounterBase + 4*Address);
	return 1;
    }

    U32bit AddressBit = Bit(Address);

    // second readout to the same channel triggers new data block readout //
    if (_SingleReadFlags & AddressBit) {
	if (
	    (_NumberOfLocalBufferWords == 0) && 
	    ((WordAt(regStatus) & bitDataReady) == 0)
	){
	    return 0;
	}

	// read the header word //
	if (_IsHeaderEnabled) {
	    _HeaderValue = GetNextDataWord();
	}
	else {
	    _HeaderValue = 0;
	}

	// read data elements //
	for (int Channel = 0; Channel < _NumberOfChannels; Channel++) {
	    if (Bit(Channel) & _EnabledChannelBitList) {
		_SingleReadBuffer[Channel] = GetNextDataWord();
	    }
	    else {
		_SingleReadBuffer[Channel] = 0;
	    }
	}
	_SingleReadFlags = 0;
    }
    _SingleReadFlags |= AddressBit;

    if (Address >= _NumberOfChannels) {
	if (Address == _NumberOfChannels) {
	    Data = (_HeaderValue) & 0x0000fffful;
	}
	else {
	    Data = 0;
	}
	return 1;
    }

    Data = _SingleReadBuffer[Address];

    return 1;
}

int TVmeScaler_CAEN_V830::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _BufferWindowSize;
}

int TVmeScaler_CAEN_V830::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    if (WordAt(regStatus) & bitAlmostFull) {
	return DmaRead(regEventBuffer, Data, _BufferWindowSize);
    }
    else {
	int WordIndex = 0;
	while ((WordAt(regStatus) & bitDataReady) != 0) {
	    ((U32bit*) Data)[WordIndex++] = DoubleWordAt(regEventBuffer);
	}	
	return WordIndex * sizeof(U32bit);
    }
}

int TVmeScaler_CAEN_V830::WriteRegister(int Address, int Data) throw(THardwareException)
{
    //...
    WordAt(Address) = Data;
    //DoubleWordAt(Address) = Data;

    return 1;
}

int TVmeScaler_CAEN_V830::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    //...
    Data = WordAt(Address) & 0xffff;
    //Data = DoubleWordAt(Address) & 0xffff;

    return 1;
}

int TVmeScaler_CAEN_V830::EnableInterrupt(int SignalId) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    _InterruptLevel = VmeAccessProperty()->InterruptNumber;
    _InterruptVector = VmeAccessProperty()->InterruptVector;

    if (_InterruptLevel == 0) {
        return 0;
    }

    TRoomVmeModule::EnableInterrupt(SignalId);

    WordAt(regInterruptVector) = _InterruptVector & 0x00ff;
    WordAt(regInterruptLevel) = _InterruptLevel & 0x0007;
#endif

    return 0;
}

int TVmeScaler_CAEN_V830::DisableInterrupt(void) throw(THardwareException)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    if (_InterruptLevel == 0) {
        return 0;
    }

    WordAt(regInterruptLevel) = 0;
    TRoomVmeModule::DisableInterrupt();
#endif

    return 0;
}

int TVmeScaler_CAEN_V830::ClearInterrupt(void) throw(THardwareException)
{
    return 0;
}

void TVmeScaler_CAEN_V830::EnableServiceRequest(void) throw(THardwareException)
{
}

void TVmeScaler_CAEN_V830::DisableServiceRequest(void) throw(THardwareException)
{
}

void TVmeScaler_CAEN_V830::ClearServiceRequest(void) throw(THardwareException)
{
}

bool TVmeScaler_CAEN_V830::IsRequestingService(void) throw(THardwareException)
{
    return (WordAt(regStatus) & bitDataReady) != 0;
}

bool TVmeScaler_CAEN_V830::IsSignalOnServiceRequestAvailable(void)
{
    // VME interrupt feature is disabled because IRQ cannot be 
    // automatically cleared during an acknowledge cycle.

#if 0
    return true;
#else
    return false;
#endif
}

int TVmeScaler_CAEN_V830::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;

    if (CommandName == "start") {
	ControlId = ControlId_Start;
    }
    else if (CommandName == "startPeriodical") {
	ControlId = ControlId_StartPeriodical;
    }
    else if (CommandName == "stop") {
	ControlId = ControlId_Stop;
    }
    else if (CommandName == "trigger") {
	ControlId = ControlId_Trigger;
    }
    else if (CommandName == "setDwellTime") {
	ControlId = ControlId_SetDwellTime;
    }
    else if (CommandName == "enableHeader") {
	ControlId = ControlId_EnableHeader;
    }
    else if (CommandName == "disableHeader") {
	ControlId = ControlId_DisableHeader;
    }
    else if (CommandName == "select32bitFormat") {
	ControlId = ControlId_Select32bitFormat;
    }
    else if (CommandName == "select26bitFormat") {
	ControlId = ControlId_Select26bitFormat;
    }
    else if (CommandName == "selectDirectRead") {
	ControlId = ControlId_SelectDirectRead;
    }
    else if (CommandName == "selectBufferedRead") {
	ControlId = ControlId_SelectBufferedRead;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TVmeScaler_CAEN_V830::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    int Result = 1;

    switch (ControlId) {
      case ControlId_Start:
	WordAt(regBitSet) = bitAcqMode_Random;
	break;
      case ControlId_StartPeriodical:
	WordAt(regBitSet) = bitAcqMode_Periodical;
	break;
      case ControlId_Stop:
	WordAt(regBitClear) = bitAcqMode_Random | bitAcqMode_Periodical;
	break;
      case ControlId_Trigger:
	WordAt(regSoftwareTrigger) = 1;
	break;
      case ControlId_SetDwellTime:
	Result = SetDwellTime(ArgumentList, NumberOfArguments);
	break;
      case ControlId_EnableHeader:
	WordAt(regBitSet) = bitHeaderEnable;
	_IsHeaderEnabled = true;
	break;
      case ControlId_DisableHeader:
	WordAt(regBitClear) = bitHeaderEnable;
	_IsHeaderEnabled = false;
	break;
      case ControlId_Select32bitFormat:
	WordAt(regBitClear) = bitDataFormat_26bit;
	_Is32bitFormatSelected = true;
	break;
      case ControlId_Select26bitFormat:
	WordAt(regBitSet) = bitDataFormat_26bit;
	_Is32bitFormatSelected = false;
	break;
      case ControlId_SelectDirectRead:
	_IsDirectReadSelected = true;
	break;
      case ControlId_SelectBufferedRead:
	_IsDirectReadSelected = false;
	break;
      default:
	Result = 0;
    }

    return Result;
}

int TVmeScaler_CAEN_V830::SetDwellTime(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setDwellTime(dwell_time_parameter)", 
	    "too few argument[s]"
	);
    }

    DoubleWordAt(regDwellTime) = (U32bit) ArgumentList[0];

    return 1;
}



#if 0
datasource VmeScaler
{
    int base_address = 0x08300000;
    int readout_channels = #0..#3;

    VmeCrate crate;
    VmeController controller("SBS-620");
    VmeModule scaler("CAEN-V830");
    
    crate.installController(controller);
    crate.installModule(scaler, base_address);

    on run_begin {
	scaler.disable(~readout_channels);  
	scaler.start();
    };

    on trigger(scaler) {
        scaler.read(readout_channels);
	//scaler.blockRead();
    }
}
#endif
