/* KinokoDataAnalyzer.cc */
/* Created by Enomoto Sanshiro on 23 October 1999. */
/* Last updated by Enomoto Sanshiro on 2 September 2001. */


#include <iostream>
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoNestedDataSection.hh"
#include "KinokoDataAnalyzer.hh"

using namespace std;


TKinokoDataAnalyzer::TKinokoDataAnalyzer(void)
{
    _DataSource = 0;
}

TKinokoDataAnalyzer::~TKinokoDataAnalyzer()
{
}

bool TKinokoDataAnalyzer::IsSharable(void)
{
    return false;
}

void TKinokoDataAnalyzer::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    _DataSource = DataSource;
}

int TKinokoDataAnalyzer::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if (DataSource == 0) {
	DataSource = _DataSource;
    }

    if (DataSource == 0) {
	throw TKinokoException(
	    "TKinokoDataAnalyzer::ProcessDataPacket()",
	    "reference to an uninitialized DataSource object"
	);
    }

    int SectionId = TKinokoDataSectionScanner::SectionIdOf(DataPacket);
    if (DataSection == 0) {
	if ((DataSection = DataSource->DataSection(SectionId)) == 0) {
	    return 0;
	}
    }

    //... Should I use RTTI?
    switch (DataSection->SectionType()) {
      case TKinokoDataSection::SectionType_Block:
	return ProcessBlockData(
	    DataPacket, (TKinokoBlockDataSection*) DataSection
	);
      case TKinokoDataSection::SectionType_Indexed:
	return ProcessIndexedData(
	    DataPacket, (TKinokoIndexedDataSection*) DataSection
	);
      case TKinokoDataSection::SectionType_Tagged:
	return ProcessTaggedData(
	    DataPacket, (TKinokoTaggedDataSection*) DataSection
	);
      case TKinokoDataSection::SectionType_Nested:
	return ProcessNestedData(
	    DataPacket, (TKinokoNestedDataSection*) DataSection
	);
      default:
	  ;
    }

    return 0;
}

int TKinokoDataAnalyzer::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return 0;
}

int TKinokoDataAnalyzer::ProcessCommandPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return 0;
}

void TKinokoDataAnalyzer::Reset(void) throw(TKinokoException)
{
}

int TKinokoDataAnalyzer::ProcessBlockData(void* DataPacket, TKinokoBlockDataSection* DataSection) throw(TKinokoException)
{
    return 0;
}

int TKinokoDataAnalyzer::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    return 0;
}

int TKinokoDataAnalyzer::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* Section) throw(TKinokoException)
{
    return 0;
}

int TKinokoDataAnalyzer::ProcessNestedData(void* DataPacket, TKinokoNestedDataSection* Section) throw(TKinokoException)
{
    int Result = 0;

    TKinokoNestedDataSectionScanner* Scanner = Section->Scanner();
    int DataSize = Scanner->DataSizeOf(DataPacket);
    U8bit* SubPacket = (U8bit*) Scanner->DataAreaOf(DataPacket);

    int SubSectionId;
    TKinokoDataSection* SubSection;
    int SubPacketSize;
    int ProcessedSize = 0;
    while (ProcessedSize < DataSize) {
	SubSectionId = TKinokoDataSectionScanner::SectionIdOf(SubPacket);
	SubPacketSize = TKinokoDataSectionScanner::PacketSizeOf(SubPacket);
	SubSection = Section->DataSection(SubSectionId);

	if (SubSection != 0) {
	    Result += ProcessDataPacket(SubPacket, _DataSource, SubSection);
	}

	ProcessedSize += SubPacketSize;
	SubPacket += SubPacketSize;
    }

    return Result;
}

void TKinokoDataAnalyzer::Dump(std::ostream& os, const std::string& Indent)
{
    os << "UNKNOWN" << endl;
}



TKinokoDataAnalyzerList::TKinokoDataAnalyzerList(void)
{
    _AnalyzerListCapacity = 64;
    _NumberOfAnalyzers = 0;

    _AnalyzerList = new TKinokoDataAnalyzer* [_AnalyzerListCapacity];
}

TKinokoDataAnalyzerList::~TKinokoDataAnalyzerList()
{
    delete[] _AnalyzerList;
}

void TKinokoDataAnalyzerList::Destroy(void)
{
    DestroyAnalyzer();
}

void TKinokoDataAnalyzerList::AddAnalyzer(TKinokoDataAnalyzer* Analyzer)
{
    if (_NumberOfAnalyzers >= _AnalyzerListCapacity) {
	TKinokoDataAnalyzer** OldAnalyzerList = _AnalyzerList;
	_AnalyzerListCapacity *= 2;
	_AnalyzerList = new TKinokoDataAnalyzer* [_AnalyzerListCapacity];
	for (int i = 0; i < _NumberOfAnalyzers; i++) {
	    _AnalyzerList[i] = OldAnalyzerList[i];
	}
	delete[] OldAnalyzerList;
    }

    _AnalyzerList[_NumberOfAnalyzers++] = Analyzer;
}

void TKinokoDataAnalyzerList::DestroyAnalyzer(void)
{
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	_AnalyzerList[i]->Destroy();
	delete _AnalyzerList[i];
    }

    _NumberOfAnalyzers = 0;
}

void TKinokoDataAnalyzerList::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	_AnalyzerList[i]->ReadDataSource(DataSource);
    }
}

int TKinokoDataAnalyzerList::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    int Result = 0;
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	Result += _AnalyzerList[i]->ProcessDataPacket(
	    DataPacket, DataSource, DataSection
	);
    }

    return Result;
}

int TKinokoDataAnalyzerList::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    int Result = 0;
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	Result += _AnalyzerList[i]->ProcessTrailerPacket(Packet, DataSource);
    }

    return Result;
}

int TKinokoDataAnalyzerList::ProcessCommandPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    int Result = 0;
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	Result += _AnalyzerList[i]->ProcessCommandPacket(Packet, DataSource);
    }

    return Result;
}

void TKinokoDataAnalyzerList::Reset(void) throw(TKinokoException)
{
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	_AnalyzerList[i]->Reset();
    }
}

void TKinokoDataAnalyzerList::Dump(std::ostream& os, const std::string& Indent)
{
    for (int i = 0; i < _NumberOfAnalyzers; i++) {
	_AnalyzerList[i]->Dump(os, Indent);
    }
}

