/* KinokoDataChecker.cc */
/* Created by Enomoto Sanshiro on 23 Septemter 2001. */
/* Last updated by Enomoto Sanshiro on 23 Septemter 2001. */


#include <string>
#include "KinokoSectionDataAnalyzer.hh"
#include "KinokoDataChecker.hh"

using namespace std;


TKinokoDataChecker::TKinokoDataChecker(void)
{
}

TKinokoDataChecker::~TKinokoDataChecker()
{
}



TKinokoElementDataChecker::TKinokoElementDataChecker(const string& SectionPath, const TKinokoDataElementSpecifier& ElementSpecifier, int Value, int OperatorId)
{
    AddSection(SectionPath);

    _SectionPath = SectionPath;
    _ElementSpecifier = ElementSpecifier;

    _IntValue = Value;
    _FloatValue = Value;

    _OperatorId = OperatorId;

    _IntData = 0;
    _FloatData = 0;

    _UseFloat = false;
}

TKinokoElementDataChecker::TKinokoElementDataChecker(const string& SectionPath, const TKinokoDataElementSpecifier& ElementSpecifier, float Value, int OperatorId)
{
    AddSection(SectionPath);

    _SectionPath = SectionPath;
    _ElementSpecifier = ElementSpecifier;

    _IntValue = (int) Value;
    _FloatValue = Value;

    _OperatorId = OperatorId;

    _IntData = 0;
    _FloatData = 0;

    _UseFloat = true;
}

TKinokoElementDataChecker::~TKinokoElementDataChecker()
{
}

int TKinokoElementDataChecker::ProcessData(TKinokoSectionData* SectionData) throw(TKinokoException)
{
    int Address;
    int Result;

    if (_UseFloat) {
	Result = SectionData->GetNextOf(_ElementSpecifier, Address, _FloatData);
	_IntData = (int) _FloatData;
    }
    else {
	Result = SectionData->GetNextOf(_ElementSpecifier, Address, _IntData);
	_FloatData = _IntData;
    }
    
    return Result;
}

bool TKinokoElementDataChecker::IsConditionSatisfied(void)
{
    bool Result;

    switch (_OperatorId) {
      case Operator_Equal:
	Result = _UseFloat ? (_FloatData == _FloatValue) : (_IntData == _IntValue);
	break;
      case Operator_NotEqual:
	Result = _UseFloat ? (_FloatData != _FloatValue) : (_IntData != _IntValue);
	break;
      case Operator_GreaterThan:
	Result = _UseFloat ? (_FloatData > _FloatValue) : (_IntData > _IntValue);
	break;
      case Operator_GreaterEqual:
	Result = _UseFloat ? (_FloatData >= _FloatValue) : (_IntData >= _IntValue);
	break;
      case Operator_LessThan:
	Result = _UseFloat ? (_FloatData < _FloatValue) : (_IntData < _IntValue);
	break;
      case Operator_LessEqual:
	Result = _UseFloat ? (_FloatData <= _FloatValue) : (_IntData <= _IntValue);
	break;
      case Operator_BitAnd:
	Result = ((_IntData & _IntValue) != 0);
	break;
      case Operator_BitXor:
	Result = ((_IntData ^ _IntValue) != 0);
	break;
      default:
	Result = false;
    }

    return Result;
}

void TKinokoElementDataChecker::Dump(ostream& os, const string& Indent)
{
    os << "[\"" << _SectionPath << "\", ";
    os << _ElementSpecifier.AsString() << "] ";

    switch (_OperatorId) {
      case Operator_Equal:
	os << " == ";
	break;
      case Operator_NotEqual:
	os << " != ";
	break;
      case Operator_GreaterThan:
	os << " > ";
	break;
      case Operator_GreaterEqual:
	os << " >= ";
	break;
      case Operator_LessThan:
        os << " < ";
	break;
      case Operator_LessEqual:
	os << " >= ";
	break;
      case Operator_BitAnd:
        os << " & ";
	break;
      case Operator_BitXor:
        os << " ^ ";
	break;
      default:
	;
    }

    os << (_UseFloat ? _FloatValue : _IntValue) << dec;
}



TKinokoComplexDataChecker::TKinokoComplexDataChecker(TKinokoDataChecker* LeftDataChecker, TKinokoDataChecker* RightDataChecker, int OperatorId)
{
    _LeftDataChecker = LeftDataChecker;
    _RightDataChecker = RightDataChecker;
    _OperatorId = OperatorId;
}

TKinokoComplexDataChecker::~TKinokoComplexDataChecker()
{
    delete _RightDataChecker;
    delete _LeftDataChecker;
}

void TKinokoComplexDataChecker::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    if (_LeftDataChecker) {
	_LeftDataChecker->ReadDataSource(DataSource);
    }
    if (_RightDataChecker) {
	_RightDataChecker->ReadDataSource(DataSource);
    }
}

int TKinokoComplexDataChecker::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    int Result = 0;

    if (_LeftDataChecker) {
	Result += _LeftDataChecker->ProcessDataPacket(DataPacket, DataSource, DataSection);
    }
    if (_RightDataChecker) {
	Result += _RightDataChecker->ProcessDataPacket(DataPacket, DataSource, DataSection);
    }

    return Result;
}

int TKinokoComplexDataChecker::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    int Result = 0;

    if (_LeftDataChecker) {
	Result += _LeftDataChecker->ProcessTrailerPacket(Packet, DataSource);
    }
    if (_RightDataChecker) {
	Result += _RightDataChecker->ProcessTrailerPacket(Packet, DataSource);
    }

    return Result;
}

int TKinokoComplexDataChecker::ProcessData(TKinokoSectionData* SectionData) throw(TKinokoException)
{
    return 0;
}

bool TKinokoComplexDataChecker::IsConditionSatisfied(void)
{
    bool Result;

    switch (_OperatorId) {
      case Operator_Not:
	Result = ! (_RightDataChecker->IsConditionSatisfied());
	break;
      case Operator_And:
	Result = _LeftDataChecker->IsConditionSatisfied() && _RightDataChecker->IsConditionSatisfied();
	break;
      case Operator_Or:
	Result = _LeftDataChecker->IsConditionSatisfied() || _RightDataChecker->IsConditionSatisfied();
	break;
      default:
	Result = false;
    }

    return Result;
}

void TKinokoComplexDataChecker::Dump(std::ostream& os, const string& Indent)
{
    if (_LeftDataChecker) {
	os << "(";
	_LeftDataChecker->Dump(os);
	os << ") ";
    }

    switch (_OperatorId) {
      case Operator_Not:
	os << "! ";
	break;
      case Operator_And:
	os << "&& ";
	break;
      case Operator_Or:
	os << "|| ";
	break;
      default:
	os << "?? ";
	;
    }
    
    if (_RightDataChecker) {
	os << "(";
	_RightDataChecker->Dump(os);
	os << ")";
    }
}

