/* KinokoDataDumper.cc */
/* Created by Enomoto Sanshiro on 6 May 2001. */
/* Last updated by Enomoto Sanshiro on 6 March 2008. */


#include <iostream>
#include <sstream>
#include <iomanip>
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoNestedDataSection.hh"
#include "KinokoDataDumper.hh"

using namespace std;


TKinokoDataDumper::TKinokoDataDumper(std::ostream& OutputStream)
: _OutputStream(OutputStream)
{
}

TKinokoDataDumper:: ~TKinokoDataDumper()
{
}

bool TKinokoDataDumper::IsSharable(void)
{
    return true;
}

void TKinokoDataDumper::AddTargetSection(const std::string& SectionName)
{
    _TargetSectionSet.insert(SectionName);
}

int TKinokoDataDumper::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) TKinokoDataSectionScanner::DataAreaOf(DataPacket),
	TKinokoDataSectionScanner::DataSizeOf(DataPacket)
    );

    return 1;
}

int TKinokoDataDumper::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) TKinokoDataSectionScanner::DataAreaOf(DataPacket),
	TKinokoDataSectionScanner::DataSizeOf(DataPacket)
    );

    return 1;
}

int TKinokoDataDumper::ProcessBlockData(void* DataPacket, TKinokoBlockDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) TKinokoDataSectionScanner::DataAreaOf(DataPacket),
	TKinokoDataSectionScanner::DataSizeOf(DataPacket)
    );

    return 1;
}



TKinokoPacketDumper::TKinokoPacketDumper(std::ostream& OutputStream)
: TKinokoDataDumper(OutputStream)
{
}

TKinokoPacketDumper:: ~TKinokoPacketDumper()
{
}

int TKinokoPacketDumper::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) DataPacket,
	TKinokoDataSectionScanner::PacketSizeOf(DataPacket)
    );

    return 1;
}

int TKinokoPacketDumper::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) DataPacket,
	TKinokoDataSectionScanner::PacketSizeOf(DataPacket)
    );

    return 1;
}

int TKinokoPacketDumper::ProcessBlockData(void* DataPacket, TKinokoBlockDataSection* DataSection) throw(TKinokoException)
{
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(DataSection->FullSectionName()) == 0) {
	    return 0;
	}
    }

    _OutputStream.write(
	(char*) DataPacket,
	TKinokoDataSectionScanner::PacketSizeOf(DataPacket)
    );

    return 1;
}



TKinokoTextDataDumper::TKinokoTextDataDumper(ostream& OutputStream, int FormatVersion)
: TKinokoDataDumper(OutputStream)
{
    _FormatVersion = (FormatVersion == 0) ? 2 : FormatVersion;
    _NestCount = 0;
}

TKinokoTextDataDumper::~TKinokoTextDataDumper()
{
}

int TKinokoTextDataDumper::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if ((DataSource != 0) && (_NestCount == 0)) {
	_OutputStream << "### [" << DataSource->DataSourceName() << "] ###" << endl;
    }

    ++_NestCount;
    int Result = TKinokoDataDumper::ProcessDataPacket(
	DataPacket, DataSource, DataSection
    );
    --_NestCount;

    return Result;
}

int TKinokoTextDataDumper::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    bool ShowsSubAddress = (_FormatVersion >= 2);
    bool ShowsDataIndex = (_FormatVersion >= 2);

    const string& SectionName = DataSection->FullSectionName();
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(SectionName) == 0) {
	    return 0;
	}
    }

    TKinokoIndexedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfElements = Scanner->NumberOfElements(DataPacket);
    int SubAddressWidth = DataSection->SubAddressWidth();
    int Depth = DataSection->ElementDepth();

    int Address, SubAddress; string Data;
    for (int Index = 0; Index < NumberOfElements; Index++) {
	Scanner->ReadStringFrom(
	    DataPacket, Index, Address, SubAddress, Data
	);

	_OutputStream << SectionName << " ";

	_OutputStream << Address;
	if (ShowsSubAddress && (SubAddressWidth > 0)) {
	    _OutputStream << ":" << SubAddress;
	}
	if (ShowsDataIndex && (Depth > 1)) {
	    _OutputStream << ";" << (Index % Depth);
	}
	_OutputStream << " ";

	_OutputStream << Data << endl;
    }

    return 1;
}

int TKinokoTextDataDumper::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    const string& SectionName = DataSection->FullSectionName();
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(SectionName) == 0) {
	    return 0;
	}
    }


    TKinokoTaggedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfFields = DataSection->NumberOfFields();

    string FieldName, Value;
    for (int Index = 0; Index < NumberOfFields; Index++) {
	FieldName = DataSection->FieldNameOf(Index);
	Scanner->ReadStringFrom(DataPacket, Index, Value);
	_OutputStream << SectionName << " " << FieldName << " ";
	_OutputStream << Value << endl;
    }

    return 1;
}

int TKinokoTextDataDumper::ProcessBlockData(void* DataPacket, TKinokoBlockDataSection* DataSection) throw(TKinokoException)
{
    const string& SectionName = DataSection->FullSectionName();
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(SectionName) == 0) {
	    return 0;
	}
    }

    TKinokoBlockDataSectionScanner* Scanner = DataSection->Scanner();
    int DataSize = Scanner->DataSizeOf(DataPacket);
    void* DataArea = Scanner->DataAreaOf(DataPacket);

    _OutputStream << hex << setfill('0');
    _OutputStream << "[" << SectionName << "]" << flush;
    for (int Offset = 0; Offset < DataSize; Offset++) {
	if (Offset % 16 == 0) {
	    _OutputStream << endl << "  ";
	    _OutputStream << setw(4) << (Offset >> 16) << " ";
	    _OutputStream << setw(4) << (Offset & 0xffff) << ":  ";
	}
	else if (Offset % 8 == 0) {
	    _OutputStream << "  ";
	}
	
	_OutputStream << setw(2) << (int) (((U8bit*) DataArea)[Offset]) << " ";
    }

    _OutputStream << dec << endl;

    return 1;
}

int TKinokoTextDataDumper::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    if (DataSource != 0) {
	_OutputStream << "### [" << DataSource->DataSourceName() << "] ###" << endl;
    }

    _OutputStream << endl;

    return 1;
}

int TKinokoTextDataDumper::ProcessCommandPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    if (_FormatVersion >= 2) {
	int EventTime = TKinokoDataStreamScanner::CommandTimeOf(Packet);
	_OutputStream << "# EventTime: " << EventTime << endl;
    }

    return TKinokoDataAnalyzer::ProcessCommandPacket(Packet, DataSource);
}



TKinokoDataTabler::TKinokoDataTabler(ostream& OutputStream, int FormatVersion)
: _OutputStream(OutputStream)
{
    _FormatVersion = (FormatVersion == 0) ? 2 : FormatVersion;

    _EventCount = 0;
    _EventTime = 0;
    _StartTime = 0;
}

TKinokoDataTabler::~TKinokoDataTabler()
{
}

void TKinokoDataTabler::AddTargetSection(const std::string& SectionName)
{
    _TargetSectionSet.insert(SectionName);
}

void TKinokoDataTabler::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    TKinokoDataAnalyzer::ReadDataSource(DataSource);
    _DataSourceName = DataSource->DataSourceName();
}

int TKinokoDataTabler::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    bool ShowsSubAddress = (_FormatVersion >= 2);
    bool ShowsDataIndex = (_FormatVersion >= 2);

    const string& SectionName = DataSection->FullSectionName();
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(SectionName) == 0) {
	    return 0;
	}
    }

    TKinokoIndexedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfElements = Scanner->NumberOfElements(DataPacket);
    int SubAddressWidth = DataSection->SubAddressWidth();
    int Depth = DataSection->ElementDepth();
    if (NumberOfElements == 0) {
	return 0;
    }

    int Address, SubAddress; string Data;
    for (int Index = 0; Index < NumberOfElements; Index++) {
	Scanner->ReadStringFrom(
	    DataPacket, Index, Address, SubAddress, Data
	);
	_IndexedDataList.push_back(make_pair(Address, Data));

	if (_FieldNameList.size() < _IndexedDataList.size()) {
	    ostringstream os;
	    os << SectionName << "." << Address;
	    if (ShowsSubAddress && (SubAddressWidth > 0)) {
		os << ":" << SubAddress;
	    }
	    if (ShowsDataIndex && (Depth > 1)) {
		os << ";" << (Index % Depth);
	    }

	    _FieldNameList.push_back(make_pair(Address, os.str()));
	}
    }

    return 1;
}

int TKinokoDataTabler::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    const string& SectionName = DataSection->FullSectionName();
    if (! _TargetSectionSet.empty()) {
	if (_TargetSectionSet.count(SectionName) == 0) {
	    return 0;
	}
    }

    TKinokoTaggedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfFields = DataSection->NumberOfFields();
    if (NumberOfFields == 0) {
	return 0;
    }
    
    string FieldName, Data;
    for (int Index = 0; Index < NumberOfFields; Index++) {
	FieldName = SectionName + "." + DataSection->FieldNameOf(Index);
	Scanner->ReadStringFrom(DataPacket, Index, Data);
	_TaggedDataList.push_back(make_pair(FieldName, Data));
    }

    return 1;
}

int TKinokoDataTabler::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    bool ShowsTimeInfo = (_FormatVersion >= 2);
    bool CommentsOutTagged = (_FormatVersion >= 2);
    
    if (! _IndexedDataList.empty()) {
	if (_EventCount == 0) {
	    _OutputStream << "# DataSource: " << _DataSourceName << endl;
	    _OutputStream << "# Fields: index ";
	    if (ShowsTimeInfo) {
		_OutputStream << "time ";
	    }
	    for (unsigned i = 0; i < _FieldNameList.size(); i++) {
		_OutputStream << _FieldNameList[i].second << " ";
	    }
	    _OutputStream << endl;

	    _OutputStream << "# FieldTypes: int";
	    if (ShowsTimeInfo) {
		_OutputStream << " int";
	    }
	    for (unsigned i = 0; i < _FieldNameList.size(); i++) {
		_OutputStream << " float";
	    }
	    _OutputStream << endl;

	    if (ShowsTimeInfo) {
		_OutputStream << "# StartTime: " << _StartTime << endl;
	    }
	    _OutputStream << endl;
	}
	
	_OutputStream << _EventCount << " ";
	if (ShowsTimeInfo) {
	    _OutputStream << (_EventTime - _StartTime) << " ";
	}
	unsigned i, j;
	for (i = 0, j = 0; i < _IndexedDataList.size(); i++, j++) {
	    while (j < _FieldNameList.size()) {
		if (_IndexedDataList[i].first != _FieldNameList[j].first) {
		    _OutputStream << "- ";
		    j++;
		}
		else {
		    break;
		}
	    }
	    if (j == _FieldNameList.size()) {
		_OutputStream << "# ";
	    }
	    _OutputStream << _IndexedDataList[i].second << " ";
	}	
	_OutputStream << endl;

	_IndexedDataList.clear();
	_EventCount++;
    }

    if (! _TaggedDataList.empty()) {
	for (unsigned i = 0; i < _TaggedDataList.size(); i++) {
	    if (CommentsOutTagged) {
		_OutputStream << "# ";
	    }
	    _OutputStream << _TaggedDataList[i].first << " ";
	    _OutputStream << _TaggedDataList[i].second << endl;
	}
	_TaggedDataList.clear();
    }

    return 1;
}

int TKinokoDataTabler::ProcessCommandPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    _EventTime = TKinokoDataStreamScanner::CommandTimeOf(Packet);

    if (_StartTime == 0) {
	_StartTime = _EventTime;
    }

    return TKinokoDataAnalyzer::ProcessCommandPacket(Packet, DataSource);
}
