/* KinokoSectionDataAnalyzer.cc */
/* Created by Enomoto Sanshiro on 2 September 2001. */
/* Last updated by Enomoto Sanshiro on 13 July 2007. */


#include <sstream>
#include <string>
#include "KinokoDataSource.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoSectionDataAnalyzer.hh"

using namespace std;


TKinokoSectionDataAnalyzer::TKinokoSectionDataAnalyzer(const string& SectionPath)
{
    if (! SectionPath.empty()) {
	_SectionPathList.push_back(SectionPath);
    }
}

TKinokoSectionDataAnalyzer::~TKinokoSectionDataAnalyzer()
{
    for (unsigned i = 0; i < _SectionDataList.size(); i++) {
	delete _SectionDataList[i];
    }
}

int TKinokoSectionDataAnalyzer::AddSection(const string& SectionPath)
{
    int SectionIndex = _SectionPathList.size();
    _SectionPathList.push_back(SectionPath);

    return SectionIndex;
}

void TKinokoSectionDataAnalyzer::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    for (unsigned i = 0; i < _SectionPathList.size(); i++) {
	string SectionPath = _SectionPathList[i];
	string SectionName = TKinokoDataSection::SectionNameOf(SectionPath);

	TKinokoDataSection* DataSection = DataSource->DataSection(SectionName);
	if (DataSection == 0) {
	    // section not found  //
	    continue;
	}

	TKinokoSectionData* SectionData = (
	    TKinokoSectionData::CreateSectionDataFor(SectionPath, DataSection)
	);
	if (SectionData != 0) {
	    SectionData->SetSectionIndex(i);
	    _SectionDataList.push_back(SectionData);
	}
    }
}

int TKinokoSectionDataAnalyzer::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    int Result = 0;
    for (unsigned i = 0; i < _SectionDataList.size(); i++) {
	if (_SectionDataList[i]->ProcessDataPacket(DataPacket)) {
	    Result += ProcessData(_SectionDataList[i]);
	}
    }

    return Result;
}

int TKinokoSectionDataAnalyzer::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return ProcessTrailer(TKinokoDataStreamScanner::TrailerValueOf(Packet));
}

int TKinokoSectionDataAnalyzer::ProcessCommandPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    int EventTime = TKinokoDataStreamScanner::CommandTimeOf(Packet);

    for (unsigned i = 0; i < _SectionDataList.size(); i++) {
	_SectionDataList[i]->SetEventTime(EventTime);
    }

    return 1;
}

void TKinokoSectionDataAnalyzer::Dump(ostream& os, const string& Indent)
{
    os << Indent << "UNKNOWN" << endl;
}



TKinokoSectionData::TKinokoSectionData(TKinokoDataSection* DataSection)
{
    _DataSection = DataSection;
    _SectionIndex = 0;
    _EventTime = 0;
}

TKinokoSectionData::~TKinokoSectionData()
{
}

void TKinokoSectionData::SetSectionIndex(int SectionIndex)
{
    _SectionIndex = SectionIndex;
}

TKinokoSectionData* TKinokoSectionData::CreateSectionDataFor(const string& SectionPath, TKinokoDataSection* DataSection)
{
    if (DataSection == 0) {
	return 0;
    }

    TKinokoSectionData* SectionData = 0;

    switch (DataSection->SectionType()) {
    case TKinokoDataSection::SectionType_Indexed:
	SectionData = new TKinokoIndexedSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Tagged:
	SectionData = new TKinokoTaggedSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Block:
	SectionData = new TKinokoBlockSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Nested:
	SectionData = new TKinokoNestedSectionData(SectionPath, DataSection);
	break;
    default:
	SectionData = 0;
	break;
    }

    return SectionData;
}

void TKinokoSectionData::InitializeElementSpecifier(TKinokoDataElementSpecifier& ElementSpecifier)
{
    if (! ElementSpecifier.IsInitialized()) {
	int SpecifiedAddress = IndexOf(ElementSpecifier.FieldName());
	if (SpecifiedAddress < 0) {
	    SpecifiedAddress = TKinokoDataElementSpecifier::Address_None;
	}
	ElementSpecifier.SetAddress(SpecifiedAddress);
    }
}

int TKinokoSectionData::IndexOf(const string& FieldName)
{
    return -1;
}

string TKinokoSectionData::NameOf(int Index)
{
    ostringstream FieldNameStream;
    FieldNameStream << Index;

    return FieldNameStream.str();
}

template<class TDataType>
bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, TDataType& Data)
{
    if (! ElementSpecifier.IsInitialized()) {
	InitializeElementSpecifier(ElementSpecifier);
    }

    Address = ElementSpecifier.Address();
    if (Address == TKinokoDataElementSpecifier::Address_None) {
	return false;
    }
    else if (Address == TKinokoDataElementSpecifier::Address_All) {
	return GetNext(Address, Data);
    }
    else {
	return GetNextAt(Address, Data);
    }
}

template bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, int& Data);

template bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, float& Data);

template bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, std::string& Data);

bool TKinokoSectionData::GetNext(int& Address, float& Data)
{
    int IntData;

    bool Result = GetNext(Address, IntData);
    if (Result) {
	Data = IntData;
    }

    return Result;
}

bool TKinokoSectionData::GetNextAt(int Address, float& Data)
{
    int IntData;

    bool Result = GetNextAt(Address, IntData);
    if (Result) {
	Data = IntData;
    }

    return Result;
}

bool TKinokoSectionData::GetNext(int& Address, std::string& Data)
{
    int IntData;

    bool Result = GetNext(Address, IntData);
    if (Result) {
	ostringstream os;
	os << IntData;
	Data = os.str();
    }

    return Result;
}

bool TKinokoSectionData::GetNextAt(int Address, std::string& Data)
{
    int IntData;

    bool Result = GetNextAt(Address, IntData);
    if (Result) {
	ostringstream os;
	os << IntData;
	Data = os.str();
    }

    return Result;
}

void TKinokoSectionData::SetEventTime(int EventTime)
{
    _EventTime = EventTime;
}

int TKinokoSectionData::GetEventTime(void)
{
    return _EventTime;
}



TKinokoIndexedSectionData::TKinokoIndexedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _IndexedDataSection = (TKinokoIndexedDataSection*) DataSection;
    _Scanner = _IndexedDataSection->Scanner();
    _SectionId = _IndexedDataSection->SectionId();

    _NumberOfElements = 0;
    _CurrentIndex = 0;
}

TKinokoIndexedSectionData::~TKinokoIndexedSectionData()
{
}

bool TKinokoIndexedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_NumberOfElements = 0;
	return false;
    }

    _DataPacket = DataPacket;
    _CurrentIndex = 0;
    _NumberOfElements = _Scanner->NumberOfElements(_DataPacket);

    return true;
}

bool TKinokoIndexedSectionData::GetNext(int& Address, int& Data)
{
    if (_CurrentIndex >= _NumberOfElements) {
	return false;
    }
    
    _Scanner->ReadFrom(_DataPacket, _CurrentIndex, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoIndexedSectionData::GetNextAt(int Address, int& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	if (_CurrentIndex < _NumberOfElements) {
	    Data = _NumberOfElements;
	    _CurrentIndex = _NumberOfElements;
	    return true;
	}
    }
    else {
	int ThisAddress;
	while (_CurrentIndex < _NumberOfElements) {
	    _Scanner->ReadFrom(
		_DataPacket, _CurrentIndex++, ThisAddress, Data
	    );
	    if (ThisAddress == Address) {
		return true;
	    }
	}
    }

    return false;
}

bool TKinokoIndexedSectionData::GetNext(int& Address, float& Data)
{
    if (_CurrentIndex >= _NumberOfElements) {
	return false;
    }
    
    _Scanner->ReadFloatFrom(_DataPacket, _CurrentIndex, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoIndexedSectionData::GetNextAt(int Address, float& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	return TKinokoSectionData::GetNextAt(Address, Data);
    }
    else {
	int ThisAddress;
	while (_CurrentIndex < _NumberOfElements) {
	    _Scanner->ReadFloatFrom(
		_DataPacket, _CurrentIndex++, ThisAddress, Data
	    );
	    if (ThisAddress == Address) {
		return true;
	    }
	}
    }

    return false;
}

bool TKinokoIndexedSectionData::GetNext(int& Address, std::string& Data)
{
    if (_CurrentIndex >= _NumberOfElements) {
	return false;
    }
    
    _Scanner->ReadStringFrom(_DataPacket, _CurrentIndex, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoIndexedSectionData::GetNextAt(int Address, std::string& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	return TKinokoSectionData::GetNextAt(Address, Data);
    }
    else {
	int ThisAddress;
	while (_CurrentIndex < _NumberOfElements) {
	    _Scanner->ReadStringFrom(
		_DataPacket, _CurrentIndex++, ThisAddress, Data
	    );
	    if (ThisAddress == Address) {
		return true;
	    }
	}
    }

    return false;
}

void* TKinokoIndexedSectionData::GetDataBlock(void)
{
    return TKinokoDataSectionScanner::DataAreaOf(_DataPacket);
}

size_t TKinokoIndexedSectionData::DataBlockSize(void)
{
    return TKinokoDataSectionScanner::DataSizeOf(_DataPacket);
}



TKinokoTaggedSectionData::TKinokoTaggedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _TaggedDataSection = (TKinokoTaggedDataSection*) DataSection;
    _Scanner = _TaggedDataSection->Scanner();
    _SectionId = _TaggedDataSection->SectionId();
    _NumberOfFields = _TaggedDataSection->NumberOfFields();

    _ReadoutFlags = new char[_NumberOfFields+1];
    _IsDataAvailable = false;
}

TKinokoTaggedSectionData::~TKinokoTaggedSectionData()
{
    delete[] _ReadoutFlags;
}

int TKinokoTaggedSectionData::IndexOf(const string& FieldName)
{
    int Index;
    try {
	Index = _TaggedDataSection->FieldIndexOf(FieldName);
    }
    catch (TKinokoException &e) {
	Index = -1;
    }

    return Index;
}

string TKinokoTaggedSectionData::NameOf(int Index)
{
    return _TaggedDataSection->FieldNameOf(Index);
}

bool TKinokoTaggedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	return _IsDataAvailable = false;
    }

    _DataPacket = DataPacket;
    _CurrentIndex = 0;
    memset(_ReadoutFlags, 0, _NumberOfFields+1);

    return _IsDataAvailable = true;
}

bool TKinokoTaggedSectionData::GetNext(int& Address, int& Data)
{
    if (! _IsDataAvailable || (_CurrentIndex >= _NumberOfFields)) {
	return false;
    }
    
    Address = _CurrentIndex;
    _Scanner->ReadFrom(_DataPacket, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoTaggedSectionData::GetNextAt(int Address, int& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	if (_ReadoutFlags[_NumberOfFields] == 0) {
	    Data = _NumberOfFields;
	    _ReadoutFlags[_NumberOfFields] = 1;
	    return true;
	}
    }
    else if ((Address >= 0) && (Address < _NumberOfFields)) {
	if (_ReadoutFlags[Address] == 0) {
	    _Scanner->ReadFrom(_DataPacket, Address, Data);
	    _ReadoutFlags[Address] = 1;
	    return true;
	}
    }

    return false;
}

bool TKinokoTaggedSectionData::GetNext(int& Address, float& Data)
{
    if (! _IsDataAvailable || (_CurrentIndex >= _NumberOfFields)) {
	return false;
    }
    
    Address = _CurrentIndex;
    _Scanner->ReadFloatFrom(_DataPacket, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoTaggedSectionData::GetNextAt(int Address, float& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	return TKinokoSectionData::GetNextAt(Address, Data);
    }
    else if ((Address >= 0) && (Address < _NumberOfFields)) {
	if (_ReadoutFlags[Address] == 0) {
	    _Scanner->ReadFloatFrom(_DataPacket, Address, Data);
	    _ReadoutFlags[Address] = 1;
	    return true;
	}
    }

    return false;
}

bool TKinokoTaggedSectionData::GetNext(int& Address, std::string& Data)
{
    if (! _IsDataAvailable || (_CurrentIndex >= _NumberOfFields)) {
	return false;
    }
    
    Address = _CurrentIndex;
    _Scanner->ReadStringFrom(_DataPacket, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoTaggedSectionData::GetNextAt(int Address, std::string& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	return TKinokoSectionData::GetNextAt(Address, Data);
    }
    else if ((Address >= 0) && (Address < _NumberOfFields)) {
	if (_ReadoutFlags[Address] == 0) {
	    _Scanner->ReadStringFrom(_DataPacket, Address, Data);
	    _ReadoutFlags[Address] = 1;
	    return true;
	}
    }

    return false;
}

void* TKinokoTaggedSectionData::GetDataBlock(void)
{
    return TKinokoDataSectionScanner::DataAreaOf(_DataPacket);
}

size_t TKinokoTaggedSectionData::DataBlockSize(void)
{
    return TKinokoDataSectionScanner::DataSizeOf(_DataPacket);
}



TKinokoBlockSectionData::TKinokoBlockSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _BlockDataSection = (TKinokoBlockDataSection*) DataSection;
    _Scanner = _BlockDataSection->Scanner();
    _SectionId = _BlockDataSection->SectionId();

    _DataBlock = 0;
}

TKinokoBlockSectionData::~TKinokoBlockSectionData()
{
}

bool TKinokoBlockSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_DataBlock = 0;
	return false;
    }

    _DataBlock = TKinokoDataSectionScanner::DataAreaOf(DataPacket);
    _DataSize = TKinokoDataSectionScanner::DataSizeOf(DataPacket);
    _CurrentAddress = 0;
	
    return true;
}

bool TKinokoBlockSectionData::GetNext(int& Address, int& Data)
{
    if ((_DataBlock == 0) || (_CurrentAddress >= _DataSize)) {
	return false;
    }

    Address = _CurrentAddress;
    Data = ((U32bit*) _DataBlock)[Address / sizeof(U32bit)];

    _CurrentAddress += sizeof(U32bit);

    return true;
}

bool TKinokoBlockSectionData::GetNextAt(int Address, int& Data)
{
    if (_DataBlock == 0) {
	return false;
    }

    if ((Address >= _DataSize) || (_CurrentAddress >= _DataSize)) {
	return false;
    }

    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	Data = _DataSize;
	_CurrentAddress = _DataSize;
    }
    else if (Address < _CurrentAddress) {
	return false;
    }
    else {
	Data = ((U32bit*) _DataBlock)[Address / sizeof(U32bit)];
	_CurrentAddress = sizeof(U32bit) * (Address / sizeof(U32bit) + 1);
    }
    
    return true;
}

void* TKinokoBlockSectionData::GetDataBlock(void)
{
    return _DataBlock;
}

size_t TKinokoBlockSectionData::DataBlockSize(void)
{
    return _DataSize;
}



TKinokoNestedSectionData::TKinokoNestedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _NestedDataSection = (TKinokoNestedDataSection*) DataSection;
    _Scanner = _NestedDataSection->Scanner();
    _SectionId = _NestedDataSection->SectionId();
    _IsDataAvailable = false;

    string SubSectionPath = TKinokoDataSection::SubSectionPathOf(SectionPath);
    string SubSectionName = TKinokoDataSection::SectionNameOf(SubSectionPath);

    TKinokoDataSection* SubSection = _NestedDataSection->DataSection(SubSectionName);
    if (SubSection != 0) {
	_SubSectionData = CreateSectionDataFor(SubSectionPath, SubSection);
    }
    else {
	_SubSectionData = 0;
    }
}

TKinokoNestedSectionData::~TKinokoNestedSectionData()
{
    delete _SubSectionData;
}

int TKinokoNestedSectionData::IndexOf(const std::string& FieldName)
{
    if (_SubSectionData) {
	return _SubSectionData->IndexOf(FieldName);
    }
    else {
	return TKinokoSectionData::IndexOf(FieldName);
    }
}

string TKinokoNestedSectionData::NameOf(int Index)
{
    if (_SubSectionData) {
	return _SubSectionData->NameOf(Index);
    }
    else {
	return TKinokoSectionData::NameOf(Index);
    }
}

bool TKinokoNestedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_SubSectionData == 0) {
	return false;
    }

    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_IsDataAvailable = false;
	return false;
    }

    int DataSize = _Scanner->DataSizeOf(DataPacket);
    U8bit* SubPacket = (U8bit*) _Scanner->DataAreaOf(DataPacket);

    int SubPacketSize;
    int ProcessedSize = 0;
    while (ProcessedSize < DataSize) {
	_IsDataAvailable = _SubSectionData->ProcessDataPacket(SubPacket);
	if (_IsDataAvailable) {
	    break;
	}

	SubPacketSize = TKinokoDataSectionScanner::PacketSizeOf(SubPacket);
	ProcessedSize += SubPacketSize;
	SubPacket += SubPacketSize;
    }

    return _IsDataAvailable;
}

bool TKinokoNestedSectionData::GetNext(int& Address, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNext(Address, Data);
}

bool TKinokoNestedSectionData::GetNextAt(int Address, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNextAt(Address, Data);
}

bool TKinokoNestedSectionData::GetNext(int& Address, float& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNext(Address, Data);
}

bool TKinokoNestedSectionData::GetNextAt(int Address, float& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNextAt(Address, Data);
}

bool TKinokoNestedSectionData::GetNext(int& Address, std::string& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNext(Address, Data);
}

bool TKinokoNestedSectionData::GetNextAt(int Address, std::string& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNextAt(Address, Data);
}

void* TKinokoNestedSectionData::GetDataBlock(void)
{
    if (! _IsDataAvailable) {
	return 0;
    }

    return _SubSectionData->GetDataBlock();
}

size_t TKinokoNestedSectionData::DataBlockSize(void)
{
    if (! _IsDataAvailable) {
	return 0;
    }

    return _SubSectionData->DataBlockSize();
}



TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(void)
{
    _Address = Address_All;
}

TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(int Address)
{
    _Address = Address;

    ostringstream FieldNameStream;
    FieldNameStream << Address;
    _FieldName = FieldNameStream.str();
}

TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(const string& FieldName)
{
    _FieldName = FieldName;

    if (_FieldName == "$data_size") {
	_Address = Address_DataSize;
    }
    else {
	_Address = Address_Unknown;
    }
}

TKinokoDataElementSpecifier::~TKinokoDataElementSpecifier()
{
}

void TKinokoDataElementSpecifier::SetAddress(int Address)
{
    _Address = Address;
}

string TKinokoDataElementSpecifier::AsString(void) const
{
    if (_Address == Address_Unknown) {
	return '\"' + _FieldName + '\"';
    }
    else if (_Address == Address_All) {
	return string("ALL");
    }
    else {
	ostringstream os;
	os << _Address;
	return os.str();
    }
}
