/* KinokoBufferClient.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Updated by Enomoto Sanshiro on 21 August 2002. */
/* Last updated by Enomoto Sanshiro on 16 January 2010. */


#include <iostream>
#include <fstream>
#include "MushDefs.hh"
#include "MushIpc.hh"
#include "KinokoDefs.hh"
#include "KinokoBufferServer.hh"
#include "KinokoBufferClient.hh"

using namespace std;


TKinokoBufferClient::TKinokoBufferClient(int SharedMemoryProjectId, int MessageQueueProjectId, const std::string& ProjectPath, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
{
    _SharedMemory = 0;
    _MessageQueue = 0;

    try {
	_SharedMemory = new TMushClientSharedMemory(
	    SharedMemoryProjectId, ProjectPath
	);
	_MessageQueue = new TMushClientMessageQueue(
	    MessageQueueProjectId, ProjectPath
	);
    }
    catch (TSystemCallException &e) {
	delete _MessageQueue;
	delete _SharedMemory;
	_SharedMemory = 0;
	_MessageQueue = 0;

        throw TKinokoException(
	    "TKinokoBufferClient::TKinokoBufferClient()",
	    "system call exception: " + e.Message()
        );
    }

    _TimeKeeper = TimeKeeper;
    _Logger = Logger;

    _BaseAddress = (void*) _SharedMemory->BasePointer();
}

TKinokoBufferClient::~TKinokoBufferClient()
{
    delete _MessageQueue;
    delete _SharedMemory;
}

long TKinokoBufferClient::MessageQueueAddress(void)
{
    return _MessageQueue->Address();
}



TKinokoBufferWriter::TKinokoBufferWriter(int SharedMemoryProjectId, int MessageQueueProjectId, const string& ProjectPath, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
: TKinokoBufferClient(SharedMemoryProjectId, MessageQueueProjectId, ProjectPath, TimeKeeper, Logger)
{
    _NumberOfReserved = 0;
    _IsAttached = false;

    Attach();
}

TKinokoBufferWriter::~TKinokoBufferWriter()
{
    if (_IsAttached) {
        try {
	    Detach();
	}
	catch (TSystemCallException &e) {
            cerr << "ERROR: ~KinokoBufferWriter(): " << e << endl;
	}
    }
}

void TKinokoBufferWriter::Attach(void)
{
    if (_IsAttached) {
	return;
    }

    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptWriterAttachRequest;

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: BufferWriter::Attach()" + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: BufferWriter::Attach(): " + 
		e.Message() + " (no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}
	_Logger->Write(
	    "no response from BufferServer",
	    TKinokoBufferLogger::mlError
	);
    }
    else {
	_IsAttached = true;
    }
}

void TKinokoBufferWriter::Detach(void)
{
    if (! _IsAttached) {
	return;
    }

    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptWriterDetachRequest;

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: BufferWriter::Detach(): " + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: BufferWriter::Detach(): " + 
		e.Message() + "(no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}

	_Logger->Write(
	    "no response from BufferServer",
	    TKinokoBufferLogger::mlError
	);
	return;
    }

    _IsAttached = false;
}

int TKinokoBufferWriter::ReserveWriteEntry(size_t Capacity)
{
    if (_NumberOfReserved > 0) {
	return _NumberOfReserved;
    }

    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptWriteRequest;
    _Packet.Size = Capacity;

    int SentLength = 0;
    try {
	//_TimeKeeper->Start();
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
	//_TimeKeeper->Stop();
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: ReserveWriteEntry(): " + 
	    e.Message() + " (continue processing...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength > 0) {
	_NumberOfReserved++;
    }

    return _NumberOfReserved;
}

bool TKinokoBufferWriter::GetWriteEntry(void*& Address, size_t Capacity)
{
    if (ReserveWriteEntry(Capacity) == 0) {
	// Timed out //
	Address = 0;
	return false;
    }

    _TimeKeeper->Start();
    int ReceivedLength = _MessageQueue->Receive(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    _TimeKeeper->Stop();

    if (ReceivedLength > 0) {
	Address = (Int8*) _BaseAddress + _Packet.OffsetAddress;
	_NumberOfReserved--;
    }
    else {
	// timed out //
	Address = 0;
    }

    return (Address != 0);
}

bool TKinokoBufferWriter::SendStrobe(const void* Address, size_t Size)
{
    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptDataStrobe;
    _Packet.OffsetAddress = (off_t) ((Int8*) Address - (Int8*) _BaseAddress);
    _Packet.Size = Size;

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: SendStrobe(): " + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: SendStrobe(): " + 
		e.Message() + " (no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}
	if (SentLength <= 0) {
	    _Logger->Write(
		"no response from BufferServer", 
		TKinokoBufferLogger::mlError
	    );
	}
    }

    return (SentLength > 0);
}



TKinokoBufferReader::TKinokoBufferReader(int SharedMemoryProjectId, int MessageQueueProjectId, const string& ProjectPath, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
: TKinokoBufferClient(SharedMemoryProjectId, MessageQueueProjectId, ProjectPath, TimeKeeper, Logger)
{
    _IsAttached = false;
    _IsPacketDataAvailable = false;

    Attach();
}

TKinokoBufferReader::~TKinokoBufferReader()
{
    if (_IsAttached) {
        try {
	    Detach();
	}
	catch (TSystemCallException &e) {
            cerr << "ERROR: ~KinokoBufferReader(): " << e << endl;
	}
    }
}

void TKinokoBufferReader::Attach(void)
{
    if (_IsAttached) {
	return;
    }

    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptReaderAttachRequest;

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_OutgoingPacket, sizeof(_OutgoingPacket)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: BufferReader::Attach()" + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_OutgoingPacket, 
		sizeof(_OutgoingPacket)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: BufferReader::Attach(): " + 
		e.Message() + " (no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}
	_Logger->Write(
	    "no response from BufferServer",
	    TKinokoBufferLogger::mlError
	);
    }
    else {
	_IsAttached = true;
    }
}

void TKinokoBufferReader::Detach(void)
{
    if (! _IsAttached) {
	return;
    }

    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptReaderDetachRequest;

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_OutgoingPacket, 
	    sizeof(_OutgoingPacket)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: BufferReader::Detach(): " + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_OutgoingPacket, 
		sizeof(_OutgoingPacket)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: BufferReader::Detach(): " + 
		e.Message() + "(no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}

	_Logger->Write(
	    "no response from BufferServer",
	    TKinokoBufferLogger::mlError
	);
	return;
    }
    
    // flush remaining packets and wait for detach acknowledge //
    int NumberOfTries = 0;
    int MaxNumberOfTries = 128;
    while (_IsAttached) {
	_TimeKeeper->Start();
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket *) &_IncomingPacket, 
	    sizeof(_IncomingPacket)
	);
	_TimeKeeper->Stop();
	
	if (ReceivedLength > 0) {
	    switch (_IncomingPacket.PacketType) {
              case TKinokoBufferPacket::ptReadRequest:
		SendAcknowledge(
		    (Int8*) _BaseAddress + _IncomingPacket.OffsetAddress
		);
		break;
	      case TKinokoBufferPacket::ptReaderDetachAcknowledge:
		_IsAttached = false;
		break;
	      default:
		break;
	    }
	}
	else if (++NumberOfTries > MaxNumberOfTries) {
	    _Logger->Write(
		"no response from BufferServer for detach request",
		TKinokoBufferLogger::mlError
	    );
	    return;
	}
    }

    _IsAttached = false;
}

bool TKinokoBufferReader::HasData(void)
{
    if (! _IsPacketDataAvailable) {
	static const bool NoWait = true;
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket), 
	    NoWait
	);

	_IsPacketDataAvailable = (ReceivedLength > 0);
    }

    return _IsPacketDataAvailable;
}

int TKinokoBufferReader::NextDataSize(void)
{
    if (! _IsPacketDataAvailable) {
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket)
	);

	_IsPacketDataAvailable = (ReceivedLength > 0);
    }

    return _IsPacketDataAvailable ? _IncomingPacket.Size : 0;
}

bool TKinokoBufferReader::GetReadEntry(void*& Address, size_t& Size)
{
    if (_IsPacketDataAvailable) {
	Address = (Int8*) _BaseAddress + _IncomingPacket.OffsetAddress;
	Size = _IncomingPacket.Size;
    }
    else {
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket)
	);

	if (ReceivedLength > 0) {
	    Address = (Int8*) _BaseAddress + _IncomingPacket.OffsetAddress;
	    Size = _IncomingPacket.Size;
	}
	else {
	    Address = 0;
	    Size = 0;
	}
    }

    _IsPacketDataAvailable = false;

    return (Address != 0);
}

bool TKinokoBufferReader::SendAcknowledge(const void* Address)
{
    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptDataAcknowledge;
    _OutgoingPacket.OffsetAddress = (off_t) ((Int8*) Address - (Int8*) _BaseAddress);

    int SentLength = 0;
    try {
	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_OutgoingPacket, 
	    sizeof(_OutgoingPacket)
	);
    }
    catch (TSystemCallException &e) {
	_Logger->Write(
	    "MessageQueue Error: SendAcknowledge(): " + 
	    e.Message() + " (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );
	SentLength = 0;
    }

    if (SentLength <= 0) {
	_Logger->Write(
	    "no response from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	_TimeKeeper->Suspend();
	try {
	    SentLength = _MessageQueue->Send(
		(TMushMessageQueuePacket*) &_OutgoingPacket, 
		sizeof(_OutgoingPacket)
	    );
	}
	catch (TSystemCallException &e) {
	    _Logger->Write(
		"MessageQueue Error: SendAcknowledge(): " + 
		e.Message() + " (no more try)",
		TKinokoBufferLogger::mlWarning
	    );
	}
	if (SentLength <= 0) {
	    _Logger->Write(
		"no response from BufferServer (interrupted)", 
		TKinokoBufferLogger::mlError
	    );
	}
    }

    return (SentLength > 0);

}
