/* KinokoBufferServer.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Updated by Enomoto Sanshiro on 21 August 2002. */
/* Last updated by Enomoto Sanshiro on 16 January 2010. */


#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <cstdlib>
#include "MushDefs.hh"
#include "MushIpc.hh"
#include "MushMisc.hh"
#include "KinokoDefs.hh"
#include "KinokoArena.hh"
#include "KinokoBufferLogger.hh"
#include "KinokoBufferServer.hh"

using namespace std;
using namespace mush;


//#define DEBUG

#ifdef DEBUG
#define blog ofstream("Dump-Buffer", ios::app)
static void DumpPacket(const TKinokoBufferPacket& KinokoBufferPacket);
#endif


static const int KinokoBufferServerMessageQueueAddress = 1;
static const int PrescaledLoggingInterval_sec = 10;


TKinokoBufferServer::TKinokoBufferServer(int SharedMemoryProjectId, int MessageQueueProjectId, const std::string& ProjectPath, unsigned long BufferSize, int InitialEntryTableSize, TKinokoBufferLogger* Logger) throw(TKinokoException)
{
    _BufferSize = BufferSize;
    _Logger = Logger;

    _PrescaledLogger = new TKinokoPrescaledBufferLogger(
	_Logger, PrescaledLoggingInterval_sec
    );

    _SharedMemory = 0;
    _MessageQueue = 0;
    try {
	int AccessPermission = 0600;
	bool IsSelfish = true;
	_SharedMemory = new TMushServerSharedMemory(
	    _BufferSize, SharedMemoryProjectId, ProjectPath, 
	    AccessPermission, IsSelfish
	);
	_MessageQueue = new TMushServerMessageQueue(
	    MessageQueueProjectId, ProjectPath, 
	    AccessPermission, IsSelfish
	);
    }
    catch (TSystemCallException &e) {
	delete _MessageQueue;
	delete _SharedMemory;
	delete _PrescaledLogger;
	
	_SharedMemory = 0;
	_MessageQueue = 0;
	_PrescaledLogger = 0;
    
#if 1
	cerr << "ERROR: Unable to allocate IPC resource(s) for KinokoBuffer: ";
	cerr << e << endl;
	cerr << "  Project Path: " << ProjectPath << endl;
	cerr << "  Buffer Size: " << _BufferSize << endl;
	cerr << "  SharedMemory Project ID: " << SharedMemoryProjectId << endl;
	cerr << "  MessageQueue Project ID: " << MessageQueueProjectId << endl;
#endif
        throw TKinokoException(
	    "TKinokoBufferServer::TKinokoBufferServer()",
	    "system call exception: " + e.Message()
        );
    }

    _MessageQueue->SetAddress(this->MessageQueueAddress());
    _Arena.Initialize(_BufferSize, InitialEntryTableSize);

    _NumberOfReaders = 0;
    _NumberOfWriters = 0;

    int MaxNumberOfQueuePackets = (
	_MessageQueue->QueueCapacity() / sizeof(TKinokoBufferPacket)
    );
    _MaxTolerableNumberOfUnreadPackets = MaxNumberOfQueuePackets - 4;
    _NumberOfUnreadPackets = 0;

    _DataRate = 0;
    _PacketRate = 0;
    _WrittenDataSize = 0;
    _NumberOfWrittenPackets = 0;
    _LastReportTime = TMushDateTime().SecSinceEpoch();
}

TKinokoBufferServer::~TKinokoBufferServer()
{
    _Arena.Finalize();

    delete _MessageQueue;
    delete _SharedMemory;
    delete _PrescaledLogger;
}

long TKinokoBufferServer::MessageQueueAddress(void)
{
    return KinokoBufferServerMessageQueueAddress;
}

int TKinokoBufferServer::SharedMemoryId(void)
{
    return _SharedMemory->IpcId();
}

int TKinokoBufferServer::MessageQueueId(void)
{
    return _MessageQueue->IpcId();
}

int TKinokoBufferServer::DoTransaction(TMushIntervalTimer *TimeKeeper)
{
    // firstly try nowait receive since starting a timer may take time
    static const bool NoWait = true;
    int Length = _MessageQueue->Receive(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet), NoWait
    );

    // secondly use an interval timer for timeout processings
    if (Length <= 0) {
	TimeKeeper->Start();
	Length = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
	TimeKeeper->Stop();
    }

    if (Length > 0) {
#ifdef DEBUG
	DumpPacket(_Packet);
	blog << "FreeArea (before transaction): ";
	blog << _Arena.AvailableSize() << endl;
#endif

	DispatchMessage(_Packet);

#ifdef DEBUG
	blog << "FreeArea (after transaction): ";
	blog << _Arena.AvailableSize() << endl << endl;	
#endif
    }

    return (Length > 0);
}

void TKinokoBufferServer::DispatchMessage(const TKinokoBufferPacket &Packet)
{
    switch (Packet.PacketType) {
      case TKinokoBufferPacket::ptWriteRequest:
        if (! ProcessWriteRequest(TKinokoBufferRequest(Packet))) {
	    _WriteRequestQueue.push_back(TKinokoBufferRequest(Packet));
	}
	break;
      case TKinokoBufferPacket::ptDataStrobe:
        if (! ProcessReadRequest(TKinokoBufferRequest(Packet))) {
	    _ReadRequestQueue.push_back(TKinokoBufferRequest(Packet));
	}
	break;
      case TKinokoBufferPacket::ptDataAcknowledge:
	DetachEntry(Packet.OffsetAddress);
	ProcessPendingRequests();
	break;
      case TKinokoBufferPacket::ptReaderAttachRequest:
	AttachReader(Packet.SenderAddress);
	ProcessPendingRequests();
	break;
      case TKinokoBufferPacket::ptReaderDetachRequest:
	DetachReader(Packet.SenderAddress);
	ProcessPendingRequests();
	break;
      case TKinokoBufferPacket::ptWriterAttachRequest:
	AttachWriter(Packet.SenderAddress);
	ProcessPendingRequests();
	break;
      case TKinokoBufferPacket::ptWriterDetachRequest:
	DetachWriter(Packet.SenderAddress);
	ProcessPendingRequests();
	break;
      default:
	_Logger->Write("unknown buffer request", TKinokoBufferLogger::mlPanic);
    }
}

void TKinokoBufferServer::ProcessPendingRequests(void)
{
    while (! _ReadRequestQueue.empty()) {
	const TKinokoBufferRequest& ReadRequest = _ReadRequestQueue.front();
	if (ProcessReadRequest(ReadRequest)) {
	    _ReadRequestQueue.pop_front();
	}
	else {
	    break;
	}
    }

    while (! _WriteRequestQueue.empty()) {
	const TKinokoBufferRequest& WriteRequest = _WriteRequestQueue.front();
	if (ProcessWriteRequest(WriteRequest)) {
	    _WriteRequestQueue.pop_front();
	}
	else {
	    break;
	}
    }
}

bool TKinokoBufferServer::ProcessWriteRequest(const TKinokoBufferRequest& WriteRequest)
{
    bool IsRequestProcessed = false;

    if (_NumberOfReaders == 0) {
	_PrescaledLogger->Write(
            "no reader process (allocation deferred)", 
            TKinokoBufferLogger::mlWarning
        );
    }
    
    else if (_Arena.IsAllocatable(WriteRequest.Size)) {
	off_t OffsetAddress;
	if (_Arena.CreateEntry(
	    WriteRequest.Size, OffsetAddress, _NumberOfReaders
	)){
	    if (OffsetAddress + WriteRequest.Size > _BufferSize) {
		_Logger->Write(
		    "Invalid address allocated", TKinokoBufferLogger::mlPanic
		);
		abort();
	    }
	    
	    SendEntryReady(WriteRequest, OffsetAddress);
	    _WrittenDataSize += WriteRequest.Size;
	    _NumberOfWrittenPackets++;
	    IsRequestProcessed = true;
	}
	else {
	    if (_PrescaledLogger->Write(
		"unable to create buffer entry (allocation deferred)", 
		TKinokoBufferLogger::mlWarning
	    )){
		ReportStatus();
	    }
	}
    }
    else if (WriteRequest.Size > _BufferSize) {
	_Logger->Write(
	    "too large data area requested (request ignored)", 
	    TKinokoBufferLogger::mlError
	);

        // throw away the request
	IsRequestProcessed = true;
    }
    else {
	if (_PrescaledLogger->Write(
	    "buffer full (allocation deferred)", 
	    TKinokoBufferLogger::mlWarning
	)){
	    ReportStatus();
	}
    }

    return IsRequestProcessed;
}

void TKinokoBufferServer::SendEntryReady(const TKinokoBufferRequest& WriteRequest, off_t OffsetAddress)
{
    _Packet.ReceiverAddress = WriteRequest.WriterMessageAddress;
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptEntryReady;
    _Packet.OffsetAddress = OffsetAddress;
    _Packet.Size = WriteRequest.Size;

#ifdef DEBUG
    blog << endl;
    DumpPacket(_Packet);
    blog << "FreeArea (after allocation): " << _Arena.AvailableSize() << endl;
#endif
    
    int SentLength =  _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    if (SentLength == 0) {
	_Logger->Write(
	    "SendEntryReady(): "
	    "incomplete packet processing (interrupted)", 
	    TKinokoBufferLogger::mlError
	);
	ReportStatus();
    }
}

bool TKinokoBufferServer::ProcessReadRequest(const TKinokoBufferRequest& ReadRequest)
{
    if (_NumberOfUnreadPackets < _MaxTolerableNumberOfUnreadPackets) {
	SendReadRequest(ReadRequest);
	return true;
    }
    else if (_MaxTolerableNumberOfUnreadPackets < _NumberOfReaders) {
	// we cannot avoid deadlocks anyways... //
	if (_NumberOfUnreadPackets == 0) {
	    SendReadRequest(ReadRequest);
	    return true;
	}
	else {
	    return false;
	}
    }
    else {
	return false;
    }
}

void TKinokoBufferServer::SendReadRequest(const TKinokoBufferRequest& ReadRequest)
{
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptReadRequest;
    _Packet.OffsetAddress = ReadRequest.OffsetAddress;
    _Packet.Size = ReadRequest.Size;

    int SentLength;
    for (int i = 0; i < _NumberOfReaders; i++) {
	_Packet.ReceiverAddress = _ReaderAddressList[i];

	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
	if (SentLength == 0) {
	    _Logger->Write(
		"SendReadRequest(): "
		"incomplete packet processing (interrupted)", 
		TKinokoBufferLogger::mlError
	    );
	    ReportStatus();
	    break;
	}

	_NumberOfUnreadPackets++;
    }
}

void TKinokoBufferServer::DetachEntry(off_t OffsetAddress)
{
    if (! _Arena.DetachEntry(OffsetAddress)) {
	ostringstream MessageStream;
	MessageStream << "releasing invalid address: " << OffsetAddress;
	_Logger->Write(
	    MessageStream.str(), TKinokoBufferLogger::mlPanic
	);
    }

    _NumberOfUnreadPackets--;
}

void TKinokoBufferServer::AttachReader(long ReaderAddress)
{
    _ReaderAddressList.push_back(ReaderAddress);
    _NumberOfReaders++;
    _MaxTolerableNumberOfUnreadPackets -= 2;

    if (_MaxTolerableNumberOfUnreadPackets < _NumberOfReaders) {
	_Logger->Write(
	    "too many readers/writers attached, or message queue too small " 
	    "(message queue deadlocks expected)", 
            TKinokoBufferLogger::mlWarning
	);
    }

    ostringstream os;
    os << "reader attached: pid=" << ReaderAddress;
    _Logger->Write(os.str(), TKinokoBufferLogger::mlInformation);
}

void TKinokoBufferServer::DetachReader(long ReaderAddress)
{
    vector<long>::iterator Reader = find(
        _ReaderAddressList.begin(), _ReaderAddressList.end(), ReaderAddress
    );

    if (Reader == _ReaderAddressList.end()) {
	_Logger->Write(
	    "detach request from unattached reader", 
            TKinokoBufferLogger::mlPanic
	);
	return;
    }

    _ReaderAddressList.erase(Reader);
    _NumberOfReaders--;
    _MaxTolerableNumberOfUnreadPackets += 2;

    _Packet.ReceiverAddress = ReaderAddress;
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptReaderDetachAcknowledge;

    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    if (SentLength == 0) {
	_Logger->Write(
	    "DetachReader(): "
	    "incomplete packet processing (interrupted)", 
	    TKinokoBufferLogger::mlError
	);
	ReportStatus();
    }

    ostringstream os;
    os << "reader detached: pid=" << ReaderAddress;
    _Logger->Write(os.str(), TKinokoBufferLogger::mlInformation);
}

void TKinokoBufferServer::AttachWriter(long WriterAddress)
{
    _NumberOfWriters++;
    _MaxTolerableNumberOfUnreadPackets -= 1;

    if (_MaxTolerableNumberOfUnreadPackets < _NumberOfReaders) {
	_Logger->Write(
	    "too many readers/writers attached, or message queue too small " 
	    "(message queue deadlocks expected)", 
            TKinokoBufferLogger::mlWarning
	);
    }

    ostringstream os;
    os << "writer attached: pid=" << WriterAddress;
    _Logger->Write(os.str(), TKinokoBufferLogger::mlInformation);
}

void TKinokoBufferServer::DetachWriter(long WriterAddress)
{
    _NumberOfWriters--;
    _MaxTolerableNumberOfUnreadPackets += 1;


    ostringstream os;
    os << "writer dettached: pid=" << WriterAddress;
    _Logger->Write(os.str(), TKinokoBufferLogger::mlInformation);
}

void TKinokoBufferServer::ReportStatus(void)
{
    int PassedTime = TMushDateTime::SecSince(_LastReportTime);
    int NumberOfUnread = _ReadRequestQueue.size() * _NumberOfReaders;
    _DataRate = (float) _WrittenDataSize / PassedTime;
    _PacketRate = (float) _NumberOfWrittenPackets / PassedTime;

    ostringstream os;
    os << "used: " << _Arena.UsedSize();
    os << "/" << _BufferSize << ", ";
    os << "entries: " << _Arena.NumberOfEntries() << ", ";
    os << "head:tail: " << _Arena.DataHeadOffset();
    os << ":" << _Arena.NextStartOffset() << ", ";
    os << "written(size:entries)/time: " << _WrittenDataSize;
    os << ":" << _NumberOfWrittenPackets;
    os << "/" << PassedTime << ", ";
    os << "unread(queue1:queue2): " << NumberOfUnread;
    os << ":" << _NumberOfUnreadPackets;
    os << "/" << _MaxTolerableNumberOfUnreadPackets;

    _Logger->Write(os.str(), TKinokoBufferLogger::mlLog);

    _WrittenDataSize = 0;
    _NumberOfWrittenPackets = 0;
    _LastReportTime = TMushDateTime::SecSinceEpoch();
}

void TKinokoBufferServer::GetStatistics(float& BufferUsage, float& NumberOfEntries, float& DataRate, float& PacketRate)
{
    int PassedTime = TMushDateTime::SecSince(_LastReportTime);

    BufferUsage = (float) _Arena.UsedSize() / _BufferSize;
    NumberOfEntries = (float) _Arena.NumberOfEntries();
    if (PassedTime > 0) {
	DataRate = (float) _WrittenDataSize / PassedTime;
	PacketRate = (float) _NumberOfWrittenPackets / PassedTime;
    }
    else {
	DataRate = _DataRate;
	PacketRate = _PacketRate;
    }
}

#ifdef DEBUG
static void DumpPacket(const TKinokoBufferPacket& KinokoBufferPacket)
{
    if (KinokoBufferPacket.PacketType == TKinokoBufferPacket::ptWriteRequest) {
	blog << "========================" << endl;
    }

    blog << "From: " << KinokoBufferPacket.SenderAddress << endl;
    blog << "To: " << KinokoBufferPacket.ReceiverAddress << endl;
    
    switch (KinokoBufferPacket.PacketType) {
      case TKinokoBufferPacket::ptWriteRequest:
	blog << "Message-Type: " << "WriteRequest" << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptEntryReady:
	blog << "Message_Type: " << "EntryReady" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptDataStrobe:
	blog << "Message-Type: " << "DataStrobe" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptReadRequest:
	blog << "Message-Type: " << "ReadRequest" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptDataAcknowledge:
	blog << "Message-Type: " << "DataAcknowledge" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptReaderAttachRequest:
	blog << "Message-Type: " << "ReaderAttachRequest" << endl;
	break;
      case TKinokoBufferPacket::ptReaderDetachRequest:
	blog << "Message-Type: " << "ReaderDetachRequest" << endl;
	break;
      default:
	blog << "Message-Type: " << "UNDEFINED" << endl;
    }
}
#endif
