/* KinokoBuilder.cc */
/* Created by Enomoto Sanshiro on 25 August 2009. */
/* Updated by Enomoto Sanshiro on 19 May 2010. */
/* Last updated by Enomoto Sanshiro on 7 July 2010. */


#include <string>
#include <iostream>
#include <vector>
#include <exception>
#include "KinokoArena.hh"
#include "KinokoDataProcessor.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoBuilderInlet.hh"
#include "KinokoEventChopper.hh"
#include "KinokoEventSorter.hh"
#include "KinokoSorterProcessor.hh"
#include "KinokoEventEnvelope.hh"
#include "KinokoSorterController.hh"
#include "KinokoBuilderMonitor.hh"
#include "KinokoBuilder.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)

static const int InitialBufferSize = 0x10000;


TKinokoBuilder::TKinokoBuilder(void)
{
    _MyMonitor = new TKinokoBuilderMonitor();

    _Logger = 0;
    _Monitor = _MyMonitor;
    _SorterController = 0;

    _Buffer = 0;
    _BufferSize = InitialBufferSize;

    _EnvelopeWriter = 0;

    _StartDataSize = 0;
    _StartPacketCycles = 0;

    _IsInputPreserved = false;

    DEBUG(cout << "debug flag enabled for KinokoBuilder.cc" << endl);
}

TKinokoBuilder::~TKinokoBuilder()
{
    delete _SorterController;
    delete _EnvelopeWriter;

    _Arena.Finalize();
    delete[] _Buffer;

    delete _MyMonitor;
}

void TKinokoBuilder::SetOutputDataSourceName(const std::string& DataSourceName)
{
    _OutputDataSourceName = DataSourceName;
}

void TKinokoBuilder::SetBufferSize(size_t BufferSize)
{
    if (_BufferSize > 0) {
	_BufferSize = BufferSize;
    }
    else {
	_BufferSize = InitialBufferSize;
    }
}

void TKinokoBuilder::SetStartDataSize(size_t StartDataSize)
{
    _StartDataSize = StartDataSize;
}

void TKinokoBuilder::SetStartPacketCycles(int NumberOfPacketCycles)
{
    _StartPacketCycles = NumberOfPacketCycles;
}

void TKinokoBuilder::SetStartChannelList(const std::vector<long>& ChannelList)
{
    _StartChannelList = ChannelList;
}

void TKinokoBuilder::SetBuilderLogger(TKinokoBuilderLogger* Logger)
{
    _Logger = Logger;
}

void TKinokoBuilder::SetBuilderMonitor(TKinokoBuilderMonitor* Monitor)
{
    _Monitor = Monitor;
}

void TKinokoBuilder::SetInputPreservation(bool IsInputPreserved)
{
    _IsInputPreserved = IsInputPreserved;
}

void TKinokoBuilder::SetActivityWindowLength(int WindowLength)
{
    TKinokoChannelWatcher::SetDefaultActivityWindowLength(WindowLength);
}

void TKinokoBuilder::Construct(const std::string& Name, TKinokoInputStream* InputStream, TKinokoOutputStream* OutputStream, int DataSourceId) throw(TKinokoException)
{
    if (_OutputDataSourceName.empty()) {
	_OutputDataSourceName = Name;
    }
    
    TKinokoDataProcessor::Construct(
	_OutputDataSourceName, InputStream, OutputStream, DataSourceId
    );
}

void TKinokoBuilder::OnConstruct(void) throw(TKinokoException) 
{
    if (_Buffer) {
	return;
    }

    try {
	_Buffer = new char[_BufferSize];
    }
    catch (std::bad_alloc& e) {
	_Buffer = 0;
    }
    if (_Buffer == 0) {
	throw TKinokoException(
	    "TKinokoBuilder::OnConstruct()",
	    "unable to allocate buffer: size too large?"
	);
    }
    _Arena.Initialize(_BufferSize);

    _EnvelopeWriter = new TKinokoEventEnvelopeWriter(this);
    _EnvelopeWriter->SetLogger(_Logger);

    TKinokoEventSorter& Sorter = BuilderConstruct();
    _SorterController = new TKinokoSorterController(Sorter, _Arena);
    _Monitor->SetSorter(&Sorter);

    if (! _StartChannelList.empty()) {
	for (unsigned i = 0; i < _StartChannelList.size(); i++) {
	    _SorterController->RegisterStream(_StartChannelList[i]);
	}
    }
    if (_StartDataSize != 0) {
	_SorterController->SetStartDataSize(_StartDataSize);
    }
    if (_StartPacketCycles != 0) {
	_SorterController->SetStartPacketCycles(_StartPacketCycles);
    }
}

void TKinokoBuilder::OnDestruct(void) throw(TKinokoException)
{
    if (_Buffer == 0) {
	return;
    }

    delete _SorterController;
    delete _EnvelopeWriter;

    _Arena.Finalize();
    delete[] _Buffer;

    _SorterController = 0;
    _EnvelopeWriter = 0;
    _Buffer = 0;
}

void TKinokoBuilder::BuildDataSource(TKinokoDataSource* DataSource)
{
    if (_EnvelopeWriter == 0) {
	OnConstruct();
    }
    _EnvelopeWriter->BuildDataSource(DataSource);
}

void TKinokoBuilder::OnRunBegin(void) throw(TKinokoException)
{
    _SorterController->ProcessRunBegin();
    BuilderBegin();
}

void TKinokoBuilder::OnRunEnd(void) throw(TKinokoException)
{
    BuilderEnd();

    DEBUG(cout << "##### SORTER STATUS ON THE RUN END #####" << endl);
    DEBUG(_Monitor->Dump(cout));
    DEBUG(cout << "##### - E - N - D - of - E - N - D #####" << endl);
}

void TKinokoBuilder::OnReceivePacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    if (BuilderProcess(Packet, PacketSize)) {
	DEBUG(cout << "##### SORTER STATUS #####" << endl);
	DEBUG(_Monitor->Dump(cout));
	DEBUG(cout << "##### - E - N - D - #####" << endl);

	_SorterController->DoTransaction();

	if (_IsInputPreserved) {
	    SendPacket(Packet, PacketSize);
	}
    }
    else {
	SendPacket(Packet, PacketSize);
    }
    
}
