/* KinokoBuilderMonitor.cc */
/* Created by Enomoto Sanshiro on 1 September 2009. */
/* Last updated by Enomoto Sanshiro on 7 July 2010. */


#include <string>
#include <iostream>
#include <sstream>
#include "KinokoEventSorter.hh"
#include "KinokoBuilderLogger.hh"
#include "KinokoBuilderMonitor.hh"

using namespace std;


TKinokoBuilderMonitor::TKinokoBuilderMonitor(void)
{
}

TKinokoBuilderMonitor::~TKinokoBuilderMonitor(void)
{
}

void TKinokoBuilderMonitor::SetSorter(const TKinokoEventSorter* Sorter)
{
    _Sorter = Sorter;
}



TKinokoBuilderLoggingMonitor::TKinokoBuilderLoggingMonitor(TKinokoBuilderLogger* Logger)
{
    _Logger = Logger;
}

TKinokoBuilderLoggingMonitor::~TKinokoBuilderLoggingMonitor()
{
}

void TKinokoBuilderLoggingMonitor::Report(void)
{
    if (_Sorter == 0) {
	return;
    }

    {
	ostringstream os;
	os << "built: " << _Sorter->NumberOfBuiltEvents() << ", ";
	os << "pending: " << _Sorter->HoldingEventSet().size() << ", ";
	os << "last built key: " << _Sorter->LastBuiltEventKey() << ", ";
	os << "head:tail key: ";
	if (_Sorter->HoldingEventSet().empty()) {
	    os << "none/none";
	}
	else {
	    os << _Sorter->HoldingEventSet().begin()->first;
	    os << ":" << _Sorter->HoldingEventSet().rbegin()->first;
	}
	_Logger->Write(os.str(), TKinokoBuilderLogger::mlLog);
    }

    {
	ostringstream os;
	const TKinokoArena& Arena = _Sorter->Arena();
	os << "buffer used: " << Arena.UsedSize();
	os << "/" << Arena.Capacity() << ", ";
	os << "entries: " << Arena.NumberOfEntries() << ", ";
	os << "head:tail: " << Arena.DataHeadOffset();
	os << ":" << Arena.NextStartOffset();
	_Logger->Write(os.str(), TKinokoBuilderLogger::mlLog);
    }
}



void TKinokoBuilderMonitor::Dump(std::ostream& Output)
{
    if (_Sorter == 0) {
	return;
    }

    // event queue status //
    int NumberOfBuiltEvents = _Sorter->NumberOfBuiltEvents();
    Output << "# of built: " << NumberOfBuiltEvents << endl;

    TEventKey LastBuiltEventKey = _Sorter->LastBuiltEventKey();
    if (LastBuiltEventKey >= 0) {
	Output << "last built: " << LastBuiltEventKey << endl;
    }
    else {
	Output << "last built: none" << endl;
    }

    Output << "pending: ";
    const map<TEventKey, pair<int, size_t> >& HoldingEventSet = (
	_Sorter->HoldingEventSet()
    );
    map<TEventKey, pair<int, size_t> >::const_iterator EventEntry;
    for (
	EventEntry = HoldingEventSet.begin();
	EventEntry != HoldingEventSet.end();
	EventEntry++
    ){
	Output << EventEntry->first << "(";
	Output << EventEntry->second.first << ",";
	Output << EventEntry->second.second << ") ";
    }
    if (HoldingEventSet.empty()) {
	Output << "none";
    }
    Output << endl;
    
    // arena details //
    const TKinokoArena& Arena = _Sorter->Arena();
    Output << "buffer: ";
    Output << "size=" << Arena.UsedSize();
    Output << "(" << (int) (100.0*Arena.UsedSize()/Arena.Capacity()) << "%), ";
    Output << "entries=" << Arena.NumberOfEntries() << endl;

    // channel details //

    const vector<TChannelKey>& ChannelKeyList = (
	_Sorter->ChannelKeyList()
    );
    const vector<TKinokoChannelWatcher>& ChannelWatcherList = (
	_Sorter->ChannelWatcherList()
    );
    const vector<TEventKey>& CompleteEventKeyList = (
	_Sorter->CompleteEventKeyList()
    );
    const vector<std::deque<TEventKey> >& EventKeyQueueList = (
	_Sorter->EventKeyQueueList()
    );
    const vector<std::set<TEventKey> >& EarlyArrivalEventKeySetList = (
	_Sorter->EarlyArrivalEventKeySetList()
    );
    for (unsigned i = 0; i < ChannelKeyList.size(); i++) {
	Output << "channel " << ChannelKeyList[i] << ": ";
	if (ChannelWatcherList[i].IsAlive()) {
	    Output << "ALIVE";
	}
	else {
	    Output << "DEAD";
	}
	Output << "(" << ChannelWatcherList[i].AbsenceCount() << "/";
	Output << (int) (100*ChannelWatcherList[i].Occupancy()) << "%), ";

	if (ChannelWatcherList[i].AbandonedCount() > 0) {
	    Output << "abandoned ";
	    Output << ChannelWatcherList[i].AbandonedCount() << ", ";
	}
	if (ChannelWatcherList[i].SuspensionCount() > 0) {
	    Output << "suspended ";
	    Output << ChannelWatcherList[i].SuspensionCount() << ", ";
	}

	if (CompleteEventKeyList[i] >= 0) {
	    Output << "completed " << CompleteEventKeyList[i] << ", ";
	}
	else {
	    Output << "no complete event, ";
	}

	if (! EventKeyQueueList[i].empty()) {
	    Output << "pending ";
	    Output << *EventKeyQueueList[i].begin() << "-";
	    Output << *EventKeyQueueList[i].rbegin() << ":";
	    Output << EventKeyQueueList[i].size() << ", ";
	}
	else {
	    Output << "no pending pieces, ";
	}

	if (! EarlyArrivalEventKeySetList[i].empty()) {
	    Output << "early arrivals ";
	    Output << *EarlyArrivalEventKeySetList[i].begin() << "-";
	    Output << *EarlyArrivalEventKeySetList[i].rbegin() << ":";
	    Output << EarlyArrivalEventKeySetList[i].size();
	}
	else {
	    Output << "no early arrivals";
	}
	Output << endl;
    }    
}
