/* KinokoDataChunk.hh */
/* Created by Enomoto Sanshiro on 11 August 2009. */
/* Last updated by Enomoto Sanshiro on 11 August 2009. */


#ifndef __KinokoDataChunk_hh__
#define __KinokoDataChunk_hh__

#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <cstring>
#include "KinokoDefs.hh"


class TKinokoDataChunk {
  public:
    inline TKinokoDataChunk(void);
    inline TKinokoDataChunk(void* Start, size_t Size);
    inline TKinokoDataChunk(const TKinokoDataChunk& DataChunk);
    inline TKinokoDataChunk& operator=(const TKinokoDataChunk& DataChunk);
    inline void PushFront(void* Start, size_t Size);
    inline void PushBack(void* Start, size_t Size);
    inline void CutFrontBy(size_t Size);
    inline void CutBackTo(size_t NewSize);
    inline void Clear(void);
    inline U32bit& operator[](unsigned WordIndex);
    inline const U32bit& operator[](unsigned WordIndex) const;
    inline int Size(void) const;
    inline bool IsEmpty(void) const;
    inline int WriteTo(void* Destination) const;
    inline void Dump(std::ostream& Output) const;
  private:
    char* _PrimaryStart;
    size_t _PrimarySize;
    char* _SecondaryStart;
    size_t _SecondarySize;
};


inline TKinokoDataChunk::TKinokoDataChunk(void)
{
    _PrimaryStart = 0;
    _PrimarySize = 0;
    _SecondaryStart = 0;
    _SecondarySize = 0;
}

inline TKinokoDataChunk::TKinokoDataChunk(void* Start, size_t Size)
{
    _PrimaryStart = (char*) Start;
    _PrimarySize = Size;
    _SecondaryStart = 0;
    _SecondarySize = 0;
}

inline TKinokoDataChunk::TKinokoDataChunk(const TKinokoDataChunk& DataChunk)
{
    _PrimaryStart = DataChunk._PrimaryStart;
    _PrimarySize = DataChunk._PrimarySize;
    _SecondaryStart = DataChunk._SecondaryStart;
    _SecondarySize = DataChunk._SecondarySize;
}

inline TKinokoDataChunk& TKinokoDataChunk::operator=(const TKinokoDataChunk& DataChunk)
{
    _PrimaryStart = DataChunk._PrimaryStart;
    _PrimarySize = DataChunk._PrimarySize;
    _SecondaryStart = DataChunk._SecondaryStart;
    _SecondarySize = DataChunk._SecondarySize;
    return *this;
}

inline void TKinokoDataChunk::PushFront(void* Start, size_t Size)
{
    if (_SecondaryStart != 0) {
	std::abort();
    }
    _SecondaryStart = _PrimaryStart;
    _SecondarySize = _PrimarySize;

    _PrimaryStart = (char*) Start;
    _PrimarySize = Size;
}

inline void TKinokoDataChunk::PushBack(void* Start, size_t Size)
{
    if (_PrimaryStart == 0) {
	_PrimaryStart = (char*) Start;
	_PrimarySize = Size;
    }
    else if (_SecondaryStart == 0) {
	_SecondaryStart = (char*) Start;
	_SecondarySize = Size;
    }
    else {
	std::abort();
    }
}

inline void TKinokoDataChunk::CutFrontBy(size_t Size)
{
    if (Size < _PrimarySize) {
	_PrimaryStart += Size;
	_PrimarySize -= Size;
    }
    else if (Size < _PrimarySize + _SecondarySize) {
	_PrimaryStart = _SecondaryStart + (Size - _PrimarySize);
	_PrimarySize = _SecondarySize - (Size - _PrimarySize);
	_SecondaryStart = 0;
	_SecondarySize = 0;
    }
    else {
	_PrimaryStart = 0;
	_PrimarySize = 0;
	_SecondaryStart = 0;
	_SecondarySize = 0;
    }
}

inline void TKinokoDataChunk::CutBackTo(size_t NewSize)
{
    if (NewSize <= _PrimarySize) {
	_PrimarySize = NewSize;
	_SecondaryStart = 0;
	_SecondarySize = 0;
    }
    else {
	_SecondarySize = NewSize - _PrimarySize;
    }
}

inline void TKinokoDataChunk::Clear(void)
{
    _PrimaryStart = 0;
    _PrimarySize = 0;
    _SecondaryStart = 0;
    _SecondarySize = 0;
}

inline U32bit& TKinokoDataChunk::operator[](unsigned WordIndex)
{
    if ((WordIndex << 2) < _PrimarySize) {
	return ((U32bit*) _PrimaryStart)[WordIndex];
    }
    else {
	return ((U32bit*) _SecondaryStart)[WordIndex - (_PrimarySize>>2)];
    }
}

inline const U32bit& TKinokoDataChunk::operator[](unsigned WordIndex) const
{
    if ((WordIndex << 2) < _PrimarySize) {
	return ((U32bit*) _PrimaryStart)[WordIndex];
    }
    else {
	return ((U32bit*) _SecondaryStart)[WordIndex - (_PrimarySize>>2)];
    }
}

inline int TKinokoDataChunk::Size(void) const
{
    return _PrimarySize + _SecondarySize;
}

inline bool TKinokoDataChunk::IsEmpty(void) const
{
    return (_PrimarySize == 0);
}

inline int TKinokoDataChunk::WriteTo(void* Destination) const
{
    if (_PrimaryStart && (Destination != _PrimaryStart)) {
	std::memcpy(Destination, _PrimaryStart, _PrimarySize);
    }

    if (_SecondaryStart) {
	std::memcpy(
	    (char*) Destination + _PrimarySize, 
	    _SecondaryStart, _SecondarySize
	);
    }
    
    return _PrimarySize + _SecondarySize;
}

inline void TKinokoDataChunk::Dump(std::ostream& Output) const
{
    for (int i = 0; i < Size()/4; i++) {
	Output << std::hex << std::setfill('0') << std::setw(8);
	Output << (*this)[i] << std::dec << std::endl;
    }
}

#endif
