/* KinokoEventSegmentPacker.cc */
/* Created by Enomoto Sanshiro on 23 April 2010. */
/* Last updated by Enomoto Sanshiro on 23 April 2010. */


#include <iostream>
#include "KinokoEventPiece.hh"
#include "KinokoEventSegmentPacker.hh"

using namespace std;


TKinokoEventSegmentPacker::TKinokoEventSegmentPacker(TKinokoEventSegmentProcessor* SegmentProcessor, TStreamKey StreamKey)
{
    _SegmentProcessor = SegmentProcessor;
    _StreamKey = StreamKey;

    _Buffer = 0;
    _BufferSize = 0;

    _SinglePieceBuffer = 0;
    _SinglePieceBufferSize = 0;

    _CurrentSegmentIterator = 0;
}

TKinokoEventSegmentPacker::~TKinokoEventSegmentPacker()
{
    delete[] _Buffer;
    delete[] _SinglePieceBuffer;
}

void TKinokoEventSegmentPacker::ProcessRunBegin(void)
{
    _SegmentProcessor->ProcessRunBegin();
}

void TKinokoEventSegmentPacker::ProcessRunEnd(void)
{
    _SegmentProcessor->ProcessRunEnd();
}

void TKinokoEventSegmentPacker::ProcessEventBegin(TEventKey EventKey, int PieceType, int NumberOfPieces, size_t TotalSize)
{
    if (_CurrentSegmentIterator != 0) {
	cerr << "INTERNAL ERROR: ";
	cerr << "TKinokoEventPiecePacker::ProcessEventBegin(): ";
	cerr << "unmatched ProcessEventBegin()/End()" << endl;
	abort();
    }

    _CurrentPieceType = PieceType;
    _CurrentEventNumberOfPieces = NumberOfPieces;
    _CurrentEventTotalSize = TotalSize;
    _CurrentEventKey = EventKey;

    _CurrentSegmentSize = TotalSize + sizeof(U32bit) * (2 * NumberOfPieces + 3);

    if (_BufferSize < _CurrentSegmentSize) {
	_BufferSize = 2*_CurrentSegmentSize;
	delete[] _Buffer;
	int AlignedLength = _BufferSize/sizeof(U32bit) + 1;
	_Buffer = new U32bit[AlignedLength];
    }

    _CurrentSegmentIterator = _Buffer;
    _CurrentSegmentIterator += 3;
}

void TKinokoEventSegmentPacker::ProcessEventEnd(void)
{
    if (_CurrentSegmentIterator == 0) {
	cerr << "INTERNAL ERROR: ";
	cerr << "TKinokoEventPiecePacker::ProcessEventEnd(): ";
	cerr << "unmatched ProcessEventBegin()/End()" << endl;
	abort();
    }

    TKinokoDataChunk SegmentDataChunk(_Buffer, _CurrentSegmentSize);
    TKinokoEventSegment Segment(SegmentDataChunk);
    Segment.SetHeader(
	_CurrentPieceType, _CurrentEventNumberOfPieces, _CurrentEventKey
    );
    Segment.StreamKey = _StreamKey;

    _SegmentProcessor->ProcessEventBegin(
	_CurrentEventKey, _CurrentPieceType, 
	_CurrentEventNumberOfPieces, _CurrentEventTotalSize
    );
    _SegmentProcessor->ProcessSegment(Segment);
    _SegmentProcessor->ProcessEventEnd();

    _CurrentSegmentIterator = 0;
}

int TKinokoEventSegmentPacker::ProcessPiece(const TKinokoEventPiece& EventPiece)
{
    if (_CurrentSegmentIterator == 0) {
	return ProcessSinglePiece(EventPiece);
    }
    if (EventPiece.PieceType != _CurrentPieceType) {
	return ProcessSinglePiece(EventPiece);
    }

    //... TODO: Check errors here ...//

    *(_CurrentSegmentIterator++) = EventPiece.ChannelKey;
    *(_CurrentSegmentIterator++) = EventPiece.Size();

    EventPiece.WriteTo(_CurrentSegmentIterator);
    _CurrentSegmentIterator += EventPiece.Size() / sizeof(U32bit);

    return 1;
}

int TKinokoEventSegmentPacker::ProcessSinglePiece(const TKinokoEventPiece& EventPiece)
{
    int NumberOfPieces = 1;
    int TotalSize = EventPiece.Size();
    unsigned SegmentSize = TotalSize + sizeof(U32bit) * 5;

    if (_SinglePieceBufferSize < SegmentSize) {
	_SinglePieceBufferSize = 2*SegmentSize;
	delete[] _SinglePieceBuffer;
	int AlignedLength = _SinglePieceBufferSize/sizeof(U32bit) + 1;
	_SinglePieceBuffer = new U32bit[AlignedLength];
    }
    EventPiece.WriteTo(_SinglePieceBuffer+5);

    TKinokoEventSegment Segment(
	TKinokoDataChunk(_SinglePieceBuffer, SegmentSize)
    );
    Segment.SetHeader(
	EventPiece.PieceType, NumberOfPieces, EventPiece.EventKey
    );
    Segment.StreamKey = _StreamKey;
    Segment[3] = EventPiece.ChannelKey;
    Segment[4] = EventPiece.Size();

    _SegmentProcessor->ProcessEventBegin(
	EventPiece.EventKey, EventPiece.PieceType, 
	NumberOfPieces, TotalSize
    );
    _SegmentProcessor->ProcessSegment(Segment);
    _SegmentProcessor->ProcessEventEnd();

    return 1;
}



TKinokoEventSegmentUnpacker::TKinokoEventSegmentUnpacker(TKinokoEventPieceProcessor* PieceProcessor)
{
    _PieceProcessor = PieceProcessor;
}

TKinokoEventSegmentUnpacker::~TKinokoEventSegmentUnpacker()
{
}

void TKinokoEventSegmentUnpacker::ProcessRunBegin(void)
{
    _PieceProcessor->ProcessRunBegin();
}

void TKinokoEventSegmentUnpacker::ProcessRunEnd(void)
{
    _PieceProcessor->ProcessRunEnd();
}

void TKinokoEventSegmentUnpacker::ProcessEventBegin(TEventKey EventKey, int PieceType, int NumberOfPieces, size_t TotalPieceSize)
{
    _PieceProcessor->ProcessEventBegin(
	EventKey, PieceType, NumberOfPieces, TotalPieceSize
    );
}

void TKinokoEventSegmentUnpacker::ProcessEventEnd(void)
{
    _PieceProcessor->ProcessEventEnd();
}

int TKinokoEventSegmentUnpacker::ProcessSegment(const TKinokoEventSegment& EventSegment)
{
    int Result = 0;

    int PieceType = EventSegment.PieceType();
    TEventKey EventKey = EventSegment.EventKey();

    TKinokoDataChunk SegmentChunk(EventSegment);
    SegmentChunk.CutFrontBy(3*sizeof(U32bit));

    while (SegmentChunk.Size() >= (int) (2 * sizeof(U32bit))) {
	TKinokoEventPiece EventPiece(SegmentChunk);
	TChannelKey ChannelKey = EventPiece[0];
	size_t Size = EventPiece[1];
	EventPiece.CutFrontBy(2*sizeof(U32bit));
	EventPiece.CutBackTo(Size);

	EventPiece.PieceType = PieceType;
	EventPiece.EventKey = EventKey;
	EventPiece.ChannelKey = ChannelKey;

	Result += _PieceProcessor->ProcessPiece(EventPiece);
	SegmentChunk.CutFrontBy(2*sizeof(U32bit) + Size);
    }

    return Result;
}



TKinokoEventSegmentConsolidator::TKinokoEventSegmentConsolidator(TKinokoEventSegmentProcessor* OutputProcessor, TStreamKey StreamKey)
{
    _OutputProcessor = OutputProcessor;
    _StreamKey = StreamKey;

    _Buffer = 0;
    _BufferSize = 0;

    _CurrentSegmentIterator = 0;
}

TKinokoEventSegmentConsolidator::~TKinokoEventSegmentConsolidator()
{
    delete[] _Buffer;
}

void TKinokoEventSegmentConsolidator::ProcessRunBegin(void)
{
    _OutputProcessor->ProcessRunBegin();
}

void TKinokoEventSegmentConsolidator::ProcessRunEnd(void)
{
    _OutputProcessor->ProcessRunEnd();
}

void TKinokoEventSegmentConsolidator::ProcessEventBegin(TEventKey EventKey, int PieceType, int NumberOfPieces, size_t TotalSize)
{
    if (_CurrentSegmentIterator != 0) {
	cerr << "INTERNAL ERROR: ";
	cerr << "TKinokoEventSegmentConsolidator::ProcessEventBegin(): ";
	cerr << "unmatched ProcessEventBegin()/End()" << endl;
	abort();
    }

    _CurrentPieceType = PieceType;
    _CurrentEventNumberOfPieces = 0;
    _CurrentEventTotalSize = 0;
    _CurrentEventKey = EventKey;

    int NumberOfSegments = NumberOfPieces;
    int TotalSegmentSize = TotalSize;
    _CurrentSegmentSize = (
	TotalSegmentSize - sizeof(U32bit) * (3 * NumberOfSegments - 3)
    );

    if (_BufferSize < _CurrentSegmentSize) {
	_BufferSize = 2*_CurrentSegmentSize;
	delete[] _Buffer;
	int AlignedLength = _BufferSize/sizeof(U32bit) + 1;
	_Buffer = new U32bit[AlignedLength];
    }

    _CurrentSegmentIterator = _Buffer;
    _CurrentSegmentIterator += 3;
}

void TKinokoEventSegmentConsolidator::ProcessEventEnd(void)
{
    if (_CurrentSegmentIterator == 0) {
	cerr << "INTERNAL ERROR: ";
	cerr << "TKinokoEventSegmentConsolidator::ProcessEventEnd(): ";
	cerr << "unmatched ProcessEventBegin()/End()" << endl;
	abort();
    }

    TKinokoDataChunk SegmentDataChunk(_Buffer, _CurrentSegmentSize);
    TKinokoEventSegment Segment(SegmentDataChunk);
    Segment.SetHeader(
	_CurrentPieceType, _CurrentEventNumberOfPieces, _CurrentEventKey
    );
    Segment.StreamKey = _StreamKey;

    _OutputProcessor->ProcessEventBegin(
	_CurrentEventKey, _CurrentPieceType, 
	_CurrentEventNumberOfPieces, _CurrentEventTotalSize
    );
    _OutputProcessor->ProcessSegment(Segment);
    _OutputProcessor->ProcessEventEnd();

    _CurrentSegmentIterator = 0;
}

int TKinokoEventSegmentConsolidator::ProcessSegment(const TKinokoEventSegment& EventSegment)
{
    if (_CurrentSegmentIterator == 0) {
	return ProcessSingleSegment(EventSegment);
    }
    if (EventSegment.PieceType() != _CurrentPieceType) {
	return ProcessSingleSegment(EventSegment);
    }

    //... TODO: Check errors here ...//

    TKinokoEventSegment BareSegment(EventSegment);
    BareSegment.CutFrontBy(3 * sizeof(U32bit));

    BareSegment.WriteTo(_CurrentSegmentIterator);
    _CurrentSegmentIterator += BareSegment.Size() / sizeof(U32bit);

    _CurrentEventTotalSize += BareSegment.Size();
    _CurrentEventNumberOfPieces += EventSegment.NumberOfPieces();

    return 1;
}

int TKinokoEventSegmentConsolidator::ProcessSingleSegment(const TKinokoEventSegment& EventSegment)
{
    int NumberOfPieces = EventSegment.NumberOfPieces();
    int TotalSize = (
	EventSegment.Size() - sizeof(U32bit) * (3 + 2 * NumberOfPieces)
    );

    TKinokoEventSegment SingleSegment(EventSegment);
    SingleSegment.StreamKey = _StreamKey;

    _OutputProcessor->ProcessEventBegin(
	SingleSegment.EventKey(), SingleSegment.PieceType(), 
	NumberOfPieces, TotalSize
    );
    _OutputProcessor->ProcessSegment(SingleSegment);
    _OutputProcessor->ProcessEventEnd();

    return 1;
}
