/* KinokoActiveStreamComponent.cc */
/* Created by Enomoto Sanshiro on 9 October 2000. */
/* Last updated by Enomoto Sanshiro on 18 February 2002. */


#include <sstream>
#include <string>
#include <unistd.h>
#include "MushNetworkSocket.hh"
#include "MushDecoratedSocket.hh"
#include "KinokoStream.hh"
#include "KinokoFileStream.hh"
#include "KinokoSocketStream.hh"
#include "KinokoBufferStream.hh"
#include "KinokoNullStream.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataStreamFormatter.hh"
#include "KinokoStreamComponent.hh"
#include "KinokoActiveStreamComponent.hh"

using namespace std;


class TKinokoActiveStreamComponentStreamCommandProcessor: public TKinokoStreamCommandProcessor {
  public:
    TKinokoActiveStreamComponentStreamCommandProcessor(TKinokoActiveStreamComponent* Component);
    virtual ~TKinokoActiveStreamComponentStreamCommandProcessor();
    virtual void OnReceiveFirstRunBeginPacket(void);
    virtual void OnReceiveLastRunEndPacket(void);   
  protected:
    TKinokoActiveStreamComponent* _Component;
};



TKinokoActiveStreamComponentStreamCommandProcessor::TKinokoActiveStreamComponentStreamCommandProcessor(TKinokoActiveStreamComponent* Component)
{
    _Component = Component;
}

TKinokoActiveStreamComponentStreamCommandProcessor::~TKinokoActiveStreamComponentStreamCommandProcessor()
{
}

void TKinokoActiveStreamComponentStreamCommandProcessor::OnReceiveFirstRunBeginPacket(void)
{
    _Component->StartDataProcessing();
}

void TKinokoActiveStreamComponentStreamCommandProcessor::OnReceiveLastRunEndPacket(void)
{
    _Component->StopDataProcessing();
}



TKinokoActiveStreamComponent::TKinokoActiveStreamComponent(const string& TypeName)
: TKinokoStreamComponent(TypeName)
{
    _InputDataStream = new TKinokoInputNullStream();
    _OutputDataStream = new TKinokoOutputNullStream();

    _InputSocket = 0;
    _BufferReader = 0;
    
    _OutputSocket = 0;
    _BufferWriter = 0;

    _IsInputStreamUsed = false;
    _IsOutputStreamUsed = false;
    
    _StreamCommandProcessor = 0;

    _BufferLogger = 0;
    _PortNumberBase = 0;
    _DataSourceIdBase = 0;
}

TKinokoActiveStreamComponent::~TKinokoActiveStreamComponent()
{
    delete _StreamCommandProcessor;

    delete _InputSocket;
    delete _BufferReader;
    
    delete _OutputSocket;
    delete _BufferWriter;

    delete _InputDataStream;
    delete _OutputDataStream;
}

void TKinokoActiveStreamComponent::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamComponent::BuildDescriptor(Descriptor);

    Descriptor.RegisterProperty(
	PropertyId_PortNumber, TKcomPropertyDeclaration(
	    "port_number", TKcomPropertyDeclaration::Type_Int
	)
    );

    TKcomEventDeclaration SetSourceEvent("setSource");
    Descriptor.RegisterEventSlot(EventId_SetSource, SetSourceEvent);

    TKcomEventDeclaration SetSinkEvent("setSink");
    Descriptor.RegisterEventSlot(EventId_SetSink, SetSinkEvent);

    TKcomEventDeclaration SetSourceSinkEvent("setSourceSink");
    Descriptor.RegisterEventSlot(EventId_SetSourceSink, SetSourceSinkEvent);

    TKcomEventDeclaration ConnectEvent("connect");
    Descriptor.RegisterEventSlot(EventId_Connect, ConnectEvent);

    TKcomEventDeclaration WaitConnectedEvent("waitConnected");
    Descriptor.RegisterEventSlot(EventId_WaitConnected, WaitConnectedEvent);

    TKcomEventDeclaration ConstructEvent("construct");
    Descriptor.RegisterEventSlot(EventId_Construct, ConstructEvent);

    TKcomEventDeclaration DestructEvent("destruct");
    Descriptor.RegisterEventSlot(EventId_Destruct, DestructEvent);

    TKcomEventDeclaration DisconnectEvent("disconnect");
    Descriptor.RegisterEventSlot(EventId_Disconnect, DisconnectEvent);

    TKcomEventDeclaration HaltEvent("halt");
    Descriptor.RegisterEventSlot(EventId_Halt, HaltEvent);

    TKcomEventDeclaration QuitEvent("quit");
    Descriptor.RegisterEventSlot(EventId_Quit, QuitEvent);
}

int TKinokoActiveStreamComponent::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetSource:
	Result = ProcessSetSourceEvent(Event);
	break;
	    
      case EventId_SetSink:
	Result = ProcessSetSinkEvent(Event);
	break;
	
      case EventId_SetSourceSink:
	Result = ProcessSetSourceSinkEvent(Event);
	break;

      case EventId_Connect:
	Result = ProcessConnectEvent(Event);
	break;
	
      case EventId_WaitConnected:
	Result = ProcessWaitConnectedEvent(Event);
	break;
	
      case EventId_Construct:
	Result = ProcessConstructEvent(Event);
	break;
	
      case EventId_Destruct:
	Result = ProcessDestructEvent(Event);
	break;
	
      case EventId_Disconnect:
	Result = ProcessDisconnectEvent(Event);
	break;
	    
      case EventId_Halt:
	Result = ProcessHaltEvent(Event);
	break;
	
      case EventId_Quit:
	Result = ProcessQuitEvent(Event);
	break;
	
      default:
	Result = TKinokoStreamComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }
    
    return Result;
}

int TKinokoActiveStreamComponent::DoTransaction(void) throw(TKcomException)
{
    try {
	switch (CurrentState()) {
	  case State_Initial:
	    SetProperties();
	    ChangeStateNow(State_ComponentReady);	
	    break;
	    
	  case State_ComponentReady:
	    // wait for connect() event
	    _TimeKeeper->Suspend();
	    break;
	    
	  case State_Connecting:
	    Connect();
	    ChangeStateNow(State_StreamReady);	
	    break;
 	    
	  case State_StreamReady:
	    // wait for construct()/quit() event
	    _TimeKeeper->Suspend();
	    break;
 	    
	  case State_SystemReady:
	    ProcessDataIfAvailable();
	    break;
	    
          case State_Running:
	    ProcessData();	    
	    break;
	    
	  default:
	    return TKinokoStreamComponent::DoTransaction();
	}
    }
    catch (TKinokoException &e) {
	throw TKcomException(
	    "TKinokoActiveStreamComponent::DoTransaction()",
	    "kinoko exception: " + e.Message()
	);
    }

    return 1;
}

int TKinokoActiveStreamComponent::ProcessSetSourceEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setSource(): too few argument[s]"
	);
	return 0;
    }

    string SourceComponentName = Event.ArgumentList()[0];
    _Logger->WriteDebug(ComponentName(), "setSource(): " + SourceComponentName);

    SetSource(SourceComponentName);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessSetSinkEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setSink(): too few argument[s]"
	);
	return 0;
    }

    string SinkComponentName = Event.ArgumentList()[0];
    _Logger->WriteDebug(ComponentName(), "setSink(): " + SinkComponentName);

    SetSink(SinkComponentName);
	
    return 1;
}

int TKinokoActiveStreamComponent::ProcessSetSourceSinkEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 2) {
	_Logger->WriteError(
	    ComponentName(), "setSourceSink(): too few arguments"
	);
	return 0;
    }

    string SourceComponentName = Event.ArgumentList()[0];
    string SinkComponentName = Event.ArgumentList()[1];
    _Logger->WriteDebug(
	ComponentName(), 
	"setSourceSink(): " + SourceComponentName + ", " + SinkComponentName
    );

    SetSource(SourceComponentName);
    SetSink(SinkComponentName);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessConnectEvent(TKcomEvent& Event)
{
    if (CurrentState() != State_ComponentReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to connect: ignored"
	);
	return 1;
    }

    ChangeState(State_Connecting);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessWaitConnectedEvent(TKcomEvent& Event)
{
    // This is an event called after connect().
    // Since there is at least one DoTransaction() between two evnets,
    // connection must be established when this event is called.
    // By collecting all waitConnected() responses in KCOM,
    // KCOM can be sure that everything is connected and be able to proceed.

    if (CurrentState() != State_StreamReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to waitConnected: ignored"
	);
	return 1;
    }

    return 1;
}

int TKinokoActiveStreamComponent::ProcessConstructEvent(TKcomEvent& Event)
{
    if (CurrentState() != State_StreamReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to construct: ignored"
	);
	return 1;
    }

    _StreamCommandProcessor = (
	new TKinokoActiveStreamComponentStreamCommandProcessor(this)
    );

    Construct();

    ChangeState(State_SystemReady);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessDestructEvent(TKcomEvent& Event)
{
    if (CurrentState() == State_Running) {
	_Logger->WriteWarning(
	    ComponentName(), "destruct while running: stop forced"
	);
	StopDataProcessing();
    }
    else if (CurrentState() == State_StreamReady) {
	_Logger->WriteWarning(
	    ComponentName(), "already destructed: nothing done"
	);
	return 1;
    }
    else if (CurrentState() != State_SystemReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to destruct: ignored"
	);
	return 1;
    }

    Destruct();
    delete _StreamCommandProcessor;
    _StreamCommandProcessor = 0;

    ChangeState(State_StreamReady);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessHaltEvent(TKcomEvent& Event)
{
    _Logger->WriteDebug(ComponentName(), "halt()");

    StopDataProcessing();

    return 1;
}

int TKinokoActiveStreamComponent::ProcessDisconnectEvent(TKcomEvent& Event)
{
    if (CurrentState() == State_SystemReady) {
	_Logger->WriteWarning(
	    ComponentName(), "disconnect before destruct: destruct forced"
	);
	Destruct();
	delete _StreamCommandProcessor;
	_StreamCommandProcessor = 0;
    }
    else if (CurrentState() == State_ComponentReady) {
	_Logger->WriteWarning(
	    ComponentName(), "already disconnected: nothing done"
	);
	return 1;
    }
    else if (CurrentState() != State_StreamReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to disconnect: ignored"
	);
	return 1;
    }

    Disconnect();

    ChangeState(State_ComponentReady);

    return 1;
}

int TKinokoActiveStreamComponent::ProcessQuitEvent(TKcomEvent& Event)
{
    if (CurrentState() == State_StreamReady) {
	_Logger->WriteWarning(
	    ComponentName(), "quit before disconnect: disconnect forced"
	);
	Disconnect();
    }
    else if (CurrentState() == State_Shutdown) {
	_Logger->WriteWarning(
	    ComponentName(), "already quitted: nothing done"
	);
	return 1;
    }
    else if (CurrentState() != State_ComponentReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to quit"
	);
    }

    // write all log/status messages before sending event reply //
    // KCOM collects all quit replies and delete logger and exit //    
    ChangeStateNow(State_Shutdown);

    return 1;
}

void TKinokoActiveStreamComponent::SetSink(const string& SinkComponentName)
{
    _SinkComponentName = SinkComponentName;

    string SinkComponentType;
    GetPropertyOf(_SinkComponentName, "stream_type", SinkComponentType);
    if (SinkComponentType.empty()) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setSink(): unable to find sink component: " + 
	    SinkComponentName
	);
	return;
    }
    
    GetPropertyOf(_SinkComponentName, "host", _SinkHostName);

    ostringstream StreamChannelStream;
    if (
	(SinkComponentType == _StreamTypeName[StreamType_Sink]) ||
	(SinkComponentType == _StreamTypeName[StreamType_Pipe]) ||
	(SinkComponentType == _StreamTypeName[StreamType_Source])
    ){
	_SinkStreamType = StreamType_Sink;
	StreamChannelStream << "port (client)";
    }
    else if (SinkComponentType == _StreamTypeName[StreamType_Buffer]) {
	_SinkStreamType = StreamType_Buffer;
	GetPropertyOf(
	    SinkComponentName, "ipc_project_path", _OutputBufferIpcProjectPath
	);
	GetLongPropertyOf(
	    SinkComponentName, "shared_memory_id", _OutputSharedMemoryId
	);
	GetLongPropertyOf(
	    SinkComponentName, "message_queue_id", _OutputMessageQueueId
	);

	StreamChannelStream << 
	    "shm: " << _OutputSharedMemoryId << ", " <<
	    "msg: " << _OutputMessageQueueId << " " <<
	    "(@" << _OutputBufferIpcProjectPath << ")";
    }
    else {
	_Logger->WriteError(
	    ComponentName(),
	    "setSink(): unknown component type: " + SinkComponentType
	);
	return;
    }

    _IsOutputStreamUsed = true;
    
    _Logger->WriteDebug(
	ComponentName(),
	"setSink(): " + SinkComponentName + 
	" (" + SinkComponentType + ")" +
	" @" + _SinkHostName + ", " + StreamChannelStream.str()
    );
}

void TKinokoActiveStreamComponent::SetSource(const string& SourceComponentName)
{
    _SourceComponentName = SourceComponentName;

    string SourceComponentType;
    GetPropertyOf(SourceComponentName, "stream_type", SourceComponentType);
    if (SourceComponentType.empty()) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setSource(): unable to find source component: " + 
	    SourceComponentName
	);
	return;
    }

    GetPropertyOf(SourceComponentName, "host", _SourceHostName);

    ostringstream StreamChannelStream;
    if (
	(SourceComponentType == _StreamTypeName[StreamType_Source]) ||
	(SourceComponentType == _StreamTypeName[StreamType_Pipe]) ||
	(SourceComponentType == _StreamTypeName[StreamType_Sink]) 
    ){
	_SourceStreamType = StreamType_Source;

	static const int MaxNumberOfTries = 100;
	int NumberOfTries = 0;

	int InputPortNumber;
	TMushServerNetworkSocket* RawSocket = 0;
	while (RawSocket == 0) {
	    InputPortNumber = AllocatePortNumber();
	    RawSocket = new TMushServerNetworkSocket(InputPortNumber);
	    try {
		RawSocket->Bind();
	    }
	    catch (TSystemCallException& e) {
		delete RawSocket;
		RawSocket = 0;

		if (++NumberOfTries > MaxNumberOfTries) {
		    _Logger->WriteError(
			ComponentName(), "setSource(): " + e.Message()
		    );
		    return;
		}

		_Logger->WriteWarning(
		    ComponentName(), 
		    "setSource(): " + e.Message() + " (trying again...)"
		);
	    }
	}
	RawSocket->Listen();

	_InputSocket = new TMushFramedSocket(RawSocket);
	SetLongProperty(PropertyId_PortNumber, InputPortNumber);

	StreamChannelStream << "port: " << InputPortNumber;
    }
    else if (SourceComponentType == _StreamTypeName[StreamType_Buffer]) {
	_SourceStreamType = StreamType_Buffer;
	GetPropertyOf(
	    SourceComponentName, "ipc_project_path", _InputBufferIpcProjectPath
	);
	GetLongPropertyOf(
	    SourceComponentName, "shared_memory_id", _InputSharedMemoryId
	);
	GetLongPropertyOf(
	    SourceComponentName, "message_queue_id", _InputMessageQueueId
	);

	StreamChannelStream << 
	    "shm: " << _InputSharedMemoryId << ", " <<
	    "msg: " << _InputMessageQueueId << " " <<
	    "(@" << _InputBufferIpcProjectPath << ")";
    }
    else {
	_Logger->WriteError(
	    ComponentName(),
	    "setSource(): unknown component type: " + SourceComponentType
	);
	return;
    }

    _IsInputStreamUsed = true;

    _Logger->WriteDebug(
	ComponentName(),
	"setSource(): " + SourceComponentName + 
	" (" + SourceComponentType + ")"
	" @" + _SourceHostName + ", " + StreamChannelStream.str()
    );
}

void TKinokoActiveStreamComponent::Connect(void) throw(TKinokoException)
{
    if (_IsInputStreamUsed) {
	ConnectSource();
    }
    if (_IsOutputStreamUsed) {
	ConnectSink();
    }
}

void TKinokoActiveStreamComponent::Disconnect(void) throw(TKinokoException)
{
    delete _InputSocket;
    delete _BufferReader;
    delete _OutputSocket;
    delete _BufferWriter;
    delete _InputDataStream;
    delete _OutputDataStream;

    _InputSocket = 0;
    _BufferReader = 0;
    _OutputSocket = 0;
    _BufferWriter = 0;
    _InputDataStream = 0;
    _OutputDataStream = 0;
}

void TKinokoActiveStreamComponent::ConnectSink(void) throw(TKinokoException)
{
    _Logger->WriteDebug(ComponentName(), "connecting sink...");

    if (_SinkComponentName.empty()) {
	_Logger->WriteError(
	    ComponentName(), "connect(): sink component is not specified"
	);
	return;
    }

    long OutputPortNumber;
    GetLongPropertyOf(_SinkComponentName, "port_number", OutputPortNumber);

    TKinokoOutputStream* NewOutputDataStream = 0;
    if (_SinkStreamType == StreamType_Sink) {
	TMushSocket* RawSocket = 0;
	try {
	    RawSocket = new TMushClientNetworkSocket(
		_SinkHostName, OutputPortNumber
	    );
	    RawSocket->Open();
	}
	catch (TSystemCallException &e) {
	    delete RawSocket;
	    _Logger->WriteError(
		ComponentName(), "connect(): " + e.Message()
	    );
	    return;
	}

	_OutputSocket = new TMushFramedSocket(RawSocket);
	NewOutputDataStream = new TKinokoOutputSocketStream(_OutputSocket);
    }
    else if (_SinkStreamType == StreamType_Buffer) {
	if (_BufferLogger == 0) {
	    _BufferLogger = new TKinokoLoggerBufferLogger(
		ComponentName(), _Logger
	    );
	}

	try {
	    _BufferWriter = new TKinokoBufferWriter(
		_OutputSharedMemoryId, _OutputMessageQueueId, 
		_OutputBufferIpcProjectPath,
		_TimeKeeper, _BufferLogger
	    );
	}
	catch (TKinokoException &e) {
	    delete _BufferWriter;
	    _BufferWriter = 0;
	    _Logger->WriteError(
		ComponentName(), "connect(): " + e.Message()
	    );
	    return;
	}

	NewOutputDataStream = new TKinokoOutputBufferStream(_BufferWriter);
    }
    else {
	_Logger->WriteError(
	    ComponentName(), "unknown component type (internal)"
	);
	return;
    }

    if (NewOutputDataStream != 0) {
	delete _OutputDataStream;
	_OutputDataStream = NewOutputDataStream;
	_Logger->WriteDebug(ComponentName(), "sink connected");
    }
}

void TKinokoActiveStreamComponent::ConnectSource(void) throw(TKinokoException)
{
    _Logger->WriteDebug(ComponentName(), "connecting source...");

    if (_SourceComponentName.empty()) {
	_Logger->WriteError(
	    ComponentName(), "connect(): source component is not specified."
	);
	return;
    }

    TKinokoInputStream* NewInputDataStream = 0;
    if (_SourceStreamType == StreamType_Source) {
	try {
	    _InputSocket->Open();
	}
	catch (TSystemCallException &e) {
	    _Logger->WriteError(
		ComponentName(), "connect(): " + e.Message()
	    );
	    return;
	}

	NewInputDataStream = new TKinokoInputSocketStream(_InputSocket);
    }
    else if (_SourceStreamType == StreamType_Buffer) {
	if (_BufferLogger == 0) {
	    _BufferLogger = new TKinokoLoggerBufferLogger(
		ComponentName(), _Logger
	    );
	}

	try {
	    _BufferReader = new TKinokoBufferReader(
		_InputSharedMemoryId, _InputMessageQueueId, 
		_InputBufferIpcProjectPath,
		_TimeKeeper, _BufferLogger
	    );
	}
	catch (TKinokoException &e) {
	    delete _BufferReader;
	    _BufferReader = 0;
	    _Logger->WriteError(
		ComponentName(), "connect(): " + e.Message()
	    );
	    return;
	}

	NewInputDataStream = new TKinokoInputBufferStream(_BufferReader);
    }
    else {
	_Logger->WriteError(
	    ComponentName(), "unknown component type (internal)"
	);
	return;
    }

    if (NewInputDataStream != 0) {
	delete _InputDataStream;
	_InputDataStream = NewInputDataStream;
	_Logger->WriteDebug(ComponentName(), "source connected.");
    }
}

void TKinokoActiveStreamComponent::StartDataProcessing(void) throw(TKinokoException)
{
    if (CurrentState() != State_SystemReady) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to start: ignored"
	);
	return;
    }
    
    OnStart();
    ChangeState(State_Running);
}

void TKinokoActiveStreamComponent::StopDataProcessing(void) throw(TKinokoException)
{
    if (CurrentState() != State_Running) {
	_Logger->WriteError(
	    ComponentName(), "bad component state to stop: ignored"
	);
	return;
    }

    OnStop();
    ChangeState(State_SystemReady);
}

int TKinokoActiveStreamComponent::ProcessDataIfAvailable(void) throw(TKinokoException)
{
    if (_IsInputStreamUsed && _InputDataStream->HasData()) {
	ProcessData();
    }
    else {
	_TimeKeeper->Suspend();
    }

    return 1;
}

int TKinokoActiveStreamComponent::AllocatePortNumber(void)
{
    if (_PortNumberBase == 0) {
	try {
	    _PortNumberBase = TMushEnvironmentVariable("KINOKO_DATASTREAM_PORT_BASE").AsLong();
	}
	catch (TSystemCallException &e) {
	    static const int DefaultPortNumberBase = 42000;
	    _PortNumberBase = DefaultPortNumberBase;
	}
    }

    int PortNumberOffset = Registry()->GetSequenceValue("stream_port");
    int PortNumber = _PortNumberBase + PortNumberOffset;

    return PortNumber;
}

int TKinokoActiveStreamComponent::AllocateDataSourceId(void)
{
    if (_DataSourceIdBase == 0) {
	_DataSourceIdBase = TKinokoDataSource::AutomaticDataSourceIdAssignmentBase();
    }

    int DataSourceIdOffset = Registry()->GetSequenceValue("data_source_id");
    int DataSourceId = _DataSourceIdBase + DataSourceIdOffset;

    return DataSourceId;
}
