/* KinokoBuilderCom.cc */
/* Created by Enomoto Sanshiro on 8 September 2009. */
/* Last updated by Enomoto Sanshiro on 7 July 2010. */


#include "KinokoStreamPipeComponent.hh"
#include "KinokoHorizontalBuilder.hh"
#include "KinokoVerticalBuilder.hh"
#include "KinokoBuilderConfiguration.hh"
#include "KiscFancyConfigFile.hh"
#include "KinokoBuilderLogger.hh"
#include "KinokoBuilderMonitor.hh"
#include "KinokoBuilderCom.hh"

using namespace std;


static const int DefaultReportInterval = 180;
static const int ReportTimeCheckPrescaleRate = 100;


TKinokoBuilderCom::TKinokoBuilderCom(const string& TypeName)
: TKinokoStreamPipeComponent(TypeName)
{
    _Builder = 0;
    _BuilderLogger = 0;
    _BuilderMonitor = 0;

    _ReportInterval = DefaultReportInterval;
    _LastReportTime = TMushDateTime::SecSinceEpoch();
    _NumberOfTransactionsSinceLastReport = 0;
}

TKinokoBuilderCom::~TKinokoBuilderCom()
{
    delete _BuilderMonitor;
    delete _BuilderLogger;
    delete _Builder;
}

void TKinokoBuilderCom::SetConfigFile(const string& ConfigFileName, const string& BuilderName)
{
    _ConfigFileName = ConfigFileName;
    _BuilderName = BuilderName;
}

void TKinokoBuilderCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamPipeComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment("Builder: Event Builder Component");

    TKcomEventDeclaration SetConfigEvent("setConfig");
    SetConfigEvent.AddArgument(TKcomPropertyDeclaration(
        "config_file", TKcomPropertyDeclaration::Type_String
    ));
    SetConfigEvent.AddArgument(TKcomPropertyDeclaration(
        "builder_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SetConfig, SetConfigEvent);

    TKcomEventDeclaration SetReportIntervalEvent("setReportInterval");
    SetReportIntervalEvent.AddArgument(TKcomPropertyDeclaration(
        "interval_sec", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetReportInterval, SetReportIntervalEvent);
}

int TKinokoBuilderCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetConfig:
	Result = ProcessSetConfigEvent(Event);
	break;

      case EventId_SetReportInterval:
	Result = ProcessSetReportIntervalEvent(Event);
	break;

      default:
	Result = TKinokoStreamPipeComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoBuilderCom::ProcessSetConfigEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 2) {
	_Logger->WriteError(
	    ComponentName(), "setConfigFile(): too few arguments"
        );
	return 0;
    }

    string ConfigFileName = Event.ArgumentList()[0];
    string BuilderName = Event.ArgumentList()[1];
    SetConfigFile(ConfigFileName, BuilderName);

    _Logger->WriteDebug(
	ComponentName(), 
	"setConfigFile(): " + ConfigFileName + ", " + BuilderName
    );

    return 1;
}

int TKinokoBuilderCom::ProcessSetReportIntervalEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setReportInterval(): too few arguments"
        );
	return 0;
    }
    string ReportIntervalString = Event.ArgumentList()[0];

    istringstream ReportIntervalStream(ReportIntervalString);
    if (! (ReportIntervalStream >> _ReportInterval)) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setReportInterval(): invalid argument: " + ReportIntervalString
	);
	return 0;
    }

    _Logger->WriteDebug(
	ComponentName(), "setReportInterval(): " + ReportIntervalString
    );

    return 1;
}

void TKinokoBuilderCom::Construct(void) throw(TKinokoException)
{
    try {
	_Builder = this->CreateBuilder();
    }
    catch (TKinokoException& e) {
	delete _Builder; _Builder = 0; 
	throw TKinokoException(
	    "Builder configuration fault: " + e.Message()
	);
    }

    _BuilderLogger = new TKinokoLoggerBuilderLogger(
	ComponentName(), _Logger
    );
    _BuilderMonitor = new TKinokoBuilderLoggingMonitor(_BuilderLogger);
    _Builder->SetBuilderLogger(_BuilderLogger);
    _Builder->SetBuilderMonitor(_BuilderMonitor);

    try {
	TKiscFancyConfigFile ConfigFile(_ConfigFileName);
	ConfigFile.SetEntryKeyword("builder");
	TKinokoKiscBuilderConfiguration BuilderConfig(ConfigFile);
	BuilderConfig.ConfigureBuilder(_Builder, _BuilderName);
    }
    catch (TKinokoException& e) {
	delete _BuilderMonitor; _BuilderMonitor = 0;
	delete _BuilderLogger; _BuilderLogger = 0;
	delete _Builder; _Builder = 0; 
	throw TKinokoException(
	    "Builder configuration fault: " + e.Message()
	);
    }
    catch (TScriptException& e) {
	delete _BuilderMonitor; _BuilderMonitor = 0;
	delete _BuilderLogger; _BuilderLogger = 0;
	delete _Builder; _Builder = 0; 
	throw TKinokoException(
	    "Builder configuration fault: " + e.Message()
	);
    }

    // Data Processor Interface //
    _Builder->AttachPlatform(
	InputStream(), OutputStream(), _EventEmitter, _Registry, _Logger
    );
    _Builder->Construct(
	DataSourceName(), _InputDataStream, _OutputDataStream, DataSourceId()
    );
    _Builder->SetStreamCommandProcessor(_StreamCommandProcessor);
}

void TKinokoBuilderCom::Destruct(void) throw(TKinokoException)
{
    _Builder->Destruct();

    delete _BuilderMonitor;
    delete _BuilderLogger;
    delete _Builder;

    _Builder = 0;
    _BuilderLogger = 0;
    _BuilderMonitor = 0;
}

int TKinokoBuilderCom::ProcessData(void) throw(TKinokoException)
{
    if (_NumberOfTransactionsSinceLastReport++ > ReportTimeCheckPrescaleRate) {
	if (TMushDateTime::SecSince(_LastReportTime) >= _ReportInterval) {
	    _LastReportTime = TMushDateTime::SecSinceEpoch();
	    _NumberOfTransactionsSinceLastReport = 0;

	    _BuilderMonitor->Report();
	}
    }

    int Result = _Builder->ProcessData();
    if (Result <= 0) {
        _TimeKeeper->Suspend();
    }

    return Result;
}



TKinokoHorizontalBuilderCom::TKinokoHorizontalBuilderCom(void)
: TKinokoBuilderCom("KinokoHorizontalBuilderCom")
{
    _Scanner = 0;
}

TKinokoHorizontalBuilderCom::~TKinokoHorizontalBuilderCom()
{
}

void TKinokoHorizontalBuilderCom::SetPieceScanner(TKinokoEventPieceScanner* Scanner)
{
    _Scanner = Scanner;
}

TKinokoBuilder* TKinokoHorizontalBuilderCom::CreateBuilder(void) throw(TKinokoException)
{
    if (_Scanner == 0) {
	throw TKinokoException(
	    "TKinokoHorizontalBuilder: Event Piece Scanner not set"
	);
    }

    TKinokoHorizontalBuilder* Builder = new TKinokoHorizontalBuilder();
    Builder->SetPieceScanner(_Scanner);

    return Builder;
}




TKinokoVerticalBuilderCom::TKinokoVerticalBuilderCom(void)
: TKinokoBuilderCom("KinokoVerticalBuilderCom")
{
}

TKinokoVerticalBuilderCom::~TKinokoVerticalBuilderCom()
{
}

TKinokoBuilder* TKinokoVerticalBuilderCom::CreateBuilder(void) throw(TKinokoException)
{
    return new TKinokoVerticalBuilder();
}



TKinokoLoggerBuilderLogger::TKinokoLoggerBuilderLogger(const string& ComponentName, TKinokoLogger* Logger)
{
    _Name = ComponentName;
    _Logger = Logger;
}

TKinokoLoggerBuilderLogger::~TKinokoLoggerBuilderLogger()
{
}

int TKinokoLoggerBuilderLogger::Write(const string& Message, int MessageLevel)
{
    switch (MessageLevel) {
      case TKinokoBuilderLogger::mlInformation:
	_Logger->WriteDebug(_Name, Message); 
	break;
      case TKinokoBuilderLogger::mlLog:
	_Logger->WriteNotice(_Name, Message); 
	break;
      case TKinokoBuilderLogger::mlWarning:
	_Logger->WriteWarning(_Name, Message); 
	break;
      case TKinokoBuilderLogger::mlError:
	_Logger->WriteError(_Name, Message); 
	break;
      case TKinokoBuilderLogger::mlPanic:
	_Logger->WritePanic(_Name, Message); 
	break;
      default:
	throw TKinokoException(
	    "TKinokoLoggerBuilderLogger::Write()", 
	    "Bad log message level (internal)"
	);
    }   

    return 1;
}
