/* KinokoLoggerCom.cc */
/* Created by Enomoto Sanshiro on 11 October 2000. */
/* Last updated by Enomoto Sanshiro on 11 October 2000. */


#include <string>
#include "MushFileSystem.hh"
#include "MushNetworkSocket.hh"
#include "MushMisc.hh"
#include "KinokoLogger.hh"
#include "KinokoLoggerOrb.hh"
#include "KinokoLoggerCom.hh"

using namespace std;


static const string Comment = "Logger: Logbook Writer";


class TKinokoComponentLogMessageHandler: public TKinokoLogMessageHandler {
  public:
    TKinokoComponentLogMessageHandler(TKinokoLoggerCom* LoggerCom);
    virtual ~TKinokoComponentLogMessageHandler();
  protected:
    virtual void OnRemarkable(const string& Message);
    virtual void OnWarning(const string& Message);
    virtual void OnError(const string& Message);
    virtual void OnPanic(const string& Message);
  private:
    TKinokoLoggerCom* _LoggerCom;
};



TKinokoComponentLogMessageHandler::TKinokoComponentLogMessageHandler(TKinokoLoggerCom* LoggerCom)
{
    _LoggerCom = LoggerCom;
}

TKinokoComponentLogMessageHandler::~TKinokoComponentLogMessageHandler()
{
}

void TKinokoComponentLogMessageHandler::OnRemarkable(const string& Message)
{
    _LoggerCom->OnRemarkable(Message);
}

void TKinokoComponentLogMessageHandler::OnWarning(const string& Message)
{
    _LoggerCom->OnWarning(Message);
}

void TKinokoComponentLogMessageHandler::OnError(const string& Message)
{
    _LoggerCom->OnError(Message);
}

void TKinokoComponentLogMessageHandler::OnPanic(const string& Message)
{
    _LoggerCom->OnPanic(Message);
}



TKinokoLoggerCom::TKinokoLoggerCom(void)
: TKinokoSystemComponent("KinokoLoggerCom")
{
    _Logger = 0;
    _LogFile = 0;
    _LogMessageHandler = 0;
    _IsRunning = false;
}

TKinokoLoggerCom::~TKinokoLoggerCom()
{
    delete _LogMessageHandler;
    delete _LogFile;
    delete _Logger;
}

void TKinokoLoggerCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoSystemComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    _Logger = new TKinokoKoapLogger(&OutputStream());
    _ObjectMessenger = new TKinokoLoggerMessenger(_Logger);

    _LogMessageHandler = new TKinokoComponentLogMessageHandler(this);
    _Logger->AddMessageHandler(_LogMessageHandler);

    Descriptor.RegisterExportObject(
	_ObjectMessenger, TKcomObjectDeclaration(
	    _ObjectMessenger->ClassName(), "logger"
	)
    );

    TKcomEventDeclaration StartLoggingEvent("startLogging");
    StartLoggingEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_StartLogging, StartLoggingEvent);

    TKcomEventDeclaration RestartLoggingEvent("restartLogging");
    RestartLoggingEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_RestartLogging, RestartLoggingEvent);

    TKcomEventDeclaration ClearStatusEvent("clearStatus");
    ClearStatusEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_ClearStatus, ClearStatusEvent);

    TKcomEventDeclaration WriteDebugEvent("writeDebug");
    WriteDebugEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WriteDebug, WriteDebugEvent);

    TKcomEventDeclaration WriteNoticeEvent("writeNotice");
    WriteNoticeEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WriteNotice, WriteNoticeEvent);

    TKcomEventDeclaration WriteRemarkableEvent("writeRemarkable");
    WriteRemarkableEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WriteRemarkable, WriteRemarkableEvent);

    TKcomEventDeclaration WriteWarningEvent("writeWarning");
    WriteWarningEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WriteWarning, WriteWarningEvent);

    TKcomEventDeclaration WriteErrorEvent("writeError");
    WriteErrorEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WriteError, WriteErrorEvent);

    TKcomEventDeclaration WritePanicEvent("writePanic");
    WritePanicEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_WritePanic, WritePanicEvent);

    TKcomEventDeclaration ProcessRemarkableEvent("processRemarkable");
    ProcessRemarkableEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSource(EventId_ProcessRemarkable, ProcessRemarkableEvent);

    TKcomEventDeclaration ProcessWarningEvent("processWarning");
    ProcessWarningEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSource(EventId_ProcessWarning, ProcessWarningEvent);

    TKcomEventDeclaration ProcessErrorEvent("processError");
    ProcessErrorEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSource(EventId_ProcessError, ProcessErrorEvent);

    TKcomEventDeclaration ProcessPanicEvent("processPanic");
    ProcessPanicEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSource(EventId_ProcessPanic, ProcessPanicEvent);
}

void TKinokoLoggerCom::Initialize(void) throw(TKcomException)
{
    TKinokoSystemComponent::Initialize();

    OutputStream() << ".start;" << endl;
    OutputStream() << ".set title " << ComponentName() << ";" << endl;

    _IsRunning = true;
}

void TKinokoLoggerCom::Finalize(void) throw(TKcomException)
{
    TKinokoSystemComponent::Finalize();

    OutputStream() << ".stop;" << endl;
    OutputStream() << ".quit;" << endl;

    _IsRunning = false;
}

int TKinokoLoggerCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_StartLogging:
	Result = ProcessStartLoggingEvent(Event);
	break;

      case EventId_RestartLogging:
	Result = ProcessRestartLoggingEvent(Event);
	break;

      case EventId_ClearStatus:
	Result = ProcessClearStatusEvent(Event);
	break;

      case EventId_Quit:
	ProcessQuitEvent(Event);
	Result = TKinokoSystemComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
	break;

      case EventId_WriteDebug:
	Result = ProcessWriteDebugEvent(Event);
	break;

      case EventId_WriteNotice:
	Result = ProcessWriteNoticeEvent(Event);
	break;

      case EventId_WriteRemarkable:
	Result = ProcessWriteRemarkableEvent(Event);
	break;

      case EventId_WriteWarning:
	Result = ProcessWriteWarningEvent(Event);
	break;

      case EventId_WriteError:
	Result = ProcessWriteErrorEvent(Event);
	break;

      case EventId_WritePanic:
	Result = ProcessWritePanicEvent(Event);
	break;

      default:
	Result = TKinokoSystemComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoLoggerCom::ProcessStartLoggingEvent(TKcomEvent& Event)
{
    string LogFileName;
    if (Event.ArgumentList().size() < 1) {
	LogFileName = "kinoko-log.xml";
    }
    else {
	LogFileName = Event.ArgumentList()[0];
    }

    _LogFile = new TKinokoXmlLogFile();
    _LogFile->Open(LogFileName);

    _Logger->SetLogFile(_LogFile);

    string Host = TMushNetworkSocket::LocalIPAddress();
    string Path = TMushFileAttribute(LogFileName).PathName();
    string FileName = TMushFileAttribute(LogFileName).FileName();
    if (Path.empty()) {
	Path = TMushFileSystem::CurrentDirectory() + "/";
    }
    Registry()->SetValue(
	"/files/log/" + ComponentName() + "/host", Host
    );
    Registry()->SetValue(
	"/files/log/" + ComponentName() + "/path", Path
    );
    Registry()->SetValue(
	"/files/log/" + ComponentName() + "/filename", FileName
    );

    return 1;
}

int TKinokoLoggerCom::ProcessRestartLoggingEvent(TKcomEvent& Event)
{
    if (_LogFile == 0) {
	return ProcessStartLoggingEvent(Event);
    }

    if (Event.ArgumentList().size() > 0) {
	string NewLogFileName = Event.ArgumentList()[0];
	TKinokoLogFile* NewLogFile = new TKinokoXmlLogFile();
	NewLogFile->Open(NewLogFileName);

	_Logger->SetLogFile(NewLogFile);
	delete _LogFile;
	_LogFile = NewLogFile;
    }

    _Logger->Refresh();

    return 1;
}

int TKinokoLoggerCom::ProcessClearStatusEvent(TKcomEvent& Event)
{
    _Logger->ClearStatus();

    return 1;
}

int TKinokoLoggerCom::ProcessQuitEvent(TKcomEvent& Event)
{
    if (_LogFile) {
	_LogFile->Close();
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWriteDebugEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WriteDebug(Event.SourceName(), Message);
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWriteNoticeEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WriteNotice(Event.SourceName(), Message);
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWriteRemarkableEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WriteRemarkable(Event.SourceName(), Message);
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWriteWarningEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WriteWarning(Event.SourceName(), Message);
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWriteErrorEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WriteError(Event.SourceName(), Message);
    }

    return 1;
}

int TKinokoLoggerCom::ProcessWritePanicEvent(TKcomEvent& Event)
{
    string Message;
    if (Event.ArgumentList().size() > 0) {
	Message = Event.ArgumentList()[0];
    }

    if (_IsRunning) {
	_Logger->WritePanic(Event.SourceName(), Message);
    }

    return 1;
}

void TKinokoLoggerCom::OnRemarkable(const std::string& Message)
{
    if (! _IsRunning) {
	return;
    }

    TKcomEvent Event;
    Event.ArgumentList().push_back(Message);

    EmitEventOneWay(EventId_ProcessRemarkable, Event);
}

void TKinokoLoggerCom::OnWarning(const std::string& Message)
{
    if (! _IsRunning) {
	return;
    }

    TKcomEvent Event;
    Event.ArgumentList().push_back(Message);

    EmitEventOneWay(EventId_ProcessWarning, Event);
}

void TKinokoLoggerCom::OnError(const std::string& Message)
{
    if (! _IsRunning) {
	return;
    }

    TKcomEvent Event;
    Event.ArgumentList().push_back(Message);

    EmitEventOneWay(EventId_ProcessError, Event);
}

void TKinokoLoggerCom::OnPanic(const std::string& Message)
{
    if (! _IsRunning) {
	return;
    }

    TKcomEvent Event;
    Event.ArgumentList().push_back(Message);

    EmitEventOneWay(EventId_ProcessPanic, Event);
}

