/* KinokoReproducerCom.cc */
/* Created by Enomoto Sanshiro on 24 December 2001. */
/* Last updated by Enomoto Sanshiro on 24 December 2001. */


#include <iostream>
#include <sstream>
#include <string>
#include "KinokoStream.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"
#include "KinokoReproducerCom.hh"

using namespace std;


static const string Comment = "Reproducer: reproduces data stream from data files";


TKinokoReproducerCom::TKinokoReproducerCom(void)
: TKinokoStreamSourceComponent("KinokoReproducerCom")
{
    _Storage = 0;
    _DataReceiver = 0;
    _DataStreamScanner = 0;
    _DelayTimer = 0;

    _IntervalSec = 0;
    _IntervalUSec = 0;

    _DataSourceId = TKinokoDataSource::DataSourceId_Any;
    _NumberOfDataSources = 0;
}

TKinokoReproducerCom::~TKinokoReproducerCom()
{
    delete _DataStreamScanner;
    delete _DataReceiver;
    delete _Storage;
    delete _DelayTimer;
}

void TKinokoReproducerCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamSourceComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration SetInputDataEvent("setInputData");
    SetInputDataEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    SetInputDataEvent.AddArgument(TKcomPropertyDeclaration(
        "datasource_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SetInputData, SetInputDataEvent);

    TKcomEventDeclaration SetEventIntervalEvent("setEventInterval");
    SetEventIntervalEvent.AddArgument(TKcomPropertyDeclaration(
        "interval_sec", TKcomPropertyDeclaration::Type_Int
    ));
    SetEventIntervalEvent.AddArgument(TKcomPropertyDeclaration(
        "interval_usec", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetEventInterval, SetEventIntervalEvent);

    TKcomEventDeclaration RunEndEvent("runEnd");
    Descriptor.RegisterEventSource(EventId_RunEnd, RunEndEvent);
}

int TKinokoReproducerCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetInputData:
	Result = ProcessSetInputDataEvent(Event);
	break;

      case EventId_SetEventInterval:
	Result = ProcessSetEventIntervalEvent(Event);
	break;

      default:
	Result = TKinokoStreamSourceComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoReproducerCom::ProcessSetInputDataEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setInputData(): too few arguments"
        );
	return 0;
    }

    _DataFileName = Event.ArgumentList()[0];
    if (Event.ArgumentList().size() > 1) {
	_DataSourceName = Event.ArgumentList()[1];
	if (! _DataSourceName.empty()) {
	    _DataSourceId = TKinokoDataSource::DataSourceId_Unknown;
	}
    }
    
    _Logger->WriteDebug(
	ComponentName(), 
	"setInputData(): " + _DataFileName + ", " + _DataSourceName
    );

    return 1;
}

int TKinokoReproducerCom::ProcessSetEventIntervalEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setInterval(): too few arguments"
        );
	return 0;
    }

    string IntervalSecString, IntervalUSecString;

    IntervalSecString = Event.ArgumentList()[0];
    istringstream IntervalSecStream(IntervalSecString);
    if (! (IntervalSecStream >> _IntervalSec)) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setInterval(): invalid argument: " + IntervalSecString
        );
	return 0;
    }
    
    if (Event.ArgumentList().size() > 1) {
	IntervalUSecString = Event.ArgumentList()[1];
	istringstream IntervalUSecStream(IntervalUSecString);
	if (! (IntervalUSecStream >> _IntervalUSec)) {
	    _Logger->WriteError(
		ComponentName(), 
		"setInterval(): invalid argument: " + IntervalUSecString
	    );
	    return 0;
	}
    }
    
    _Logger->WriteDebug(
	ComponentName(), "setInterval(): " + 
	IntervalSecString + ", " + IntervalUSecString
    );

    return 1;
}

void TKinokoReproducerCom::Construct(void) throw(TKinokoException)
{
    _Storage = new TKinokoKdfStorage(_DataFileName);

    TKinokoStorageHeader StorageHeader;
    try {
        _Storage->ReadHeader(StorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch ? */
	_Logger->WriteError(ComponentName(), e.Message());
    }

    try {
	_InputDataStream = _Storage->GetInputStream();
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(
	    ComponentName(), "initialization fault: " + e.Message()
	);
	delete _Storage;
	_Storage = 0;
    }
    
    _DataReceiver = new TKinokoDataReceiver();
    _DataStreamScanner = new TKinokoDataStreamScanner();

    _DataReceiver->SetStreamCommandProcessor(_StreamCommandProcessor);
    _DataReceiver->ConstructInlet(_InputDataStream);

    if ((_IntervalSec > 0) || (_IntervalUSec > 0)) {
	_DelayTimer = new TMushRealTimeTimer(_IntervalSec, _IntervalUSec);
    }

    // Send DataDescriptor Packet //
    // Some compponents need DataDescriptor in prior to data taking.
    //... BUG: The following code sends only one DataDescriptor packet.
    while (ProcessData() == 0) {
	if (_InputDataStream->NextDataSize() == 0) {
	    _Logger->WriteWarning(
		ComponentName(), 
		"unable to find datasource: " + _DataSourceName
	    );
	    break;
	}
    }
}

void TKinokoReproducerCom::Destruct(void) throw(TKinokoException)
{
    delete _DelayTimer;
    delete _DataStreamScanner;
    delete _DataReceiver;
    delete _Storage;

    _DelayTimer = 0;
    _InputDataStream = 0;
    _DataStreamScanner = 0;
    _DataReceiver = 0;
    _Storage = 0;

    _IntervalSec = 0;
    _IntervalUSec = 0;

    _DataSourceId = TKinokoDataSource::DataSourceId_Any;
    _NumberOfDataSources = 0;
}

int TKinokoReproducerCom::ProcessData(void)throw(TKinokoException)
{
    void* Packet;
    int PacketSize = _InputDataStream->NextEntry(Packet);
    if (PacketSize <= 0) {
	_TimeKeeper->Suspend();
	return 0;
    }
    _DataStreamScanner->CorrectByteOrder(Packet, PacketSize);
    
    if (_DataReceiver->ProcessDataDescriptorPacket(Packet, PacketSize)) {
	if (_DataSourceId == TKinokoDataSource::DataSourceId_Unknown) {
	    TKinokoDataSource* DataSource = _DataReceiver->InputDataDescriptor()->DataSource(_DataSourceName);
	    if (DataSource != 0) {
		_DataSourceId = DataSource->DataSourceId();
	    }
	}
    }

    long DataSourceId = _DataStreamScanner->DataSourceIdOf(Packet);
    if (
	(_DataSourceId != TKinokoDataSource::DataSourceId_Any) &&
	(_DataSourceId != DataSourceId) 
    ){
	_InputDataStream->Flush(Packet);
	return 0;
    }

    int WrittenLength;
    while ((WrittenLength = _OutputDataStream->Write(Packet, PacketSize)) == 0) {
	_Logger->WriteWarning(ComponentName(), "unable to write to stream");
	_TimeKeeper->Suspend();
    }
    if (WrittenLength != PacketSize) {
	_Logger->WritePanic(ComponentName(), "stream I/O error (internal)");
	_InputDataStream->Flush(Packet);
	return 0;
    }
    
    if (TKinokoDataStreamScanner::IsTrailerPacket(Packet)) {
	int TrailerValue = TKinokoDataStreamScanner::TrailerValueOf(Packet);
	if (TrailerValue == TKinokoDataStreamScanner::Trailer_Event) {
	    if (_DelayTimer) {
		_DelayTimer->Suspend();
	    }
	}
    }
    else if (TKinokoDataStreamScanner::IsCommandPacket(Packet)) {
	int CommandValue = TKinokoDataStreamScanner::CommandValueOf(Packet);
	if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
	    _NumberOfDataSources++;
	}
	if (CommandValue == TKinokoDataStreamScanner::Command_RunEnd) {
	    _NumberOfDataSources--;
	    if (_NumberOfDataSources <= 0) {
		StopDataProcessing();
	    }
	}
    }

    _InputDataStream->Flush(Packet);

    return (PacketSize > 0);
}

void TKinokoReproducerCom::OnStop(void) throw(TKinokoException)
{
    TKcomEvent Event;
    EmitEventOneWay(EventId_RunEnd, Event);
}
