/* KinokoStandaloneComponent.cc */
/* Created by Enomoto Sanshiro on 9 August 2001. */
/* Last updated by Enomoto Sanshiro on 21 September 2004. */


#include <string>
#include "MushArgumentList.hh"
#include "MushFileSystem.hh"
#include "MushSocket.hh"
#include "MushNetworkSocket.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "MushProcess.hh"
#include "MushSocketStream.hh"
#include "KinokoComponentPlatform.hh"
#include "KinokoLogger.hh"
#include "KinokoDataProcessor.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoStandaloneComponent.hh"

using namespace std;


TKinokoStandaloneComponent::TKinokoStandaloneComponent(const string& Name, int DataSourceId)
{
    _Name = Name;
    _DataSourceId = DataSourceId;

    _StreamSocket = 0;
    _OutputSocketStream = 0;
    _InputSocketStream = 0;
    _InputStream = &cin;
    _OutputStream = &cout;

    _DelayTimer = 0;

    _Logger = 0;
}

TKinokoStandaloneComponent::~TKinokoStandaloneComponent()
{
    delete _DelayTimer;
    delete _OutputSocketStream;
    delete _InputSocketStream;
    delete _StreamSocket;

    delete _Logger;
}

void TKinokoStandaloneComponent::Start(TMushArgumentList& ArgumentList) throw(TKinokoException)
{
    if (ArgumentList.IsOptionSpecified("--port")) {
	try {
	    int PortNumber = ArgumentList.IntOptionValueOf("--port");
	    _StreamSocket = new TMushServerNetworkSocket(PortNumber);
	    _StreamSocket->EnableLocalAddressReuse();
	    _StreamSocket->Open();
	}
	catch (TSystemCallException &e) {
	    throw TKinokoException("system call exception: " + e.Message());
	}

	_InputSocketStream = new TMushInputSocketStream(_StreamSocket);
	_OutputSocketStream = new TMushOutputSocketStream(_StreamSocket);
	_InputStream = _InputSocketStream;
	_OutputStream = _OutputSocketStream;
    }

    if (ArgumentList.IsOptionSpecified("--delay")) {
	try {
	    long Delay_msec = ArgumentList.IntOptionValueOf("--delay");

	    long Delay_sec = Delay_msec / 1000;
	    long Delay_usec = (Delay_msec % 1000) * 1000;
	    _DelayTimer = new TMushRealTimeTimer(Delay_sec, Delay_usec);
	}
	catch (TSystemCallException &e) {
	    _DelayTimer = 0;
	    throw TKinokoException("system call exception: " + e.Message());
	}
    }

    string RegistryFileName;
    if (ArgumentList.IsOptionSpecified("--registry")) {
	RegistryFileName = ArgumentList.OptionValueOf("--registry");
    }

    TKcomRegistry* KcomRegistry = 0;
    try {
	if (RegistryFileName.empty()) {
	    _Registry = new TKinokoNullRegistry();
	}
	else {
	    if (TMushFileAttribute(RegistryFileName).IsReadable()) {
		TKcomRegistryRepository Repository(RegistryFileName);
		KcomRegistry = Repository.Load();
	    }
	    else {
		KcomRegistry = new TKcomRegistry();
	    }
	    _Registry = new TKinokoComponentRegistry(KcomRegistry);
	}
    }
    catch (TKcomException &e) {
	throw TKinokoException(e.Message());
    }

    _EventEmitter = new TKinokoEventEmitter();
    _Logger = new TKinokoLogger(&cout);

    if (_DataSourceId == 0) {
	// DataSourceId must be greater than 1023 and less than 0x10000.
	//... Is it better to use a hash value of the DataSourceName string?
	_DataSourceId = (TMushProcess().ProcessId() & 0x7fff) + 1024;
    }

    OpenStreamFile(ArgumentList);

    Construct();
    while (ProcessData()) {
	if (_DelayTimer) {
	    _DelayTimer->Suspend();
	}
    }
    Destruct();

    try {
	if (! RegistryFileName.empty()) {
	    TKcomRegistryRepository Repository(RegistryFileName);
	    Repository.Save(KcomRegistry);
	}
    }
    catch (TKcomException &e) {
	throw TKinokoException(e.Message());
    }
}



TKinokoStandaloneDataProcessor::TKinokoStandaloneDataProcessor(TKinokoDataProcessor* DataProcessor, const string& Name, int DataSourceId)
: TKinokoStandaloneComponent(Name, DataSourceId)
{
    _DataProcessor = DataProcessor;

    _InputDataStorage = 0;
    _OutputDataStorage = 0;
}

TKinokoStandaloneDataProcessor::~TKinokoStandaloneDataProcessor()
{
    delete _InputDataStorage;
    delete _OutputDataStorage;
}

void TKinokoStandaloneDataProcessor::OpenStreamFile(TMushArgumentList& ArgumentList) throw(TKinokoException)
{
    if (ArgumentList.NumberOfParameters() < 2) {
	throw TKinokoException(
	    "too few arguments",
	    "\nusage: " + _Name + " InputFileName OutputFileName"
	);
    }

    _InputFileName = ArgumentList[0];
    if (! TMushFileAttribute(_InputFileName).IsReadable()) {
	throw TKinokoException("unable to read file: " + _InputFileName);
    }

    _OutputFileName = ArgumentList[1];
    if (TMushFileAttribute(_OutputFileName).IsReadable()) {
	if (! ArgumentList.IsOptionSpecified("-f")) {
	    throw TKinokoException(
		"file exists: " + _OutputFileName + "\nuse -f to overwrite"
	    );
	}
    }

    if (_InputFileName == _OutputFileName) {
	throw TKinokoException(
	    "OutputFileName is same as InputFileName"
	);
    }

    bool IsRaw = ArgumentList.IsOptionSpecified("--raw");
    _InputDataStorage = new TKinokoKdfStorage(_InputFileName, IsRaw);
    _OutputDataStorage = new TKinokoKdfStorage(_OutputFileName, IsRaw);

    if (
	ArgumentList.IsOptionSpecified("--compress") ||
	ArgumentList.IsOptionSpecified("-c")
    ){
	_OutputDataStorage->EnableDataCompression();
    }
    if (ArgumentList.IsOptionSpecified("-f")) {
	_OutputDataStorage->AllowOverWrite();
    }
}

void TKinokoStandaloneDataProcessor::Construct(void) throw(TKinokoException)
{
    string Comment = "converted from " + _InputFileName;
    Comment += " by " + string(TMushUser().Name());
    Comment += " with " + _Name;
    Comment += " on " + TMushDateTime().AsString();

    TKinokoStorageHeader StorageHeader;
    _InputDataStorage->ReadHeader(StorageHeader);
    StorageHeader.AddEntry("Comment", Comment);
    _OutputDataStorage->WriteHeader(StorageHeader);

    _InputDataStream = _InputDataStorage->GetInputStream();
    _OutputDataStream = _OutputDataStorage->GetOutputStream();

    _DataProcessor->AttachPlatform(
	*_InputStream, *_OutputStream, _EventEmitter, _Registry, _Logger
    );
    _DataProcessor->Construct(
	_Name, _InputDataStream, _OutputDataStream, _DataSourceId
    );
}

void TKinokoStandaloneDataProcessor::Destruct(void) throw(TKinokoException)
{
    _DataProcessor->Destruct();

    delete _InputDataStorage;
    delete _OutputDataStorage;

    _InputDataStorage = 0;
    _OutputDataStorage = 0;
}

int TKinokoStandaloneDataProcessor::ProcessData(void) throw(TKinokoException)
{
    return _DataProcessor->ProcessData();
}



TKinokoStandaloneDataProducer::TKinokoStandaloneDataProducer(TKinokoDataProducer* DataProducer, const string& Name, int DataSourceId)
: TKinokoStandaloneComponent(Name, DataSourceId)
{
    _DataProducer = DataProducer;
    _OutputDataStorage = 0;
}

TKinokoStandaloneDataProducer::~TKinokoStandaloneDataProducer()
{
    delete _OutputDataStorage;
}

void TKinokoStandaloneDataProducer::OpenStreamFile(TMushArgumentList& ArgumentList) throw(TKinokoException)
{
    _Name = ArgumentList.ProgramName();
    if (ArgumentList.NumberOfParameters() < 1) {
	throw TKinokoException(
	    "too few arguments",
	    "\nusage: " + _Name + " OutputFileName"
	);
    }

    _OutputFileName = ArgumentList[0];
    if (TMushFileAttribute(_OutputFileName).IsReadable()) {
	if (! ArgumentList.IsOptionSpecified("-f")) {
	    throw TKinokoException(
		"file exists: " + _OutputFileName + "\nuse -f to overwrite"
	    );
	}
    }

    _OutputDataStorage = new TKinokoKdfStorage(_OutputFileName);

    if (
	ArgumentList.IsOptionSpecified("--compress") ||
	ArgumentList.IsOptionSpecified("-c")
    ){
	_OutputDataStorage->EnableDataCompression();
    }
    if (ArgumentList.IsOptionSpecified("-f")) {
	_OutputDataStorage->AllowOverWrite();
    }
}

void TKinokoStandaloneDataProducer::Construct(void) throw(TKinokoException)
{
    TKinokoStorageHeader StorageHeader;
    StorageHeader.AddEntry("Creator", _Name);
    StorageHeader.AddEntry("DateTime", TMushDateTime().AsString());
    StorageHeader.AddEntry("UserName", TMushUser().Name());
    _OutputDataStorage->WriteHeader(StorageHeader);

    _OutputDataStream = _OutputDataStorage->GetOutputStream();
    _DataProducer->AttachPlatform(
	*_InputStream, *_OutputStream, _EventEmitter, _Registry, _Logger
    );
    _DataProducer->Construct(_Name, _OutputDataStream, _DataSourceId);
}

void TKinokoStandaloneDataProducer::Destruct(void) throw(TKinokoException)
{
    _DataProducer->Destruct();

    delete _OutputDataStorage;
    _OutputDataStorage = 0;
}

int TKinokoStandaloneDataProducer::ProcessData(void) throw(TKinokoException)
{
    return _DataProducer->ProcessData();
}



TKinokoStandaloneDataConsumer::TKinokoStandaloneDataConsumer(TKinokoDataConsumer* DataConsumer, const string& Name)
: TKinokoStandaloneComponent(Name)
{
    _DataConsumer = DataConsumer;
    _InputDataStorage = 0;
}

TKinokoStandaloneDataConsumer::~TKinokoStandaloneDataConsumer()
{
    delete _InputDataStorage;
}

void TKinokoStandaloneDataConsumer::OpenStreamFile(TMushArgumentList& ArgumentList) throw(TKinokoException)
{
    if (ArgumentList.NumberOfParameters() < 1) {
	throw TKinokoException(
	    "too few arguments",
	    "\nusage: " + _Name + " InputFileName"
	);
    }

    _InputFileName = ArgumentList[0];
    if (! TMushFileAttribute(_InputFileName).IsReadable()) {
	throw TKinokoException("unable to read file: " + _InputFileName);
    }

    _InputDataStorage = new TKinokoKdfStorage(_InputFileName);
}

void TKinokoStandaloneDataConsumer::Construct(void) throw(TKinokoException)
{
    _InputDataStream = _InputDataStorage->GetInputStream();
    _DataConsumer->AttachPlatform(
	*_InputStream, *_OutputStream, _EventEmitter, _Registry, _Logger
    );
    _DataConsumer->Construct(_Name, _InputDataStream);
}

void TKinokoStandaloneDataConsumer::Destruct(void) throw(TKinokoException)
{
    _DataConsumer->Destruct();

    delete _InputDataStorage;
    _InputDataStorage = 0;
}

int TKinokoStandaloneDataConsumer::ProcessData(void) throw(TKinokoException)
{
    return _DataConsumer->ProcessData();
}
