/* KinokoConsole.cc */
/* Created by Enomoto Sanshiro on 23 August 2002. */
/* Last updated by Enomoto Sanshiro on 23 August 2002. */


#include <string>
#include <vector>
#include "MushThread.hh"
#include "MushMisc.hh"
#include "KinokoPlatform.hh"
#include "KinokoConsole.hh"

using namespace std;


TKinokoConsole::TKinokoConsole(istream& InputStream, ostream& OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry, TKinokoTerminal* Terminal)
: _InputStream(InputStream), _OutputStream(OutputStream)
{
    _EventEmitter = EventEmitter;
    _Registry = Registry;
    _Terminal = Terminal;

    _Script = new TKinokoControlScript(_EventEmitter, _Registry);

    _TokenTable = _Script->GetTokenTable();
    _SymbolTable = _Script->GetSymbolTable();
    _Package = _Script->GetPackage();
    _StatementParser = _Script->GetStatementParser();
    _Tokenizer = new TParaTokenizer(_InputStream, _TokenTable);

    _IsStopRequested = false;
}

TKinokoConsole::~TKinokoConsole()
{
    for (unsigned j = 0; j < _EntryList.size(); j++) {
	delete _EntryList[j];
    }
    for (unsigned i = 0; i < _InputStreamList.size(); i++) {
	delete _InputStreamList[i];
    }

    delete _Script;
}

int TKinokoConsole::ProcessInput(void)
{
    if (_IsStopRequested) {
        return 0;
    }

    _Terminal->SetPrompt("kinoko> ");
    if (_Tokenizer->LookAhead().IsEmpty()) {
	return 0;
    }

    _Terminal->SetPrompt("? ");
    try {
	if (_Tokenizer->LookAhead().Is(".")) {
	    ProcessCommand();
	}
	else {
	    bool IsResultPrintingEnabled = false;
	    if (_Tokenizer->LookAhead().Is("?")) {
		_Tokenizer->Next();
		IsResultPrintingEnabled = true;
	    }
	    ProcessScript();
	    
	    if (IsResultPrintingEnabled) {
		_OutputStream << _LastResult.AsString() << endl;
	    }
	}
    }
    catch (TScriptException &e) {
	_OutputStream << "ERROR: " << e << endl;
    }

    return 1;
}

void TKinokoConsole::ProcessScript(void) throw(TScriptException)
{
    TParaPackageEntry* Entry = _Package->CreateEntry(_Tokenizer);
    if (Entry != 0) {
	Entry->Parse(_Tokenizer, _StatementParser, _SymbolTable);
	_EntryList.push_back(Entry);
    }
    else {
	TParaStatement* Statement = 0;
	try {
	    Statement = _StatementParser->Parse(_Tokenizer, _SymbolTable);
	    _LastResult = Statement->Execute(_SymbolTable).ReturnValue;
	}
	catch (TScriptException &e) {
	    delete Statement;	
	    throw;
	}
	delete Statement;	
    }
}

void TKinokoConsole::ProcessCommand(void) throw(TScriptException)
{
    _Tokenizer->Next().MustBe(".");
    TParaToken Token = _Tokenizer->Next();

    if (Token.Is("execute") || Token.Is("x")) {
	string FileName = _Tokenizer->SkipWhiteSpace().GetLine().AsString();
	Execute(FileName);
    }
    else if (Token.Is("quit") || Token.Is("q")) {
	_IsStopRequested = true;
	_OutputStream << "session terminated" << endl;
    }
    else {
	throw TScriptException("unknown command: " + Token.AsString());
    }
}

void TKinokoConsole::Print(const string& Message)
{
    //... BUG
    // This output message is not displayed until [ENTER] is pressed
    // for _InputStream even in TKinokoThreadedConsole.

    _OutputStream << Message << flush;
}

void TKinokoConsole::Execute(const std::string& CommandFileName) throw(TScriptException)
{
    ifstream* InputStream = new ifstream(CommandFileName.c_str());
    _InputStreamList.push_back(InputStream);

    if (! *InputStream) {
	throw TScriptException("unable to open file: " + CommandFileName);
    }

    _Tokenizer->InputBuffer()->SetChildInput(*InputStream);
    ProcessScript();
}

void TKinokoConsole::Quit(void)
{
    _IsStopRequested = true;
}



TKinokoThreadedConsole::TKinokoThreadedConsole(istream& InputStream, ostream& OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry, TKinokoTerminal* Terminal)
: TKinokoConsole(InputStream, OutputStream, EventEmitter, Registry, Terminal)
{
    _Condition = new TMushThreadCondition();
}

TKinokoThreadedConsole::~TKinokoThreadedConsole()
{
    delete _Condition;
}

void TKinokoThreadedConsole::Run(void)
{
    EnableCancel();
    EnableAsynchronousCancel();

    while (! _IsStopRequested) {
	if (ProcessInput() == 0) {
	    TMushDateTime Time;
	    _Condition->TimedWait(
		Time.Sec(), 1000 * (Time.USec() + 1000)
	    );
	}
    }
}

void TKinokoThreadedConsole::Quit(void)
{
    TKinokoConsole::Quit();
    Cancel();
}
